﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>

#include "testUart_Common.h"

namespace nnt {
namespace uart {

/*
 * ターゲット上で利用可能な最初のポート番号を返します。
 */
int GetFirstAvailablePortIndex()
{
    for (int i = 0; i < DevPortCountMax; ++i)
    {
        if (nn::uart::HasPortForDev(i))
        {
            return i;
        }
    }

    return -1;
}

/*
 * バッファの中身を決まったパターンで埋めます。
 */
void GenerateTestData(char* data, size_t dataBytes)
{
    for (size_t i = 0; i < dataBytes; i++)
    {
        data[i] = static_cast<char>(i & 0xff);
    }
}

/*
 * バッファの中身をクリアします。
 */
void ClearBuffer(char* data, size_t dataBytes)
{
    std::memset(data, 0x00, dataBytes);
}

/*
 * バイナリの内容を表示します。
 */
void DumpBinary(const char* pData, size_t dataBytes)
{
    NN_ABORT_UNLESS_NOT_NULL(pData);

    for (size_t i = 0; i < dataBytes; ++i)
    {
        NN_LOG("%02X ", pData[i]);
    }
    NN_LOG("\n");
}

/*
 * テスト用のデータと実際のデータを比較します。
 */
bool CompareTestData(const char* pExpectedData,
                     const char* pActualData,
                     size_t dataBytes)
{
    NN_ABORT_UNLESS_NOT_NULL(pExpectedData);
    NN_ABORT_UNLESS_NOT_NULL(pActualData);

    for (size_t i = 0; i < dataBytes; ++i)
    {
        if (pExpectedData[i] != pActualData[i])
        {
            NN_LOG("\n** Data did not match at index %d **\n\n", i);
            NN_LOG("Expected | Actual\n");
            for (size_t dump = 0; dump < dataBytes; ++dump)
            {
                 NN_LOG("  %02x | %02x\n", pExpectedData[dump], pActualData[dump]);
            }
            return false;
        }
    }

    return true;
}


/*
 * 時限つきでバッファイベントを待機します。
 */
bool WaitBufferEvent(nn::uart::PortSession& portSession, nn::uart::PortEventType eventType, size_t threshold, nn::TimeSpan timeout)
{
    nn::os::SystemEventType event;

    if (!nn::uart::BindPortEvent(&event, &portSession, eventType, threshold))
    {
        NN_LOG("Could not bind port event\n");
        return false;
    }
    if (!nn::os::TimedWaitSystemEvent(&event, timeout))
    {
        NN_LOG("Specified buffer event did not happen in expected time\n");
        nn::uart::UnbindPortEvent(&event, &portSession);
        nn::os::DestroySystemEvent(&event);
        return false;
    }
    if (!nn::uart::UnbindPortEvent(&event, &portSession))
    {
        NN_LOG("Could not unbind port event\n");
        nn::os::DestroySystemEvent(&event);
        return false;
    }

    nn::os::DestroySystemEvent(&event);

    return true;
}

/*
 * 送信バッファが空になるまで待機します。
 */
bool WaitSendBufferEmpty(nn::uart::PortSession& portSession, nn::TimeSpan timeout)
{
    return WaitBufferEvent(portSession, nn::uart::PortEventType_SendBufferEmpty, 0, timeout);
}

/*
 * 指定されたサイズのデータが受信可能になるまで待機します。
 */
bool WaitDataReady(nn::uart::PortSession& portSession, size_t expectedSize, nn::TimeSpan timeout)
{
    return WaitBufferEvent(portSession, nn::uart::PortEventType_ReceiveBufferReady, expectedSize, timeout);

}

/*
 * 指定されたサイズのデータを同期受信します。
 * 注: この関数は receiveDataCount がポートの受信バッファサイズ以下でなければ正常に機能しません。
 *     大きなデータの受信は ReceiveData() を使用してください。
 */
bool ReceiveDataSync(size_t* pDoneBytes,
                     char* pOutBuffer,
                     size_t receiveDataCount,
                     nn::uart::PortSession& portSession,
                     nn::TimeSpan timeout)
{
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);

    bool isDataReady = WaitDataReady(portSession,
                                     receiveDataCount,
                                     timeout);
    if (!isDataReady)
    {
        return false;
    }

    size_t doneBytes;
    auto result = nn::uart::Receive(&doneBytes,
                                    pOutBuffer,
                                    receiveDataCount,
                                    &portSession);
    if (!result.IsSuccess())
    {
        NN_LOG("Could not receive, result=%d\n", result);
        return false;
    }

    if (pDoneBytes)
    {
        *pDoneBytes = doneBytes;
    }
    return true;
}

/*
 * バッファサイズを超えるような大きなデータをすべて送信します。基本的にフロー制御とセットで使ってください。
 */
bool SendData(nn::uart::PortSession& portSession, char* data, size_t dataBytes, size_t chunkSize, nn::TimeSpan interval)
{
    size_t remainingBytes = dataBytes;
    char*  pCurrent = data;
    while (remainingBytes > 0)
    {
        size_t bytesToSend = (remainingBytes < chunkSize) ? remainingBytes : chunkSize;
        size_t doneBytes;
        nn::uart::Send(&doneBytes, &portSession, pCurrent, bytesToSend);
        if (doneBytes == 0)
        {
            nn::os::SleepThread(interval);
            continue;
        }
        else
        {
            remainingBytes -= doneBytes;
            pCurrent += doneBytes;
        }
    }
    return true;
}

/*
 * バッファサイズを超えるような大きなデータをすべて受信します。基本的にフロー制御とセットで使ってください。
 */
bool ReceiveData(nn::uart::PortSession& portSession, char* data, size_t dataBytes, size_t chunkSize, nn::TimeSpan interval)
{
    size_t remainingBytes = dataBytes;
    char*  pCurrent = data;
    while (remainingBytes > 0)
    {
        size_t bytesToReceive = (remainingBytes < chunkSize) ? remainingBytes : chunkSize;
        size_t doneBytes;
        nn::Result result = nn::uart::Receive(&doneBytes, pCurrent, bytesToReceive, &portSession);
        if (!result.IsSuccess())
        {
            return false;
        }
        else if (doneBytes == 0)
        {
            nn::os::SleepThread(interval);
            continue;
        }
        else
        {
            remainingBytes -= doneBytes;
            pCurrent += doneBytes;
        }
    }
    return true;
}

} // uart
} // nnt
