﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Log.h>

#include <nnt/gtest/gtest.h>

#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>

#include "../Common/testUart_Common.h"
#include "../Common/testUart_Helper.h"

namespace {

const size_t UartBufferSize = 1 * nn::os::MemoryPageSize;

// 内部送信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_SendBuffer[UartBufferSize];
// 内部受信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char g_ReceiveBuffer[UartBufferSize];

const nn::uart::BaudRate TestedBaudRate = nn::uart::BaudRate_115200;

}

/*
 * 全イベント共通のテスト
 */
TEST(PortEventTest, Basic)
{
    nn::uart::Initialize();
    ASSERT_TRUE(nn::uart::IsInitialized());

    // 最初に見つかった有効なポートをテストに使用
    int devPortIndex = nnt::uart::GetFirstAvailablePortIndex();
    ASSERT_GE(devPortIndex, 0);

    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        TestedBaudRate,
        g_SendBuffer,
        sizeof(g_SendBuffer),
        g_ReceiveBuffer,
        sizeof(g_ReceiveBuffer)
        );
    nn::uart::OpenPortForDev(&portSession, devPortIndex, portConfig);

    // テスト対象のイベント
    const nn::uart::PortEventType TestPortEvents[] =
    {
        nn::uart::PortEventType_SendBufferEmpty,
        nn::uart::PortEventType_SendBufferReady,
        nn::uart::PortEventType_ReceiveBufferReady,
        nn::uart::PortEventType_ReceiveEnd,
    };

    // 送受信はしないので、事前条件を満たせば閾値は何でも良い
    const size_t TestThreshold = 1;

    for (auto portEvent : TestPortEvents)
    {
        if (!nn::uart::IsSupportedPortEventForDev(devPortIndex, portEvent))
        {
            NN_LOG("PortEvent %d is not supported\n", portEvent);
            continue;
        }

        nn::os::SystemEventType systemEvent;

        // イベントの登録
        ASSERT_TRUE(
                nn::uart::BindPortEvent(&systemEvent,
                                        &portSession,
                                        portEvent,
                                        TestThreshold));

        // 二重設定が失敗することを確認
        ASSERT_FALSE(
                nn::uart::BindPortEvent(&systemEvent,
                                        &portSession,
                                        portEvent,
                                        TestThreshold));

        // イベントの解除
        ASSERT_TRUE(nn::uart::UnbindPortEvent(&systemEvent, &portSession));

        // 解除済みで再度解除したら失敗することを確認
        ASSERT_FALSE(nn::uart::UnbindPortEvent(&systemEvent, &portSession));

        // イベントの破棄はドライバ使用者が行う
        nn::os::DestroySystemEvent(&systemEvent);

        // 同じイベントに対して bind -> Unbind を繰り返せること確認
        // イベントの登録
        ASSERT_TRUE(
                nn::uart::BindPortEvent(&systemEvent,
                                        &portSession,
                                        portEvent,
                                        TestThreshold));

        // イベントの解除
        ASSERT_TRUE(nn::uart::UnbindPortEvent(&systemEvent, &portSession));

        // イベントの破棄はドライバ使用者が行う
        nn::os::DestroySystemEvent(&systemEvent);
    }

    nn::uart::ClosePort(&portSession);
    nn::uart::Finalize();
    ASSERT_FALSE(nn::uart::IsInitialized());
}
