﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nnt/gtest/gtest.h>

#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>

#include "../Common/testUart_Helper.h"

namespace {

const size_t UartBufferSize = 32 * nn::os::MemoryPageSize;

// 内部送信バッファ
NN_ALIGNAS(nn::os::MemoryPageSize) char s_SendBuffer[UartBufferSize];
// 内部受信バッファ（未使用）
// NN_ALIGNAS(nn::os::MemoryPageSize) char s_ReceiveBuffer[UartBufferSize];

// サポートされている最速ボーレートを使うこと
const nn::uart::BaudRate TestedBaudRate = nn::uart::BaudRate_115200;

}

/**
    送信の実測速度がボーレートと比して妥当な速度であることのテスト

    テスト手順
        1. 適当なポートに、サポートされている最速のボーレート設定でまとまったデータを Send
        2. 送信バッファが空になるのを BindPortEvent でタイムアウトつきで待機する
            - タイムアウト値を、ボーレートから計算した所要時間の許容上限値にし、タイムアウトしたら失敗
        3. 1-2 を数回繰り返し、いずれも許容時間内で転送完了することをテスト
 */
TEST(PerfTest, Send)
{
    const size_t TestStringSize = UartBufferSize;

    // 前準備
    nn::uart::Initialize();

    // bluetooth チップ向けポートを使用する
    // CTS が disable になっているポート向けにテストすると失敗します。
    const int BluetoothPortIndex = 3;

    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        TestedBaudRate,
        s_SendBuffer,
        sizeof(s_SendBuffer),
        nullptr,
        0
        );
    nn::uart::OpenPortForDev(&portSession, BluetoothPortIndex, portConfig);

    // ボーレートの 70% 値を下限 bps とし、その速度で転送に要する時間を許容上限値とする
    const nn::TimeSpan LongestAcceptablePassedTime = nn::TimeSpan::FromSeconds((TestStringSize * 8) / (static_cast<size_t>(TestedBaudRate) * 7 / 10));
    NN_LOG("Longest Acceptable Passed Time: %16lld [msec]\n", LongestAcceptablePassedTime.GetMilliSeconds());

    // 複数回計測する
    for (int i = 0; i < 3; ++i)
    {
        // 現在時刻
        nn::TimeSpan startTime = nn::os::GetSystemTick().ToTimeSpan();
        NN_LOG(" Start time: %16lld [msec]\n", startTime.GetMilliSeconds());

        // 送信バッファと同じサイズのテストデータを転送（なので一回で全部送れるはず）
        static const char TestString[TestStringSize] = {'A'};
        size_t doneBytes;
        nn::uart::Send(&doneBytes, &portSession, TestString, TestStringSize);
        ASSERT_EQ(TestStringSize, doneBytes);

        // 許容上限時間内に送信が完了するのをテストする
        ASSERT_TRUE(nnt::uart::WaitSendBufferEmpty(portSession, LongestAcceptablePassedTime));

        // 経過時間を計測して表示
        nn::TimeSpan endTime = nn::os::GetSystemTick().ToTimeSpan();
        nn::TimeSpan passedTime = endTime - startTime;
        NN_LOG("   End time: %16lld [msec]\n", endTime.GetMilliSeconds());
        NN_LOG("Passed time: %16lld [msec]\n", passedTime.GetMilliSeconds());

        // 実際の経過時間が許容上限値以下に収まっていることをテスト
        // gtest マクロの仕様により、usec 単位に変換して比較する
        ASSERT_LE(passedTime.GetMicroSeconds(), LongestAcceptablePassedTime.GetMicroSeconds());

        // 少し待って次のトライへ
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    }

    // 後始末
    nn::uart::ClosePort(&portSession);
    nn::uart::Finalize();
}
