﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using ZarfCreator.Parser;
using ZarfCreator.ZarfDefinitionData;
using Nintendo.Zarf.v1;
using System.Collections;
using System.Linq;

namespace ZarfCreatorTest.ParserTest
{
    /// <summary>
    /// InstructionCommandParseTest の概要の説明
    /// </summary>
    [TestClass]
    public class InstructionCommandParseTest
    {
        /// <summary>
        /// Internal-Parser のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest1()
        {
            var parser = new InstructionCommandParser();
            var data1 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "unzip" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-bin}\\${File1}"
                    }
                },
                { "OnFailure", "DoNothing" },
            };

            var result = parser.Parse(data1);

            Assert.AreEqual(result.Handler, Command.HandlerKind.InternalParser);
            Assert.AreEqual(result.CmdSpecifier, "unzip");
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(), new List<string>() { @"{fulldir-deploy-bin}\\${File1}" });
            Assert.AreEqual(result.OnFailure, Command.FailureAction.DoNothing);


            var data2 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "replacementCopy" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-temp}\\NintendoSDK",
                        "{release-installdir}"
                    }
                }
            };
            result = parser.Parse(data2);

            Assert.AreEqual(result.Handler, Command.HandlerKind.InternalParser);
            Assert.AreEqual(result.CmdSpecifier, "replacementCopy");
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(),
                new List<string>() { @"{fulldir-temp}\\NintendoSDK", "{release-installdir}" });
            Assert.IsNull(result.OnFailure);

            var data3 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "unzip" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-bin}\\${File1}"
                    }
                },
                { "SuccessReturnCodes", new List<object>() {"0"} },  //TODO これは別に問題ない？
                { "OnFailure", "ShowStdErr" },
            };

            result = parser.Parse(data3);

            Assert.AreEqual(result.Handler, Command.HandlerKind.InternalParser);
            Assert.AreEqual(result.CmdSpecifier, "unzip");
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(),
                new List<string>() { @"{fulldir-deploy-bin}\\${File1}" });
            CollectionAssert.AreEquivalent(result.SuccessReturnCodes.ToList(), new List<int>() { 0 });
            Assert.AreEqual(result.OnFailure, Command.FailureAction.ShowStdErr);


            var data4 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "uninstallCopiedFiles" },
                { "CmdArgs", new List<object>() },
            };
            result = parser.Parse(data4);

            Assert.AreEqual(result.Handler, Command.HandlerKind.InternalParser);
            Assert.AreEqual(result.CmdSpecifier, "uninstallCopiedFiles");
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(), new List<string>());

        }

        /// <summary>
        /// External-Cmd のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest2()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "External-Cmd" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-logic}\\install.bat",
                        "arg1"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0", "-3" }}
            };

            var result = parser.Parse(data);

            Assert.AreEqual(result.Handler, Command.HandlerKind.ExternalCmd);
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(),
                new List<string>() { @"{fulldir-deploy-logic}\\install.bat", "arg1" });
            CollectionAssert.AreEquivalent(result.SuccessReturnCodes.ToList(),
                new List<int>() { 0, -3 });

            data = new Dictionary<string, object>()
            {
                { "Handler", "External-Cmd" },
                { "CmdArgs", new List<object>()
                    {
                        "{fulldir-deploy-bin}"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }},
                { "OnFailure", "ShowStdOut" },
            };

            result = parser.Parse(data);

            Assert.AreEqual(result.Handler, Command.HandlerKind.ExternalCmd);
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(),
                new List<string>() { "{fulldir-deploy-bin}" });
            CollectionAssert.AreEquivalent(result.SuccessReturnCodes.ToList(),
                new List<int>() { 0 });
            Assert.AreEqual(result.OnFailure, Command.FailureAction.ShowStdOut);
        }

        /// <summary>
        /// External-Exe のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest3()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" },
                { "CmdSpecifier", "msiexec" },
                { "CmdArgs", new List<object>()
                    {
                        "/i",
                        @"{fulldir-deploy-bin}\\Setup.msi",
                        "/qn",
                        "/promptrestart"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }}
            };

            var result = parser.Parse(data);

            Assert.AreEqual(result.Handler, Command.HandlerKind.ExternalExe);
            CollectionAssert.AreEquivalent(result.CmdArgs.ToList(),
                new List<string>() { "/i", @"{fulldir-deploy-bin}\\Setup.msi", "/qn", "/promptrestart" });
            CollectionAssert.AreEquivalent(result.SuccessReturnCodes.ToList(),
                new List<int>() { 0 });
        }

        /// <summary>
        /// External-Powershell2 のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest4()
        {
#warning "External-Powershell2 is not tested."
            // TODO:
        }

        /// <summary>
        /// Register-Dashboard-UpdateDevice のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest5()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "Register-Dashboard-UpdateDevice" },
                { "DashboardDetails", new Dictionary<string, object>()
                    {
                        { "GroupName", "Launch Hardware Update" },
                        { "GroupDisplayName", new Dictionary<string, object>()
                            {
                                { "Ja", "開発機を更新する" },
                                { "En", "Launch Develop Hardware Update" }
                            }
                        },
                        { "Icon", "HddOutline" },
                        { "DisplayText", new Dictionary<string, object>()
                            {
                                { "Ja", "SDEV/EDEV を更新する" },
                                { "En", "Launch SDEV/EDEV Update" }
                            }
                        }
                    }
                }
            };

            var isPostInstall = true;
            var result = parser.Parse(data, isPostInstall);

            Assert.AreEqual(Command.HandlerKind.RegisterDashboardUpdateDevice, result.Handler);
            Assert.AreEqual("Launch Hardware Update", result.DashboardDetails.GroupName);
            Assert.AreEqual(
                new TextInfo
                {
                    Japanese = "開発機を更新する",
                    English = "Launch Develop Hardware Update"
                },
                result.DashboardDetails.GroupDisplayName);
            Assert.AreEqual("HddOutline", result.DashboardDetails.Icon);
            Assert.AreEqual(
                new TextInfo {
                    Japanese = "SDEV/EDEV を更新する",
                    English = "Launch SDEV/EDEV Update" },
                result.DashboardDetails.DisplayText);
        }

        /// <summary>
        /// Register-Dashboard-InstallSoftware のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest6()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "Register-Dashboard-InstallSoftware" },
                { "CmdArgs", new List<object>()
                    {
                        "NintendoSDK Samples",
                        "0.14.0"
                    }
                },
                { "DashboardDetails", new Dictionary<string, object>()
                    {
                        { "GroupName", "Samples" },
                        { "GroupDisplayName", new Dictionary<string, object>()
                            { { "En", "Samples" } }
                        },
                        { "DisplayText", new Dictionary<string, object>()
                            { { "En", "Install NintendoSDK Samples" } }
                        }
                    }
                }
            };

            var isPostInstall = true;
            var result = parser.Parse(data, isPostInstall);

            Assert.AreEqual(Command.HandlerKind.RegisterDashboardInstallSoftware, result.Handler);
        }

        /// <summary>
        /// Register-Dashboard-External-Cmd のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest7()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "Register-Dashboard-External-Cmd" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-local-env}\NintendoSDK\Installers\Scripts\open_directory.bat",
                        @"{release-installdir}\Samples"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }},
                { "DashboardDetails", new Dictionary<string, object>()
                    {
                        { "GroupName", "Samples" },
                        { "GroupDisplayName", new Dictionary<string, object>()
                            { { "En", "Samples" } }
                        },
                        { "DisplayText", new Dictionary<string, object>()
                            { { "En", "Open Folder" } }
                        }
                    }
                }
            };

            var isPostInstall = true;
            var result = parser.Parse(data, isPostInstall);

            Assert.AreEqual(Command.HandlerKind.RegisterDashboardExternalCmd, result.Handler);
        }

        /// <summary>
        /// Register-Dashboard-External-Powershell2 のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest8()
        {
#warning "Register-Dashboard-External-Powershell2 is not tested."
            // TODO:
        }

        /// <summary>
        /// Register-Dashboard-External-Exe のテスト
        /// </summary>
        [TestMethod]
        public void ParseSuccessTest9()
        {
            var parser = new InstructionCommandParser();
            var data = new Dictionary<string, object>()
            {
                { "Handler", "Register-Dashboard-External-Exe" },
                { "CmdSpecifier", "cscript.exe" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-local-env}\NintendoSDK\Installers\Vsi\Scripts\run_sequence.vbs",
                        @"{fulldir-local-env}\\intendoSDK",
                        "OasisSetup.0.12.2.15962_Hotfix4.msi"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }},
                { "OnFailure", "AlertUser" },
                { "Predicate", "Component-Installed" },
                { "PredicateArgs", new List<object>() { "Nintendo.NDI.Modules.NCLPortal" } },
                { "DashboardDetails", new Dictionary<string, object>()
                    {
                        { "GroupName", "PC Configuration" },
                        { "GroupDisplayName", new Dictionary<string, object>()
                            { { "En", "PC Configuration" } }
                        },
                        { "DisplayText", new Dictionary<string, object>()
                            { { "En", "Install VSI for NX" }, }
                        }
                    }
                }
            };

            var isPostInstall = true;
            var result = parser.Parse(data, isPostInstall);

            Assert.AreEqual(Command.HandlerKind.RegisterDashboardExternalExe, result.Handler);
            Assert.AreEqual(Command.PredicateKind.ComponentInstalled, result.Predicate);
        }

        /// <summary>
        /// 必須要素が足りない場合
        /// </summary>
        [TestMethod]
        public void ParseFailByNotMeetRequirementsTest()
        {
            var parser = new InstructionCommandParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>() { };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "CmdSpecifier", "unzip" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-bin}\\${File1}"
                    }
                }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Required items have not been completed.");
        }

        /// <summary>
        /// Internal-Parser で必須要素が足りない時
        /// </summary>
        [TestMethod]
        public void ParseFailAtInternalParserTest()
        {
            var parser = new InstructionCommandParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifier", "unzip" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Required items have not been completed.");
        }

        /// <summary>
        /// External-Cmd で必須要素が足りない場合
        /// </summary>
        [TestMethod]
        public void ParseFailAtExternalCmdTest()
        {
            var parser = new InstructionCommandParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Handler", "External-Cmd" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Handler", "External-Cmd" },
                { "CmdArgs", new List<object>()
                    {
                        @"{fulldir-deploy-logic}\\install.bat"
                    }
                }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Required items have not been completed.");
        }

        /// <summary>
        /// External-Exe で必須要素が足りない場合
        /// </summary>
        [TestMethod]
        public void ParseFailAtExternalExeTest()
        {
            var parser = new InstructionCommandParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" },
                { "CmdSpecifier", "MSBuild" },
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Required items have not been completed.");

            Console.Error.WriteLine("---");
            var data3 = new Dictionary<string, object>()
            {
                { "Handler", "External-Exe" },
                { "CmdArgs", new List<object>()
                    {
                        "/i",
                        @"{fulldir-deploy-bin}\\Setup.msi",
                        "/qn",
                        "/promptrestart"
                    }
                },
                { "SuccessReturnCodes", new List<object>() { "0" }}
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data3), "Required items have not been completed.");
        }

        /// <summary>
        /// typo している場合
        /// </summary>
        [TestMethod]
        public void ParseFailByTypographicalErrorTest()
        {
            var parser = new InstructionCommandParser();

            Console.Error.WriteLine("---");
            var data1 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Exe" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data1), "Invalid parameter in the 'Handler' field.");

            Console.Error.WriteLine("---");
            var data2 = new Dictionary<string, object>()
            {
                { "Handler", "Internal-Parser" },
                { "CmdSpecifierHoge", "unzip" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data2), "Unknown key has specified.");

            Console.Error.WriteLine("---");
            var data3 = new Dictionary<string, object>()
            {
                { "Handler", "External-Cmd" },
                { "CmdSpecifier", "unzip" },
                { "OnFailure", "hoge" }
            };

            TestUtil.ExpectException<FormatException>(() => parser.Parse(data3), "Invalid parameter in the 'OnFailure' field.");
        }
    }
}
