﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;

namespace UpdaterTest
{
    [TestClass]
    public class UpdaterTest
    {
        public TestContext TestContext { get; set; }

        private static bool VerifyBootImageHashes(string log)
        {
            int count = 0;
            bool same = true;
            foreach(var l in log.Split('\n'))
            {
                var trim = l.Trim();
                if(trim.StartsWith("bct") || trim.StartsWith("package1") || trim.StartsWith("package2"))
                {
                    count++;
                    var split = trim.Split('|');
                    string[] hashes = { split[1].Trim(), split[2].Trim(), split[3].Trim() };
                    if(hashes[0] != hashes[1] || hashes[0] != hashes[2])
                    {
                        same = false;
                    }
                }
            }
            if(count == 6 && same)
            {
                return true;
            }
            return false;
        }
        private static bool VerifyBootImageHashesAreCorrect(string log)
        {
            return VerifyBootImageHashes(log) == true;
        }
        private static bool VerifyBootImageHashesAreBroken(string log)
        {
            return VerifyBootImageHashes(log) == false;
        }

        [ClassInitialize]
        public static void ClassInit(TestContext context)
        {
            // 通常版 BootImagePackage のみがインストールされ、それが正しく適用された状態にする
            var lib = new TestOnTargetLibrary(context);
            lib.RunDevMenuCommandSystem("systemprogram uninstall 0x0100000000000819");
            lib.RunDevMenuCommandSystem("systemprogram uninstall 0x010000000000081A");
            lib.RunDevMenuCommandSystem("systemprogram uninstall 0x010000000000081B");
            lib.RunDevMenuCommandSystem("systemprogram uninstall 0x010000000000081C");

            var bip = lib.GetBipPath("BootImagePackage");
            var bipsafe = lib.GetBipPath("BootImagePackageSafe");

            lib.RunDevMenuCommandSystem("systemprogram install "+bip);
            lib.RunDevMenuCommandSystem("systemprogram install "+bipsafe);

            lib.RunDevMenuCommandSystem("systemupdate update-bootimages");
            lib.RunDevMenuCommandSystem("systemupdate update-bootimages-safe");
            lib.RunDevMenuCommandSystem("systemupdate mark-verifying-required 0");
            lib.RunDevMenuCommandSystem("systemupdate mark-verifying-required-safe 0");

            var resetHello = new RunningOptions() { WithReset = true, SuccessPattern = "Hello, world." };
            var hello = lib.GetPath("Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/HelloWorld/Develop/HelloWorld.nca");

            lib.Run(hello, "", resetHello);
        }

        private void BreakTest(int phase)
        {
            // BootImages の一部を破壊して、再起動で正しい状態に復旧するかを確認
            var lib = new TestOnTargetLibrary(this.TestContext);

            // まずは一致するかの確認
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreCorrect);

            // 検証必要フラグを立てて、破壊し、ハッシュがおかしいことを確認
            lib.RunDevMenuCommandSystem("systemupdate mark-verifying-required 1");
            lib.RunDevMenuCommandSystem("systemupdate break-bootimages " + phase.ToString());
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreBroken);

            // 再起動
            var resetHello = new RunningOptions() { WithReset = true, SuccessPattern = "Hello, world." };
            var hello = lib.GetPath("Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/HelloWorld/Develop/HelloWorld.nca");

            lib.Run(hello, "", resetHello)
                .ExpectInUartLog("Nintendo NX bootloader.", "[boot] BootImages are updated", "Nintendo NX bootloader.");

            // 再起動後、ハッシュと検証フラグが正しい状態かを確認
            lib.RunDevMenuCommandSystem("systemupdate get-verifying-required-mark")
                .ExpectInLog("Normal: 0, Safe: 0");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreCorrect);
        }

        [TestMethod]
        public void BreakPhase1()
        {
            BreakTest(1);
        }
        [TestMethod]
        public void BreakPhase2()
        {
            BreakTest(2);
        }
        [TestMethod]
        public void BreakPhase3()
        {
            BreakTest(3);
        }
        [TestMethod]
        public void BreakPhase4()
        {
            // bl binary で復旧されるようにならないとできない (SIGLO-35398)
            BreakTest(4);
        }
        [TestMethod]
        public void BreakPhase5()
        {
            BreakTest(5);
        }
        [TestMethod]
        public void BreakPhase6()
        {
            BreakTest(6);
        }
        [TestMethod]
        public void ExFat()
        {
            // ExFat 有効版の BootImagePackage が存在したら、そちらを優先するかの確認
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 何もインストールされていない状態
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id")
                .ExpectInLog("BootImagePackage: 0100000000000819");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id-safe")
                .ExpectInLog("BootImagePackageSafe: 010000000000081a");

            var bip_ex = lib.GetBipPath("BootImagePackageExFat");

            // 通常モードだけ ExFat
            lib.RunDevMenuCommandSystem("systemprogram install " + bip_ex);
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id")
                .ExpectInLog("BootImagePackage: 010000000000081b");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id-safe")
                .ExpectInLog("BootImagePackageSafe: 010000000000081a");

            // ExFat 無効に戻す
            lib.RunDevMenuCommandSystem("systemprogram uninstall 0x010000000000081B");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id")
                .ExpectInLog("BootImagePackage: 0100000000000819");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id-safe")
                .ExpectInLog("BootImagePackageSafe: 010000000000081a");

        }
        [TestMethod]
        public void ExFatWithReboot()
        {
            // 再起動時も、ExFat 有効版を使って更新するか
            var lib = new TestOnTargetLibrary(this.TestContext);

            // 何もインストールされていない状態
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id")
                .ExpectInLog("BootImagePackage: 0100000000000819");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id-safe")
                .ExpectInLog("BootImagePackageSafe: 010000000000081a");

            var bip_ex = lib.GetBipPath("BootImagePackageExFat");

            // 通常を ExFat 版にする
            lib.RunDevMenuCommandSystem("systemprogram install " + bip_ex);
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id")
                .ExpectInLog("BootImagePackage: 010000000000081b");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-package-id-safe")
                .ExpectInLog("BootImagePackageSafe: 010000000000081a");

            // ExFat 有効版と無効版とでハッシュが異なる場合に、これを利用できる
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreBroken);

            // 検証フラグは立っていないので、再起動しても更新されない
            var resetHello = new RunningOptions() { WithReset = true, SuccessPattern = "Hello, world." };
            var hello = lib.GetPath("Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/HelloWorld/Develop/HelloWorld.nca");

            lib.Run(hello, "", resetHello)
                .NotExpectInUartLog("[boot] BootImages are updated");

            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreBroken);

            // 検証フラグを立てて再起動
            lib.RunDevMenuCommandSystem("systemupdate mark-verifying-required 1");
            lib.Run(hello, "", resetHello)
                .ExpectInUartLog("Nintendo NX bootloader.", "[boot] BootImages are updated", "Nintendo NX bootloader.");

            lib.RunDevMenuCommandSystem("systemupdate get-verifying-required-mark")
                .ExpectInLog("Normal: 0, Safe: 0");
            lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                .ExpectInLog(VerifyBootImageHashesAreCorrect);
        }
        [TestMethod]
        public void ResizeBootImages()
        {
#if false
            // BootImages がどういうサイズでも更新できるか (0-padding)
            var lib = new TestOnTargetLibrary(this.TestContext);
            for (int i = 0; i < 210; ++i)
            {
                lib.RunDevMenuCommandSystem("systemprogram install " + lib.GetPath(string.Format("./Tests/Tools/Sources/Tests/UpdaterTest/MultiSizeBootImages/outputs/output-{0}.nsp", i)));
                lib.RunDevMenuCommandSystem("systemupdate update-bootimages");
                lib.RunDevMenuCommandSystem("systemupdate get-bootimage-hashes")
                    .ExpectInLog(VerifyBootImageHashesAreCorrect);
            }
#endif
        }
        [TestMethod]
        public void InfiniteRebootTest()
        {
            // エージング用のコードなので、通常 falase にする
#if false
            var lib = new TestOnTargetLibrary(this.TestContext);
            int sleepMillisec = 0;
            while (true)
            {
                // batch 内容
                // systemupdate mark-verifying-required 1
                // systemprogram list (時間稼ぎ)
                // systemprogram list-details-asll (時間稼ぎ)
                // systemupdate update-bootimages

                lib.RunDevMenuCommandSystem(string.Format("batch {0}", lib.GetPath("Tests/Tools/Sources/Tests/UpdaterTest/testBatch.txt")), new RunningOptions { NoWait = true });
                System.Threading.Thread.Sleep(sleepMillisec);

                var resetHello = new RunningOptions() { WithReset = true, SuccessPattern = "Hello, world." };
                var hello = lib.GetPath("Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/HelloWorld/Develop/HelloWorld.nca");

                lib.Run(hello, "", resetHello);
                System.Threading.Thread.Sleep(5000);

                sleepMillisec += 9;
                if(sleepMillisec > 5500)
                {
                    // time コマンドで測る感じ、5100 ミリ秒で更新が終わる
                    sleepMillisec -= 5500;
                }
            }
#endif
        }
    }
}
