﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShellTest
{
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TargetShell;

    /// <summary>
    /// DivideMultiCommandのテスト
    /// </summary>
    [TestClass]
    public class DivideMultiCommandTest
    {
        /// <summary>
        /// テストに使用する PrivateType を保持する
        /// </summary>
        private PrivateType privateType = null;

        public DivideMultiCommandTest()
        {
            this.privateType = new PrivateType(typeof(Utility));
        }

        /// <summary>
        /// コマンドリスト配列のチェック
        /// </summary>
        private bool ComparerCommandListArray(List<string[]> expectCommandListArray,
                List<string[]> checkCommandListArray)
        {
            if (checkCommandListArray == null)
            {
                return false;
            }
            // 配列の数が同じか確認
            if (expectCommandListArray.Count() != checkCommandListArray.Count())
            {
                return false;
            }
            // コマンドの中身が同じか確認
            for (var i = 0; i < expectCommandListArray.Count(); i++)
            {
                var expectStrArray = expectCommandListArray[i].ToArray();
                var checkStrArray = checkCommandListArray[i].ToArray();
                for (var j = 0; j < expectStrArray.Count(); j++)
                {
                    if (!expectStrArray[j].Equals(checkStrArray[j]))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        /// <summary>
        /// 単独コマンドの試験
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestSingleCommand()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize"
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize"
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// 単独コマンドの試験
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestSingleCommandAllOption()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// 複数コマンドの確認
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestMuitiCommand()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "--",
                "update",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize"
            });
            expectCommandListArray.Add(new string[]
            {
                "update"
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// オプションコマンド引き継ぎの確認
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestTakeOverOption()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "--hw-type=Sdev",
                "--",
                "update",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize",
                "--hw-type=Sdev"
            });
            expectCommandListArray.Add(new string[]
            {
                "update",
                "--hw-type=Sdev"
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// すべてのオプションコマンド引き継ぎの確認
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestTakeOverAllOption()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
                "--",
                "update",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
            });
            expectCommandListArray.Add(new string[]
            {
                "update",
                "--registered",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// オプションコマンド上書きのテスト
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestOverWriteOption()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "--hw-type=Sdev",
                "--",
                "update",
                "--hw-type=Edev",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize",
                "--hw-type=Sdev",
            });
            expectCommandListArray.Add(new string[]
            {
                "update",
                "--hw-type=Edev",
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }

        /// <summary>
        /// すべてのオプションコマンド上書き確認
        /// </summary>
        [TestMethod]
        public void DivideMultiCommandTestOverWriteAllOption()
        {
            // 入力コマンド
            var inputStringArray = new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
                "--",
                "update",
                "--connection-path=Usb",
                "--hw-config=EDEV_00_03_00_00",
                "--hw-type=Edev",
                "--ip-address=169.255.*",
                "--serial-number=XAM*",
                "--name=XAW02131014681",
                "--output=C:\templog2",
                "--registered",
                "-v",
            };

            // 期待値
            var expectCommandListArray = new List<string[]>() { };
            expectCommandListArray.Add(new string[]
            {
                "initialize",
                "-v",
                "--registered",
                "--output=C:\templog",
                "--name=XAW02131014680",
                "--serial-number=XAL*",
                "--ip-address=169.254.*",
                "--hw-type=Sdev",
                "--hw-config=SDEV_00_02_00_00",
                "--connection-path=HostBridge",
            });
            expectCommandListArray.Add(new string[]
            {
                "update",
                "--connection-path=Usb",
                "--hw-config=EDEV_00_03_00_00",
                "--hw-type=Edev",
                "--ip-address=169.255.*",
                "--serial-number=XAM*",
                "--name=XAW02131014681",
                "--output=C:\templog2",
                "--registered",
                "-v",
            });

            // 試験実施
            var commandListArray = this.privateType.InvokeStatic("DivideMultiCommand",
                    new object[] { inputStringArray }) as List<string[]>;

            // 結果確認
            Assert.IsTrue(ComparerCommandListArray(expectCommandListArray, commandListArray));
        }
    }
}
