﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Nintendo.ServiceFrameworkTest.CppCode
{
    [TestClass]
    public class CmifCppCodeEmitterTest
    {
        [CppFullName(@"nnt::ISimple")]
        [CppRefPath(@"<nnt/ISimple.h>")]
        [UseImpl(false)]
        private interface ISimple : nn.sf.IServiceObject
        {
        }

        [TestMethod]
        public void TestGetExternalIncludes()
        {
            AssertionUtility.AssertAreSetEqual(new[] { @"<nnt/ISimple.h>" }, typeof(ISimple).GetSfEntity().GetExternalIncludes());
        }

        [TestMethod]
        public void TestGetAdditionalIncludes()
        {
            AssertionUtility.AssertAreSetEqual(new[] { @"<nn/sf/detail/sf_AutogenInterfaceIncludes.h>", @"<nn/sf/cmif/detail/sf_CmifAutogenInterfaceIncludes.h>" }, typeof(ISimple).GetSfEntity().GetAdditionalIncludes());
        }

        [TestMethod]
        public void TestEmitForwardDeclarationCode()
        {
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(ISimple).GetSfEntity().EmitForwardDeclarationCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

class ISimple;

}
");
        }

        [CppFullName(@"nnt::IChild")]
        private interface IChild : nn.sf.IServiceObject
        {
        }

        [CppFullName(@"nnt::IChild2")]
        private interface IChild2 : nn.sf.IServiceObject
        {
        }

        [CppFullName(@"nnt::SmallStruct")]
        public struct SmallStruct
        {
            public int a;
        }

        [CppFullName(@"nnt::MediumStruct")]
        [Pre("_.b == _.c")]
        public struct MediumStruct
        {
            public char_t a;
            public int b;
            public ulong c;
        }

        [CppFullName(@"nnt::LargeStruct")]
        [LargeData]
        public struct LargeStruct
        {
            [FixedArray(1024)]
            public char_t[] a;
        }

        [CppFullName(@"nnt::LargeStruct")]
        [LargeData]
        [BufferTransferMode(BufferTransferMode.MapAlias)]
        public struct LargeStructByMapAlias
        {
            [FixedArray(1024)]
            public char_t[] a;
        }

        [Serializable]
        [AttributeUsage(AttributeTargets.Parameter | AttributeTargets.Struct, AllowMultiple = true)]
        public class NotZeroAttribute : PreAttribute
        {
            public NotZeroAttribute()
                : base("_ != 0")
            {
            }
        }

        [CppFullName(@"nnt::IComplex")]
        [UseImpl(false)]
        private interface IComplex : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void A();
            [MethodId(1)]
            void B([Pre(@"_ == 0")] short s16);
            [MethodId(2)]
            void C([NotZero] int s32, long s64);
            [MethodId(3)]
            nn.Result D();
            [MethodId(4)]
            nn.Result E(ushort u16);
            [MethodId(5)]
            nn.Result F(uint u32, ulong u64);
            [MethodId(6)]
            ushort G();
            [MethodId(7)]
            uint H(nn.sf.InBuffer inBuffer, nn.sf.OutBuffer outBuffer);
            [MethodId(19)]
            uint L(nn.sf.InArray<int> inArray, nn.sf.OutArray<int> outArray);
            [MethodId(8)]
            void O(nn.sf.Out<int> pOut);
            [MethodId(9)]
            IChild ReturnChild();
            [MethodId(10)]
            void InChild(IChild p);
            [MethodId(11)]
            void OutChild(nn.sf.Out<IChild> pOut);
            [MethodId(12)]
            IChild InOutComplex(IChild in1, nn.sf.Out<IChild> pOut1, IChild2 in2, nn.sf.Out<IChild2> pOut2);
            [MethodId(14)]
            void OutNativeHandle(nn.sf.Out<nn.sf.NativeHandle> pOut, nn.sf.Out<nn.sf.NativeHandle> pOut2, [MoveHandle] nn.sf.Out<nn.sf.NativeHandle> pOut3);
            [MethodId(15)]
            void MoveOutNativeHandle([MoveHandle] nn.sf.Out<nn.sf.NativeHandle> pOut);
            [MethodId(16)]
            void MoveInNativeHandle([MoveHandle] nn.sf.NativeHandle x);
            [MethodId(17)]
            void CopyInNativeHandle(nn.sf.NativeHandle x);
            [MethodId(18)]
            void InProcessId([ProcessId] nn.Bit64 pid);
            [MethodId(20)]
            void SmallStruct(nn.sf.Out<SmallStruct> o, SmallStruct i);
            [MethodId(21)]
            void SmallStructL([LargeData] nn.sf.Out<SmallStruct> o, [LargeData] SmallStruct i);
            [MethodId(22)]
            void MediumStruct(nn.sf.Out<MediumStruct> o, MediumStruct i);
            [MethodId(23)]
            void MediumStructL([LargeData] nn.sf.Out<MediumStruct> o, [LargeData] MediumStruct i);
            [MethodId(24)]
            void LargeStruct(nn.sf.Out<LargeStruct> o, LargeStruct i);
            [MethodId(25)]
            void BufferByPointer([BufferTransferMode(BufferTransferMode.PointerCopy)] nn.sf.InBuffer inBuffer, [BufferTransferMode(BufferTransferMode.PointerCopy)] nn.sf.OutBuffer outBuffer);
            [MethodId(26)]
            void ArrayByMapAlias([BufferTransferMode(BufferTransferMode.MapAlias)] nn.sf.InArray<std.int32_t> inArray, [BufferTransferMode(BufferTransferMode.MapAlias)] nn.sf.OutArray<std.int32_t> outArray);
            [MethodId(27)]
            void LargeStructByMapAlias(nn.sf.Out<LargeStructByMapAlias> o, LargeStructByMapAlias i);
            [MethodId(28)]
            void LargeStructByMapAliasByParameter([BufferTransferMode(BufferTransferMode.MapAlias)] nn.sf.Out<LargeStruct> o, [BufferTransferMode(BufferTransferMode.MapAlias)] LargeStruct i);
            [MethodId(29)]
            void ArrayByAuto([BufferTransferMode(BufferTransferMode.AutoSelect)] nn.sf.InArray<std.int32_t> inArray, [AutoSelectBuffer] nn.sf.OutArray<std.int32_t> outArray);
            [MethodId(30)]
            void LargeStructByAutoSelectByParameter([AutoSelectBuffer] nn.sf.Out<LargeStruct> o, [BufferTransferMode(BufferTransferMode.AutoSelect)] LargeStruct i);
            [MethodId(31)]
            void MapTransferSecurityParameter([MapTransferSecurity(MapTransferSecurity.Secure)] nn.sf.InBuffer a, [MapTransferSecurity(MapTransferSecurity.NonDevice)] nn.sf.OutBuffer b, [MapTransferSecurity(MapTransferSecurity.NonSecure)]nn.sf.InArray<char_t> c, [NonSecureMapTransfer] nn.sf.OutArray<char_t> d);
            [MethodId(32)]
            [UseCmif(false)]
            void NoMethodId(int a, float b, nn.sf.Out<int> pOut, nn.sf.InBuffer buffer);
        }

        [TestMethod]
        public void TestEmitDefinitionCode()
        {
            GenerateTestUtility.AssertGeneratedString(g =>
            {
                g.Newline();
                typeof(IComplex).GetSfEntity().EmitDefinitionCode(g);
                g.AutoNewLine();
                g.CloseCurrentNamespace();
            }, @"
namespace nnt {

NN_SF_DETAIL_DEFINE_SERVICE_INTERFACE(IComplex, (::nn::sf::IServiceObject))

    NN_SF_DETAIL_SERVICE_INTERFACE_BEGIN_SYNC_METHOD(IComplex)
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_0(A)
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(B, (::std::int16_t s16))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(C, (::std::int32_t s32, ::std::int64_t s64))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_RESULT_0(D)
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_RESULT_N(E, (::std::uint16_t u16))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_RESULT_N(F, (::std::uint32_t u32, ::std::uint64_t u64))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VALUE_0(G, (::std::uint16_t))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VALUE_N(H, (::std::uint32_t), (const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(O, (::nn::sf::Out<::std::int32_t> pOut))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VALUE_0(ReturnChild, (::nn::sf::SharedPointer<::nnt::IChild>))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(InChild, (::nn::sf::SharedPointer<::nnt::IChild> p))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(OutChild, (::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VALUE_N(InOutComplex, (::nn::sf::SharedPointer<::nnt::IChild>), (::nn::sf::SharedPointer<::nnt::IChild> in1, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut1, ::nn::sf::SharedPointer<::nnt::IChild2> in2, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild2>> pOut2))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(OutNativeHandle, (::nn::sf::Out<::nn::sf::NativeHandle> pOut, ::nn::sf::Out<::nn::sf::NativeHandle> pOut2, ::nn::sf::Out<::nn::sf::NativeHandle> pOut3))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(MoveOutNativeHandle, (::nn::sf::Out<::nn::sf::NativeHandle> pOut))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(MoveInNativeHandle, (::nn::sf::NativeHandle&& x))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(CopyInNativeHandle, (::nn::sf::NativeHandle&& x))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(InProcessId, (::nn::Bit64 pid))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VALUE_N(L, (::std::uint32_t), (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(SmallStruct, (::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(SmallStructL, (::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(MediumStruct, (::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(MediumStructL, (::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(LargeStruct, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(BufferByPointer, (const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(ArrayByMapAlias, (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByMapAlias, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByMapAliasByParameter, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(ArrayByAuto, (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByAutoSelectByParameter, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(MapTransferSecurityParameter, (const ::nn::sf::InBuffer& a, const ::nn::sf::OutBuffer& b, const ::nn::sf::InArray<char>& c, const ::nn::sf::OutArray<char>& d))
        NN_SF_DETAIL_INTERNAL_INTERFACE_DEFINE_SYNC_METHOD_VOID_N(NoMethodId, (::std::int32_t a, float b, ::nn::sf::Out<::std::int32_t> pOut, const ::nn::sf::InBuffer& buffer))
    NN_SF_DETAIL_SERVICE_INTERFACE_END_SYNC_METHOD

    NN_SF_DETAIL_SERVICE_INTERFACE_BEGIN_ACCESSOR(IComplex)

        void A() NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_0(A);
        }

        void B(::std::int16_t s16) NN_NOEXCEPT
        {
            NN_SF_DETAIL_CALL_INTERNAL_ASSERT_PRE((NN_SF_DETAIL_APPLY_LAMBDA(_, (_ == 0), s16)), ""_ == 0 (where _ = s16)"")
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(B, (s16));
        }

        void C(::std::int32_t s32, ::std::int64_t s64) NN_NOEXCEPT
        {
            NN_SF_DETAIL_CALL_INTERNAL_ASSERT_PRE((NN_SF_DETAIL_APPLY_LAMBDA(_, (_ != 0), s32)), ""_ != 0 (where _ = s32)"")
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(C, (s32, s64));
        }

        ::nn::Result D() NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_RESULT_0(D);
        }

        ::nn::Result E(::std::uint16_t u16) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_RESULT_N(E, (u16));
        }

        ::nn::Result F(::std::uint32_t u32, ::std::uint64_t u64) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_RESULT_N(F, (u32, u64));
        }

        ::std::uint16_t G() NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VALUE_0(G, (::std::uint16_t));
        }

        ::std::uint32_t H(const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VALUE_N(H, (::std::uint32_t), (inBuffer, outBuffer));
        }

        void O(::nn::sf::Out<::std::int32_t> pOut) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(O, (pOut));
        }

        ::nn::sf::SharedPointer<::nnt::IChild> ReturnChild() NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VALUE_0(ReturnChild, (::nn::sf::SharedPointer<::nnt::IChild>));
        }

        void InChild(::nn::sf::SharedPointer<::nnt::IChild> p) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(InChild, (p));
        }

        void OutChild(::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(OutChild, (pOut));
        }

        ::nn::sf::SharedPointer<::nnt::IChild> InOutComplex(::nn::sf::SharedPointer<::nnt::IChild> in1, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut1, ::nn::sf::SharedPointer<::nnt::IChild2> in2, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild2>> pOut2) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VALUE_N(InOutComplex, (::nn::sf::SharedPointer<::nnt::IChild>), (in1, pOut1, in2, pOut2));
        }

        void OutNativeHandle(::nn::sf::Out<::nn::sf::NativeHandle> pOut, ::nn::sf::Out<::nn::sf::NativeHandle> pOut2, ::nn::sf::Out<::nn::sf::NativeHandle> pOut3) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(OutNativeHandle, (pOut, pOut2, pOut3));
        }

        void MoveOutNativeHandle(::nn::sf::Out<::nn::sf::NativeHandle> pOut) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(MoveOutNativeHandle, (pOut));
        }

        void MoveInNativeHandle(::nn::sf::NativeHandle&& x) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(MoveInNativeHandle, (::std::move(x)));
        }

        void CopyInNativeHandle(::nn::sf::NativeHandle&& x) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(CopyInNativeHandle, (::std::move(x)));
        }

        void InProcessId(::nn::Bit64 pid) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(InProcessId, (pid));
        }

        ::std::uint32_t L(const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VALUE_N(L, (::std::uint32_t), (inArray, outArray));
        }

        void SmallStruct(::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(SmallStruct, (o, i));
        }

        void SmallStructL(::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(SmallStructL, (o, i));
        }

        void MediumStruct(::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_CALL_INTERNAL_ASSERT_PRE((NN_SF_DETAIL_APPLY_LAMBDA(_, (_.b == _.c), i)), ""_.b == _.c (where _ = i)"")
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(MediumStruct, (o, i));
        }

        void MediumStructL(::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_CALL_INTERNAL_ASSERT_PRE((NN_SF_DETAIL_APPLY_LAMBDA(_, (_.b == _.c), i)), ""_.b == _.c (where _ = i)"")
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(MediumStructL, (o, i));
        }

        void LargeStruct(::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(LargeStruct, (o, i));
        }

        void BufferByPointer(const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(BufferByPointer, (inBuffer, outBuffer));
        }

        void ArrayByMapAlias(const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(ArrayByMapAlias, (inArray, outArray));
        }

        void LargeStructByMapAlias(::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(LargeStructByMapAlias, (o, i));
        }

        void LargeStructByMapAliasByParameter(::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(LargeStructByMapAliasByParameter, (o, i));
        }

        void ArrayByAuto(const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(ArrayByAuto, (inArray, outArray));
        }

        void LargeStructByAutoSelectByParameter(::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(LargeStructByAutoSelectByParameter, (o, i));
        }

        void MapTransferSecurityParameter(const ::nn::sf::InBuffer& a, const ::nn::sf::OutBuffer& b, const ::nn::sf::InArray<char>& c, const ::nn::sf::OutArray<char>& d) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(MapTransferSecurityParameter, (a, b, c, d));
        }

        void NoMethodId(::std::int32_t a, float b, ::nn::sf::Out<::std::int32_t> pOut, const ::nn::sf::InBuffer& buffer) NN_NOEXCEPT
        {
            NN_SF_DETAIL_RETURN_INTERNAL_SYNC_METHOD_VOID_N(NoMethodId, (a, b, pOut, buffer));
        }

    NN_SF_DETAIL_SERVICE_INTERFACE_END_ACCESSOR

NN_SF_DETAIL_DEFINE_SERVICE_INTERFACE_END

#if 0
    class IComplex
    {
    public:
        void A() NN_NOEXCEPT;
        void B(std::int16_t s16) NN_NOEXCEPT;
        void C(std::int32_t s32, std::int64_t s64) NN_NOEXCEPT;
        nn::Result D() NN_NOEXCEPT;
        nn::Result E(std::uint16_t u16) NN_NOEXCEPT;
        nn::Result F(std::uint32_t u32, std::uint64_t u64) NN_NOEXCEPT;
        std::uint16_t G() NN_NOEXCEPT;
        std::uint32_t H(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT;
        void O(nn::sf::Out<std::int32_t> pOut) NN_NOEXCEPT;
        nn::sf::SharedPointer<nnt::IChild> ReturnChild() NN_NOEXCEPT;
        void InChild(nn::sf::SharedPointer<nnt::IChild> p) NN_NOEXCEPT;
        void OutChild(nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut) NN_NOEXCEPT;
        nn::sf::SharedPointer<nnt::IChild> InOutComplex(nn::sf::SharedPointer<nnt::IChild> in1, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut1, nn::sf::SharedPointer<nnt::IChild2> in2, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild2>> pOut2) NN_NOEXCEPT;
        void OutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut, nn::sf::Out<nn::sf::NativeHandle> pOut2, nn::sf::Out<nn::sf::NativeHandle> pOut3) NN_NOEXCEPT;
        void MoveOutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT;
        void MoveInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT;
        void CopyInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT;
        void InProcessId(nn::Bit64 pid) NN_NOEXCEPT;
        std::uint32_t L(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT;
        void SmallStruct(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT;
        void SmallStructL(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT;
        void MediumStruct(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT;
        void MediumStructL(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT;
        void LargeStruct(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT;
        void BufferByPointer(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT;
        void ArrayByMapAlias(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT;
        void LargeStructByMapAlias(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT;
        void LargeStructByMapAliasByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT;
        void ArrayByAuto(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT;
        void LargeStructByAutoSelectByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT;
        void MapTransferSecurityParameter(const nn::sf::InBuffer& a, const nn::sf::OutBuffer& b, const nn::sf::InArray<char>& c, const nn::sf::OutArray<char>& d) NN_NOEXCEPT;
        void NoMethodId(std::int32_t a, float b, nn::sf::Out<std::int32_t> pOut, const nn::sf::InBuffer& buffer) NN_NOEXCEPT;
    };
#endif

}

// template <> struct nn::sf::cmif::detail::MethodInfos<nnt::IComplex>
NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFOS((::nnt::IComplex))

    // void A() NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(A)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS(())

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(A)

    // void B(std::int16_t s16) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(B)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(s16, (::std::int16_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(1, (s16))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(2)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(s16)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(B)

    // void C(std::int32_t s32, std::int64_t s64) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(C)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(s32, (::std::int32_t), 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(s64, (::std::int64_t), 8)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(2, (s32, s64))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(16)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(s32), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(s64)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(C)

    // nn::Result D() NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(D)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(3)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_RESULT(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS(())

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(D)

    // nn::Result E(std::uint16_t u16) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(E)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(4)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_RESULT(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(u16, (::std::uint16_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(1, (u16))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(2)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(u16)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(E)

    // nn::Result F(std::uint32_t u32, std::uint64_t u64) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(F)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(5)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_RESULT(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(u32, (::std::uint32_t), 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(u64, (::std::uint64_t), 8)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(2, (u32, u64))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(16)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(u32), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(u64)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(F)

    // std::uint16_t G() NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(G)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(6)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME, (::std::uint16_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (NN_SF_DETAIL_OUT_RETURN_NAME))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(2)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(G)

    // std::uint32_t H(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(H)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(7)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(inBuffer, 0, (const ::nn::sf::InBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(outBuffer, 1, (const ::nn::sf::OutBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (inBuffer, outBuffer))

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME, (::std::uint32_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (NN_SF_DETAIL_OUT_RETURN_NAME))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(inBuffer), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(outBuffer)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(H)

    // void O(nn::sf::Out<std::int32_t> pOut) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(O)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(8)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(pOut, (::std::int32_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (pOut))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(O)

    // nn::sf::SharedPointer<nnt::IChild> ReturnChild() NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(ReturnChild)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(9)

        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME, 0, 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS_END(1, (NN_SF_DETAIL_OUT_RETURN_NAME))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(ReturnChild)

    // void InChild(nn::sf::SharedPointer<nnt::IChild> p) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(InChild)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(10)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFO(p, 0, 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS_END(1, (p))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(p)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(InChild)

    // void OutChild(nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(OutChild)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(11)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO(pOut, 0, 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS_END(1, (pOut))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(OutChild)

    // nn::sf::SharedPointer<nnt::IChild> InOutComplex(nn::sf::SharedPointer<nnt::IChild> in1, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut1, nn::sf::SharedPointer<nnt::IChild2> in2, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild2>> pOut2) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(InOutComplex)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(12)

        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFO(in1, 0, 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFO(in2, 1, 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_OBJECT_INFOS_END(2, (in1, in2))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_COUNT(3)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME, 0, 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO(pOut1, 1, 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFO(pOut2, 2, 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_OBJECT_INFOS_END(3, (NN_SF_DETAIL_OUT_RETURN_NAME, pOut1, pOut2))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(in1), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut1), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(in2), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut2)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(InOutComplex)

    // void OutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut, nn::sf::Out<nn::sf::NativeHandle> pOut2, nn::sf::Out<nn::sf::NativeHandle> pOut3) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(OutNativeHandle)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(14)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_COUNT(3)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFO(pOut, 0, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY)
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFO(pOut2, 1, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY)
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFO(pOut3, 2, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS_END(3, (pOut, pOut2, pOut3))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut2), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut3)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(OutNativeHandle)

    // void MoveOutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(MoveOutNativeHandle)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(15)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFO(pOut, 0, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE)
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_NATIVE_HANDLE_INFOS_END(1, (pOut))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(MoveOutNativeHandle)

    // void MoveInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(MoveInNativeHandle)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(16)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFO(x, 0, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_MOVE)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS_END(1, (x))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(x)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(MoveInNativeHandle)

    // void CopyInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(CopyInNativeHandle)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(17)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFO(x, 0, NN_SF_CMIF_DETAIL_NATIVE_HANDLE_ATTRIBUTE_HIPC_COPY)
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_NATIVE_HANDLE_INFOS_END(1, (x))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(x)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(CopyInNativeHandle)

    // void InProcessId(nn::Bit64 pid) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(InProcessId)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(18)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_PROCESS_ID_ENABLE()

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(pid, (::nn::Bit64), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(1, (pid))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(8)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pid)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(InProcessId)

    // std::uint32_t L(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(L)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(19)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(inArray, 0, (const ::nn::sf::InArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(outArray, 1, (const ::nn::sf::OutArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (inArray, outArray))

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME, (::std::uint32_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (NN_SF_DETAIL_OUT_RETURN_NAME))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(NN_SF_DETAIL_OUT_RETURN_NAME), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(inArray), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(outArray)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(L)

    // void SmallStruct(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(SmallStruct)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(20)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(i, (::nnt::SmallStruct), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(1, (i))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(o, (::nnt::SmallStruct), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (o))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(SmallStruct)

    // void SmallStructL(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(SmallStructL)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(21)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::SmallStruct>), 4, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (::nnt::SmallStruct), 4, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(SmallStructL)

    // void MediumStruct(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(MediumStruct)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(22)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(i, (::nnt::MediumStruct), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(1, (i))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(16)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(o, (::nnt::MediumStruct), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (o))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(16)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(MediumStruct)

    // void MediumStructL(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(MediumStructL)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(23)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::MediumStruct>), 16, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (const ::nnt::MediumStruct&), 16, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(MediumStructL)

    // void LargeStruct(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(LargeStruct)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(24)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::LargeStruct>), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (const ::nnt::LargeStruct&), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(LargeStruct)

    // void BufferByPointer(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(BufferByPointer)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(25)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(inBuffer, 0, (const ::nn::sf::InBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(outBuffer, 1, (const ::nn::sf::OutBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (inBuffer, outBuffer))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(inBuffer), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(outBuffer)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(BufferByPointer)

    // void ArrayByMapAlias(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(ArrayByMapAlias)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(26)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(inArray, 0, (const ::nn::sf::InArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(outArray, 1, (const ::nn::sf::OutArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (inArray, outArray))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(inArray), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(outArray)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(ArrayByMapAlias)

    // void LargeStructByMapAlias(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(LargeStructByMapAlias)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(27)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::LargeStruct>), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (const ::nnt::LargeStruct&), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(LargeStructByMapAlias)

    // void LargeStructByMapAliasByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(LargeStructByMapAliasByParameter)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(28)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::LargeStruct>), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (const ::nnt::LargeStruct&), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(LargeStructByMapAliasByParameter)

    // void ArrayByAuto(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(ArrayByAuto)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(29)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(inArray, 0, (const ::nn::sf::InArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(outArray, 1, (const ::nn::sf::OutArray<::std::int32_t>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (inArray, outArray))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(inArray), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(outArray)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(ArrayByAuto)

    // void LargeStructByAutoSelectByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(LargeStructByAutoSelectByParameter)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(30)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(2)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(o, 0, (::nn::sf::Out<::nnt::LargeStruct>), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT)
            NN_SF_CMIF_DETAIL_METHOD_INFO_FIXED_SIZE_BUFFER_INFO(i, 1, (const ::nnt::LargeStruct&), 1024, NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(2, (o, i))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(o), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(i)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(LargeStructByAutoSelectByParameter)

    // void MapTransferSecurityParameter(const nn::sf::InBuffer& a, const nn::sf::OutBuffer& b, const nn::sf::InArray<char>& c, const nn::sf::OutArray<char>& d) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(MapTransferSecurityParameter)

        NN_SF_CMIF_DETAIL_METHOD_INFO_METHOD_ID(31)
        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(4)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(a, 0, (const ::nn::sf::InBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(b, 1, (const ::nn::sf::OutBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_DEVICE)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(c, 2, (const ::nn::sf::InArray<char>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_SECURE)
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(d, 3, (const ::nn::sf::OutArray<char>&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_SECURE)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(4, (a, b, c, d))

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(a), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(b), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(c), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(d)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(MapTransferSecurityParameter)

    // void NoMethodId(std::int32_t a, float b, nn::sf::Out<std::int32_t> pOut, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO(NoMethodId)

        NN_SF_CMIF_DETAIL_METHOD_INFO_RETURN_TYPE_IS_VOID(true)

        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_COUNT(1)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS
            NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFO(buffer, 0, (const ::nn::sf::InBuffer&), NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN | NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS)
        NN_SF_CMIF_DETAIL_METHOD_INFO_BUFFER_INFOS_END(1, (buffer))

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(a, (::std::int32_t), 0)
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_MEMBER(b, (float), 4)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_IN_RAW_STRUCT_END(2, (a, b))
        NN_SF_CMIF_DETAIL_METHOD_INFO_IN_RAW_SIZE(8)

        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT
            NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_MEMBER(pOut, (::std::int32_t), 0)
        NN_SF_CMIF_DETAIL_METHOD_INFO_DEFINE_OUT_RAW_STRUCT_END(1, (pOut))
        NN_SF_CMIF_DETAIL_METHOD_INFO_OUT_RAW_SIZE(4)

        NN_SF_CMIF_DETAIL_METHOD_INFO_ARGUMENT_INFOS((NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(a), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(b), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(pOut), NN_SF_CMIF_DETAIL_METHOD_INFO_GET_ARGUMENT_INFO(buffer)))

    NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFO_END(NoMethodId)

    NN_SF_CMIF_DETAIL_METHOD_INFO_REFERRED_STRUCT_ASSERTION_CHECK((::nnt::MediumStruct), 16, 8)

NN_SF_CMIF_DETAIL_DEFINE_METHOD_INFOS_END

// template <...> class nn::sf::client::detail::CmifProxy<nnt::IComplex, ...>
NN_SF_CMIF_CLIENT_DETAIL_DEFINE_PROXY((::nnt::IComplex))

    // void A() NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_0(A)
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(A)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(A)
    }

    // void B(std::int16_t s16) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(B, (::std::int16_t s16))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(B)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(s16)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(B)
    }

    // void C(std::int32_t s32, std::int64_t s64) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(C, (::std::int32_t s32, ::std::int64_t s64))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(C)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(s32)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(s64)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(C)
    }

    // nn::Result D() NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_RESULT_0(D)
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(D)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(D)
    }

    // nn::Result E(std::uint16_t u16) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_RESULT_N(E, (::std::uint16_t u16))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(E)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(u16)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(E)
    }

    // nn::Result F(std::uint32_t u32, std::uint64_t u64) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_RESULT_N(F, (::std::uint32_t u32, ::std::uint64_t u64))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(F)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(u32)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(u64)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(F)
    }

    // std::uint16_t G() NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VALUE_0(G, (::std::uint16_t))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(G)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(G)
    }

    // std::uint32_t H(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VALUE_N(H, (::std::uint32_t), (const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(H)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(NN_SF_DETAIL_OUT_RETURN_NAME)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(inBuffer)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(outBuffer)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(H)
    }

    // void O(nn::sf::Out<std::int32_t> pOut) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(O, (::nn::sf::Out<::std::int32_t> pOut))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(O)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(O)
    }

    // nn::sf::SharedPointer<nnt::IChild> ReturnChild() NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VALUE_0(ReturnChild, (::nn::sf::SharedPointer<::nnt::IChild>))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(ReturnChild)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(ReturnChild)
    }

    // void InChild(nn::sf::SharedPointer<nnt::IChild> p) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(InChild, (::nn::sf::SharedPointer<::nnt::IChild> p))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(InChild)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(p)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(InChild)
    }

    // void OutChild(nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(OutChild, (::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(OutChild)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(OutChild)
    }

    // nn::sf::SharedPointer<nnt::IChild> InOutComplex(nn::sf::SharedPointer<nnt::IChild> in1, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild>> pOut1, nn::sf::SharedPointer<nnt::IChild2> in2, nn::sf::Out<nn::sf::SharedPointer<nnt::IChild2>> pOut2) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VALUE_N(InOutComplex, (::nn::sf::SharedPointer<::nnt::IChild>), (::nn::sf::SharedPointer<::nnt::IChild> in1, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild>> pOut1, ::nn::sf::SharedPointer<::nnt::IChild2> in2, ::nn::sf::Out<::nn::sf::SharedPointer<::nnt::IChild2>> pOut2))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(InOutComplex)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(NN_SF_DETAIL_OUT_RETURN_NAME)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(in1)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut1)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(in2)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut2)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(InOutComplex)
    }

    // void OutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut, nn::sf::Out<nn::sf::NativeHandle> pOut2, nn::sf::Out<nn::sf::NativeHandle> pOut3) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(OutNativeHandle, (::nn::sf::Out<::nn::sf::NativeHandle> pOut, ::nn::sf::Out<::nn::sf::NativeHandle> pOut2, ::nn::sf::Out<::nn::sf::NativeHandle> pOut3))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(OutNativeHandle)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut2)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut3)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(OutNativeHandle)
    }

    // void MoveOutNativeHandle(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(MoveOutNativeHandle, (::nn::sf::Out<::nn::sf::NativeHandle> pOut))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(MoveOutNativeHandle)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pOut)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(MoveOutNativeHandle)
    }

    // void MoveInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(MoveInNativeHandle, (::nn::sf::NativeHandle&& x))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(MoveInNativeHandle)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(x)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(MoveInNativeHandle)
    }

    // void CopyInNativeHandle(nn::sf::NativeHandle&& x) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(CopyInNativeHandle, (::nn::sf::NativeHandle&& x))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(CopyInNativeHandle)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(x)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(CopyInNativeHandle)
    }

    // void InProcessId(nn::Bit64 pid) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(InProcessId, (::nn::Bit64 pid))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(InProcessId)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(pid)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(InProcessId)
    }

    // std::uint32_t L(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VALUE_N(L, (::std::uint32_t), (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(L)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(NN_SF_DETAIL_OUT_RETURN_NAME)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(inArray)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(outArray)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(L)
    }

    // void SmallStruct(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(SmallStruct, (::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(SmallStruct)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(SmallStruct)
    }

    // void SmallStructL(nn::sf::Out<nnt::SmallStruct> o, nnt::SmallStruct i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(SmallStructL, (::nn::sf::Out<::nnt::SmallStruct> o, ::nnt::SmallStruct i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(SmallStructL)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(SmallStructL)
    }

    // void MediumStruct(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(MediumStruct, (::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(MediumStruct)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(MediumStruct)
    }

    // void MediumStructL(nn::sf::Out<nnt::MediumStruct> o, const nnt::MediumStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(MediumStructL, (::nn::sf::Out<::nnt::MediumStruct> o, const ::nnt::MediumStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(MediumStructL)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(MediumStructL)
    }

    // void LargeStruct(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(LargeStruct, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(LargeStruct)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(LargeStruct)
    }

    // void BufferByPointer(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(BufferByPointer, (const ::nn::sf::InBuffer& inBuffer, const ::nn::sf::OutBuffer& outBuffer))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(BufferByPointer)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(inBuffer)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(outBuffer)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(BufferByPointer)
    }

    // void ArrayByMapAlias(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(ArrayByMapAlias, (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(ArrayByMapAlias)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(inArray)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(outArray)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(ArrayByMapAlias)
    }

    // void LargeStructByMapAlias(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByMapAlias, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(LargeStructByMapAlias)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(LargeStructByMapAlias)
    }

    // void LargeStructByMapAliasByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByMapAliasByParameter, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(LargeStructByMapAliasByParameter)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(LargeStructByMapAliasByParameter)
    }

    // void ArrayByAuto(const nn::sf::InArray<std::int32_t>& inArray, const nn::sf::OutArray<std::int32_t>& outArray) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(ArrayByAuto, (const ::nn::sf::InArray<::std::int32_t>& inArray, const ::nn::sf::OutArray<::std::int32_t>& outArray))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(ArrayByAuto)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(inArray)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(outArray)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(ArrayByAuto)
    }

    // void LargeStructByAutoSelectByParameter(nn::sf::Out<nnt::LargeStruct> o, const nnt::LargeStruct& i) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(LargeStructByAutoSelectByParameter, (::nn::sf::Out<::nnt::LargeStruct> o, const ::nnt::LargeStruct& i))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(LargeStructByAutoSelectByParameter)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(o)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(i)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(LargeStructByAutoSelectByParameter)
    }

    // void MapTransferSecurityParameter(const nn::sf::InBuffer& a, const nn::sf::OutBuffer& b, const nn::sf::InArray<char>& c, const nn::sf::OutArray<char>& d) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(MapTransferSecurityParameter, (const ::nn::sf::InBuffer& a, const ::nn::sf::OutBuffer& b, const ::nn::sf::InArray<char>& c, const ::nn::sf::OutArray<char>& d))
    {
        NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN(MapTransferSecurityParameter)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(a)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(b)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(c)
            NN_SF_CMIF_CLIENT_DETAIL_CALL_ARGUMENT(d)
        NN_SF_CMIF_CLIENT_DETAIL_CALL_END(MapTransferSecurityParameter)
    }

    // void NoMethodId(std::int32_t a, float b, nn::sf::Out<std::int32_t> pOut, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    NN_SF_IMPL_DETAIL_IMPL_TEMPLATE_BASE_DEFINE_SYNC_METHOD_VOID_N(NoMethodId, (::std::int32_t a, float b, ::nn::sf::Out<::std::int32_t> pOut, const ::nn::sf::InBuffer& buffer))
    {
        NN_UNUSED(a);
        NN_UNUSED(b);
        NN_UNUSED(pOut);
        NN_UNUSED(buffer);
        NN_SF_CMIF_CLIENT_DETAIL_PROXY_DEFINE_SYNC_METHOD_UNSUPPORTED
    }

NN_SF_CMIF_CLIENT_DETAIL_DEFINE_PROXY_END

// template <> class nn::sf::server::detail::CmifProcessFunctionTableGetterImpl<nnt::IComplex>
NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER((::nnt::IComplex))

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(A)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(A)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(B)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE(NN_SF_DETAIL_APPLY_LAMBDA(_, (_ == 0), NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(s16)))
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(B)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(s16)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(C)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE(NN_SF_DETAIL_APPLY_LAMBDA(_, (_ != 0), NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(s32)))
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(C)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(s32),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(s64)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(D)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(D)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(E)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(E)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(u16)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(F)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(F)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(u32),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(u64)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(G)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(G)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(H)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(inBuffer)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(outBuffer)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(H)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(inBuffer),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(outBuffer)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(O)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(O)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(pOut)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(ReturnChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(ReturnChild)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(NN_SF_DETAIL_OUT_RETURN_NAME, 0)
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(InChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_IN_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), p, 0)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(InChild)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_OBJECT(p)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(OutChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), pOut)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(OutChild)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(pOut)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(pOut, 0)
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(InOutComplex)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_IN_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), in1, 0)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_IN_OBJECT((::nn::sf::SharedPointer<::nnt::IChild2>), in2, 1)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), NN_SF_DETAIL_OUT_RETURN_NAME)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT((::nn::sf::SharedPointer<::nnt::IChild>), pOut1)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_DEFINE_OUT_OBJECT((::nn::sf::SharedPointer<::nnt::IChild2>), pOut2)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(InOutComplex)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(NN_SF_DETAIL_OUT_RETURN_NAME),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_OBJECT(in1),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(pOut1),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_OBJECT(in2),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_OBJECT(pOut2)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(NN_SF_DETAIL_OUT_RETURN_NAME, 0)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(pOut1, 1)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_SET_OUT_OBJECT(pOut2, 2)
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(OutNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE(pOut)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE(pOut2)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE(pOut3)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(OutNativeHandle)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE(pOut),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE(pOut2),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE(pOut3)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(MoveOutNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLE(pOut)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_OUT_NATIVE_HANDLES_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(MoveOutNativeHandle)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_NATIVE_HANDLE(pOut)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(MoveInNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLE(x)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(MoveInNativeHandle)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_NATIVE_HANDLE(x)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(CopyInNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLE(x)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_IN_NATIVE_HANDLES_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(CopyInNativeHandle)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_NATIVE_HANDLE(x)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(InProcessId)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OVERWRITE_IN_PROCESS_ID(pid)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(InProcessId)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(pid)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(L)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(inArray)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(outArray)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(L)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(NN_SF_DETAIL_OUT_RETURN_NAME),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(inArray),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(outArray)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(SmallStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(SmallStruct)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(SmallStructL)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(SmallStructL)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(MediumStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE(NN_SF_DETAIL_APPLY_LAMBDA(_, (_.b == _.c), NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(i)))
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(MediumStruct)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_OUT_MEMBER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_IN_MEMBER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(MediumStructL)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_CHECK_PRE(NN_SF_DETAIL_APPLY_LAMBDA(_, (_.b == _.c), NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)))
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(MediumStructL)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(LargeStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(LargeStruct)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(BufferByPointer)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(inBuffer)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(outBuffer)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(BufferByPointer)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(inBuffer),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(outBuffer)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(ArrayByMapAlias)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(inArray)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(outArray)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(ArrayByMapAlias)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(inArray),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(outArray)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(LargeStructByMapAlias)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(LargeStructByMapAlias)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(LargeStructByMapAliasByParameter)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(LargeStructByMapAliasByParameter)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(ArrayByAuto)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(inArray)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(outArray)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(ArrayByAuto)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(inArray),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(outArray)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(LargeStructByAutoSelectByParameter)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(o)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(i)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(LargeStructByAutoSelectByParameter)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(o),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(i)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD(MapTransferSecurityParameter)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(a)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(b)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(c)
                NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFER(d)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_BUFFERS_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_PREPARE_END
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE(MapTransferSecurityParameter)
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(a),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(b),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(c),
            NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_BUFFER(d)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_INVOKE_END
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_END

    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(A)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(B)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(C)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(D)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(E)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(F)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(G)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(H)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(O)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(ReturnChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(InChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(OutChild)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(InOutComplex)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(OutNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(MoveOutNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(MoveInNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(CopyInNativeHandle)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(InProcessId)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(L)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(SmallStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(SmallStructL)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(MediumStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(MediumStructL)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(LargeStruct)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(BufferByPointer)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(ArrayByMapAlias)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(LargeStructByMapAlias)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(LargeStructByMapAliasByParameter)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(ArrayByAuto)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(LargeStructByAutoSelectByParameter)
        NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_CASE(MapTransferSecurityParameter)
    NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_MESSAGE_END

NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_END

NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCHER_FUNCTION_TABLE((::nnt::IComplex))
");
        }

        [TestMethod]
        public void TestEmitReferenceCheckCode()
        {
            GenerateTestUtility.AssertGenerateEmpty(typeof(IComplex).GetSfEntity().EmitReferenceCheckCode);
        }
    }
}
