﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    [TestClass]
    public class TestReadDescFile : TestBase
    {
        private XmlFile metaFile;
        private XmlElement acidElement;

        public TestReadDescFile()
        {
            metaFile = new XmlFile();
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement programId = new XmlElement("ProgramId");
            programId.InnerText = "0x0005000C10000000";
            core.AddChild(programId);

            metaFile.AddLine(elem.GetXml());
            metaFile.WriteData();

            acidElement = new XmlElement("Acid");
            // 読み込みのみのテストなので、Acid のバイナリは何でもよい
            acidElement.InnerText = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABBQ0lEgAEAAAAAAAAAAAAAAQAAAAAAAAD//wAAAAAAAEACAAAAAAAAQAIAAA0AAABQAgAAKAAAAAAAAAAAAAAAhVNlcnZlcgRDbGllKgAAAPdjAAHPAAAAPwgAgD8BAIA/EAAAvwEAAP8XgAD/vwkA/38AAv//AgAAAAAAAAAAAA==";
        }

        [TestMethod]
        public void TestDescTag()
        {
            // Desc ファイルを読み込める
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 想定されていないタグを受け取らない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement core = new XmlElement("Core");
                elem.AddChild(core);
                XmlElement defaultElem = CreateDefaultDescParam();
                core.AddChild(defaultElem);
                core.AddChild(acidElement);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, metaFile, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                desc.AddLine(elem.GetXml());
                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestProgramIdMin()
        {
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);

                XmlElement idTag2 = new XmlElement("ProgramIdMax");
                idTag2.InnerText = "0x2";
                elem.AddChild(idTag2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestProgramIdMax()
        {
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestFsAccessControlDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("FsAccessControlDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestSrvAccessControlDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // Entry を読み込める
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }
        }

        [TestMethod, TestCategory("LongTest")]
        public void TestKernelCapabilityDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement idTag = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag);
                XmlElement idTag2 = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "KernelCapabilityDescriptor"), desc, metaFile, null, null);
            }

            // ThreadInfo を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeThreadInfoElement("0", "0", "0", "0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeThreadInfoElement("63", "63", "255", "255"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement("-1", "0", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", 6),
                    TestDataForKc.MakeThreadInfoElement("64", "0", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement(null, "0", "0", "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", "-1", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", 6),
                    TestDataForKc.MakeThreadInfoElement("0", "64", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", null, "0", "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "-1", "0"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "256", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", null, "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "-1"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "256"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", null));

                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", "1", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "1", "0"));
            }

            // EnableSystemCalls を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableSystemCall("CreateThread"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    TestDataForKc.MakeEnableSystemCall("SleepThread"),
                });
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", "1"),
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCallClass("CreateThread", "1"),
                        TestDataForKc.MakeEnableSystemCallClass("CreateThread", "1"),
                    }
                );
             }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x30000", "0x1000", "RW", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x40000", "0x1000", "Rw", "Static"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x50000", "0x3000", "Ro", "Static"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x30000", "0x2000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x40000", "0x2000", "RW", "Static"),
                    TestDataForKc.MakeMemoryMap("0x50000", "0x2000", "Ro", "Static"),
                });
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x0", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0xFFFFFF000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "MemoryMap/BeginAddress"),
                    TestDataForKc.MakeMemoryMap("10000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35),
                    TestDataForKc.MakeMemoryMap("0x1000000000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/BeginAddress"),
                    TestDataForKc.MakeMemoryMap(null, "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "MemoryMap/Size"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x0", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeMemoryMap("0x10000", "0x100000000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Size"),
                    TestDataForKc.MakeMemoryMap("0x10000", null, "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Permission", "RO", "RW"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "RX", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Permission"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", null, "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Type", "Io", "Static"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Normal"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Type"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", null));
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("10"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("1022"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("1023"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("1023"),
                    TestDataForKc.MakeEnableInterrupt("23"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("0x10"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableInterrupts", "45"),
                    new XmlElement[] {
                        TestDataForKc.MakeEnableInterrupt("45"),
                        TestDataForKc.MakeEnableInterrupt("45"),
                    }
                );
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("-1"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "EnableInterrupts", 1024),
                    TestDataForKc.MakeEnableInterrupt("1024"));
            }

            // MiscParams を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("1"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("7"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", 8),
                    TestDataForKc.MakeMiscParams("8"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeMiscParams("-1"));
            }

            // HandleTableSize を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("256"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("1023"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidValue_Bigger, "KernelCapabilityDescriptor/HandleTableSize", 1023),
                    TestDataForKc.MakeHandleTableSize("1024"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityDescriptor/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("-1"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityDescriptor/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("0x1"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("True", "True"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("True", null));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags(null, "True"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("FALSE", null));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags(null, "FALSE"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/EnableDebug"),
                    TestDataForKc.MakeMiscFlags("Success", "FALSE"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/ForceDebug"),
                    TestDataForKc.MakeMiscFlags("True", "Fail"));
            }
        }

        [TestMethod]
        public void TestDefaultFsControlData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement fs = new XmlElement("FsAccessControlData");
                defaultElem.AddChild(fs);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement fs = new XmlElement("FsAccessControlData");
                defaultElem.AddChild(fs);
                XmlElement fs2 = new XmlElement("FsAccessControlData");
                defaultElem.AddChild(fs2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "FsAccessControlData"),  desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultSrvControlData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlData");
                defaultElem.AddChild(srv);

                desc.AddLine(elem.GetXml());
                desc.WriteData();

                FileManager outputDesc = new FileManager();
                CreateDescFile(outputDesc, desc, null, null);
                SuccessExecute(outputDesc, metaFile, null);
            }

            // エントリを登録できる
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("Test", "False"));


            // エントリを複数個登録できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("Test", "False"),
                TestDataForSrv.MakeSrvEntry("Server", "True"),
            });

            // IsServer に True を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("True", "True"),
                TestDataForSrv.MakeSrvEntry("true", "true"),
                TestDataForSrv.MakeSrvEntry("TRUE", "TRUE"),
                TestDataForSrv.MakeSrvEntry("tRUE", "tRUE"),
            });

            // IsServer に False を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("False", "False"),
                TestDataForSrv.MakeSrvEntry("false", "false"),
                TestDataForSrv.MakeSrvEntry("FALSE", "FALSE"),
                TestDataForSrv.MakeSrvEntry("FALSe", "FALSe"),
            });

            XmlElement success = TestDataForSrv.MakeSrvEntry("Test", "True");

            // True/False 以外の文字列を受け取らない
            FailTestSrvAccessControlData(
                string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", "Failure"));

            // Entry が宣言されている場合、IsServer が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", null));

            // IsServer を重複して宣言出来ない
            {
                FileManager desc = new FileManager();

                // Descriptor 用
                XmlElement descriptor = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    descriptor.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    descriptor.AddChild(name);
                    name.InnerText = "Test";
                }

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement isServer2 = new XmlElement("IsServer");
                    data.AddChild(isServer2);
                    isServer2.InnerText = "False";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";
                }

                FailTestSrvAccessControlData(
                    string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "IsServer"),
                    descriptor,
                    data);
            }

            // サービス名を指定できる
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("Test", "True"));

            // 長さ 0 のサービス名を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry("", "True"));

            // 8文字の文字列を許容する
            SuccessTestSrvAccessControlDataWithSameSad(TestDataForSrv.MakeSrvEntry("12345678", "True"));

            // 9 文字以上の長さの文字列を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_InvalidStringLength, "Entry/Name", 9),
                success,
                TestDataForSrv.MakeSrvEntry("123456789", "True"));

            // Entry が指定されている場合、 Name が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry(null, "True"));

            // Name を重複して宣言出来ない
            {
                FileManager desc = new FileManager();

                // Descriptor 用
                XmlElement descriptor = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    descriptor.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    descriptor.AddChild(name);
                    name.InnerText = "Test";
                }

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";

                    XmlElement name2 = new XmlElement("Name");
                    data.AddChild(name2);
                    name.InnerText = "Name";
                }

                FailTestSrvAccessControlData(
                    string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "Name"),
                    descriptor,
                    data);
            }
        }

        [TestMethod]
        public void DescriptorTestForSrvControlData()
        {
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Clie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Cl*ie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Cl*ient", "False") }
                );

            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Client", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "True") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Ser*", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
        }

        [TestMethod, TestCategory("LongTest")]
        public void TestDefaultKernelCapabilityData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement kc = new XmlElement("KernelCapabilityData");
                defaultElem.AddChild(kc);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // ThreadInfo を読み込める
            {
                XmlElement success = TestDataForKc.MakeThreadInfoElement("63", "0", "0", "255");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("0", "0", "0", "0"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("63", "63", "255", "255"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("-1", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("64", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement(null, "0", "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "-1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "64", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", null, "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "-1", "0"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "256", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", null, "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "-1"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "256"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", null));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "1", "0"));
            }

            // EnableSystemCalls を読み込める
            {
                XmlElement[] success = TestDataForKc.MakeEnableSystemCalls(new string[] { "CreateThread", "SleepThread" });
                SuccessTestKernelCapabilityData(success, new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") });
                SuccessTestKernelCapabilityData(success, new XmlElement[] {
                    TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    TestDataForKc.MakeEnableSystemCall("SleepThread"),
                });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", "1"),
                    success,
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCallClass("CreateThread", "1"),
                        TestDataForKc.MakeEnableSystemCallClass("CreateThread", "1"),
                    }
                );
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x30000", "0x1000", "RW", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x40000", "0x1000", "Rw", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x50000", "0x3000", "Ro", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x30000", "0x2000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x40000", "0x2000", "RW", "Static"),
                    TestDataForKc.MakeMemoryMap("0x50000", "0x2000", "Ro", "Static"),
                });
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0xFFFFFF000", "0x3000", "Ro", "Io"));

                XmlElement[] success = new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Io"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"),
                };
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("10000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x1000000000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap(null, "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x0", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x1", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    null,
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x100000000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", null, "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Permission", "RO", "RW"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "RX", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Permission"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", null, "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Type", "Io", "Static"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Normal") });
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "MemoryMap/Type"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", null) });
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeEnableInterrupt("10"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeEnableInterrupt("1022"));
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("23"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("45"),
                });

                XmlElement success = TestDataForKc.MakeEnableInterrupt("1023");

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0x10"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_DuplicateDefinition, "EnableInterrupts", "45"),
                    new XmlElement[] { success },
                    new XmlElement[] {
                        TestDataForKc.MakeEnableInterrupt("45"),
                        TestDataForKc.MakeEnableInterrupt("45"),
                    }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_Invalid0, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("-1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "EnableInterrupts", 1023),
                    success,
                    TestDataForKc.MakeEnableInterrupt("1023"));
            }

            // MiscParams を読み込める
            {
                XmlElement success = TestDataForKc.MakeMiscParams("1");
                SuccessTestKernelCapabilityDataWithSameKc(success);
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("7"));

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", 8),
                    success,
                    TestDataForKc.MakeMiscParams("8"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeMiscParams("-1"));
            }

            // HandleTableSize を読み込める
            {
                XmlElement success = TestDataForKc.MakeHandleTableSize("1023");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("256"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeHandleTableSize("1023"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidValue_Bigger, "KernelCapabilityData/HandleTableSize", 1023),
                    success,
                    TestDataForKc.MakeHandleTableSize("1024"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("-1"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("0x1"));
            }

            // MiscFlags を読み込める
            {
                XmlElement success = TestDataForKc.MakeMiscFlags("True", "True");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("True", "True"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("True", null));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags(null, "True"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags("FALSE", null));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeMiscFlags(null, "FALSE"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/EnableDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("Success", "FALSE"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/ForceDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("True", "Fail"));
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void DescriptorTestForKernelCapability()
        {
            // ThreadInfo
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeThreadInfoElement("63", "0", "0", "4"), TestDataForKc.MakeThreadInfoElement("59", "22", "1", "3"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement("1", "0", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("2", "1", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "1", "2"));
            }

            // EnableSystemCalls
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread"), TestDataForKc.MakeEnableSystemCall("SleepThread") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") }
                    );

                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", "3"),
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCallClass("CreateThread", "1"), TestDataForKc.MakeEnableSystemCallClass("SleepThread", "2") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCallClass("SetHeapSize", "3") }
                    );
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Ro", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Ro", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x2000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x4000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x4000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x1000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Rw", "Io"));
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8") }
                );
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("1023") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapabilityWith, "EnableInterrupts",  8),
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );

            }

            // MiscParams を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("0"), TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("1"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("7"), TestDataForKc.MakeMiscParams("7"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscParams/ProgramType"),
                    TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("0"));
            }

            // HandleTableSize を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("10"));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "KernelCapabilityData/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("512"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("True", "True"), TestDataForKc.MakeMiscFlags("True", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "True"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags(null, null));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscFlags/EnableDebug"),
                    TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("True", null));
                FailTestKernelCapabilityData(
                    string.Format(MakeMeta.Properties.Resources.Message_OutOfCapability, "MiscFlags/ForceDebug"),
                    TestDataForKc.MakeMiscFlags("True", "False"), TestDataForKc.MakeMiscFlags("True", "True"));
            }
        }

        [TestMethod]
        public void TestDefaultIs64BitInstruction()
        {
            const string Is64BitInstructionName = "Is64BitInstruction";
            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "TRUE";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, Is64BitInstructionName), desc, metaFile, null, null);
            }

            // True という文字列が受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "TRUE";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // False という文字列が受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "false";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // True/False 以外の文字列を受け取らない ( True/False の小文字、大文字の違いは気にしない )
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "falure";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Default/" + Is64BitInstructionName), desc, metaFile, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(Is64BitInstructionName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_NotFound, "Default/" + Is64BitInstructionName), desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultProcessAddressSpace()
        {
            const string ProcessAddressSpaceName = "ProcessAddressSpace";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace64Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, "ProcessAddressSpace"), desc, metaFile, null, null);
            }

            // AddressSpace32Bit を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace32Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // AddressSpace64Bit を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace64Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // AddressSpace32BitNoReserved を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace32BitNoReserved";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 異常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString, "Default/" + ProcessAddressSpaceName, item.InnerText), desc, metaFile, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Default/" + ProcessAddressSpaceName), desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadPriority()
        {
            const string MainThreadPriorityName = "MainThreadPriority";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadPriorityName), desc, metaFile, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "0x18";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadPriorityName), desc, metaFile, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "63";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最小値 - 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "-1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadPriorityName), desc, metaFile, null, null);
            }

            // 最大値 + 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "64";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_OutOfRange, "Default/" + MainThreadPriorityName, "64"), desc, metaFile, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadPriorityName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Default/" + MainThreadPriorityName), desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadCoreNumber()
        {
            const string MainThreadCoreNumberName = "MainThreadCoreNumber";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadCoreNumberName), desc, metaFile, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadCoreNumberName), desc, metaFile, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "255";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最小値 - 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "-1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadCoreNumberName), desc, metaFile, null, null);
            }

            // 最大値 + 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "256";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, metaFile, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Default/" + MainThreadCoreNumberName), desc, metaFile, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadStackSize()
        {
            const string MainThreadStackSizeName = "MainThreadStackSize";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x4000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeMeta.Properties.Resources.Message_UnknownElement, MainThreadStackSizeName), desc, metaFile, null, null);
            }

            // 10 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "4096";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Default/" + MainThreadStackSizeName), desc, metaFile, null, null);
            }

            // 16 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0xFFFFF000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, metaFile, null, null);
            }

            // 最小値 - 0x1000
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_Invalid0, "Default/" + MainThreadStackSizeName), desc, metaFile, null, null);
            }

            // 最大値 + 0x1000
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x100000000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, metaFile, null, null);
            }

            // 0x1000 にアライメントされていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1001";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidAlignment, "Default/" + MainThreadStackSizeName, 0x1000), desc, metaFile, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultDescParam();
                elem.AddChild(defaultElem);
                elem.AddChild(acidElement);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_NotFound, "Default/" + MainThreadStackSizeName), desc, metaFile, null, null);
            }
        }

        /* KernelCapability 用のユーティリティ */

        void MakeDescForKernelCapability(ref FileManager outputDesc, XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            // Desc
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            if (descs != null)
            {
                XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
                foreach (var elem in descs)
                {
                    kcd.AddChild(elem);
                }

                descElem.AddChild(kcd);
            }

            if (metas != null)
            {
                XmlElement kc = new XmlElement("KernelCapabilityData");
                foreach (var elem in metas)
                {
                    kc.AddChild(elem);
                }

                defaultElem.AddChild(kc);
            }

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        void MakeMetaForKernelCapability(ref XmlFile outputMeta, XmlElement[] metas)
        {
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement programId = new XmlElement("ProgramId");
            programId.InnerText = "0x0005000C10000000";
            core.AddChild(programId);

            if (metas != null)
            {
                XmlElement kc = new XmlElement("KernelCapabilityData");
                foreach (var m in metas)
                {
                    kc.AddChild(m);
                }

                core.AddChild(kc);
            }

            outputMeta.AddLine(elem.GetXml());
            outputMeta.WriteData();
        }

        void SuccessTestKernelCapabilityData(XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            FileManager outputDesc = new FileManager();
            MakeDescForKernelCapability(ref outputDesc, descs, metas, kernelVersion);

            SuccessExecute(outputDesc, metaFile, kernelVersion);
        }

        void FailTestKernelCapabilityData(string errMsg, XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    kcd.AddChild(elem);
                }
            }
            descElem.AddChild(kcd);

            XmlElement kc= new XmlElement("KernelCapabilityData");
            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    kc.AddChild(elem);
                }
            }
            defaultElem.AddChild(kc);

            descElem.AddChild(acidElement);

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            FailExecute(errMsg, descFile, metaFile, kernelVersion, null);
        }

        // ラッパー
        void SuccessTestKernelCapabilityDescriptor(XmlElement elem)
        {
            SuccessTestKernelCapabilityDescriptor(new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDescriptor(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, null);
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement elem)
        {
            FailTestKernelCapabilityDescriptor(errMsg, new XmlElement[] { elem });
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement[] elems)
        {
            FailTestKernelCapabilityData(errMsg, elems, null);
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement elem)
        {
            SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, elems);
        }
        void SuccessTestKernelCapabilityData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestKernelCapabilityData(string errMsg, XmlElement desc, XmlElement meta, string kernelVersion = null)
        {
            FailTestKernelCapabilityData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta }, kernelVersion);
        }

        /* SrviceAccessControl 用のユーティリティ */

        void MakeDescFileForSrvAccessControlData(ref FileManager outputDesc, XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            XmlElement sad = new XmlElement("SrvAccessControlDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    sad.AddChild(elem);
                }
            }
            descElem.AddChild(sad);

            XmlElement sac = new XmlElement("SrvAccessControlData");
            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    sac.AddChild(elem);
                }
            }
            defaultElem.AddChild(sac);


            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        void MakeMetaFileForSrvAccessControlData(ref XmlFile outputMeta, XmlElement[] metas)
        {
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            elem.AddChild(core);
            XmlElement programId = new XmlElement("ProgramId");
            programId.InnerText = "0x0005000C10000000";
            core.AddChild(programId);

            if (metas != null)
            {
                XmlElement srv = new XmlElement("SrvAccessControlData");
                foreach (var m in metas)
                {
                    srv.AddChild(m);
                }

                core.AddChild(srv);
            }

            outputMeta.AddLine(elem.GetXml());
            outputMeta.WriteData();
        }

        void SuccessTestSrvAccessControlData(XmlElement[] descs, XmlElement[] metas)
        {
            FileManager desc = new FileManager();
            MakeDescFileForSrvAccessControlData(ref desc, descs, metas);

            SuccessExecute(desc, metaFile, null, null);
        }

        void FailTestSrvAccessControlData(string errMsg, XmlElement[] descs, XmlElement[] metas)
        {
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });

            XmlElement sad = new XmlElement("SrvAccessControlDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    sad.AddChild(elem);
                }
            }
            descElem.AddChild(sad);

            XmlElement sac = new XmlElement("SrvAccessControlData");
            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    sac.AddChild(elem);
                }
            }
            defaultElem.AddChild(sac);

            descElem.AddChild(acidElement);

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            FailExecute(errMsg, descFile, metaFile, null, null);
        }

        // ラッパー
        void SuccessTestSrvAccessControlData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestSrvAccessControlData(string errMsg, XmlElement desc, XmlElement meta)
        {
            FailTestSrvAccessControlData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void SuccessTestSrvAccessControlDataWithSameSad(XmlElement elem)
        {
            SuccessTestSrvAccessControlData(new XmlElement[] { elem }, new XmlElement[] { elem });
        }
        void SuccessTestSrvAccessControlDataWithSameSad(XmlElement[] elems)
        {
            SuccessTestSrvAccessControlData(elems, elems);
        }
        void FailTestSrvAccessControlDataWithSameSad(string errMsg, XmlElement elem)
        {
            FailTestSrvAccessControlData(errMsg, new XmlElement[] { elem }, new XmlElement[] { elem });
        }
        void FailTestSrvAccessControlDataWithSameSad(string errMsg, XmlElement[] elems)
        {
            FailTestSrvAccessControlData(errMsg, elems, elems);
        }
    }
}
