﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeMetaTest
{
    public class TestBase
    {
        public TestContext TestContext { get; set; }

        public const string DefaultIs64BitInstruction = "false";
        public const string DefaultProcessAddressSpace = "AddressSpace32Bit";
        public const string DefaultMainThreadPriority = "25";
        public const string DefaultMainThreadCoreNumber = "1";
        public const string DefaultMainThreadStackSize = "0x4000";

        public const string BaseIdHeader = "svc_BaseId.autogen.h";
        public const string ServerIdHeader = "svc_ServerId.autogen.h";
        public const string DdIdHeader = "svc_DdId.autogen.h";
        public const string DmntIdHeader = "svc_DmntId.autogen.h";
        public const string TcbIdHeader = "svc_TcbId.autogen.h";

        public const string DefaultProgramId = "0x0005000C10000000";

        public const int AciHeaderSize = 0x40;

        // META
        public const int MetaSignatureOffset = 0x0;
        public const int MetaReserved1Offset = 0x4;
        public const int MetaReserved1Size = 0x8;
        public const int MetaFlagsOffset = 0xC;
        public const int MetaReserved2Offset = 0x10;
        public const int MetaReserved2Size = 0x8;
        public const int MetaVersionOffset = 0x18;
        public const int MetaStackSizeOffset = 0x1C;
        public const int MetaNameOffset = 0x20;
        public const int MetaNameSize = 0x10;
        public const int MetaProductCodeOffset = 0x30;
        public const int MetaProductCodeSize = 0x10;
        public const int MetaReserved3Offset = 0x40;
        public const int MetaReserved3Size = 0x30;
        public const int MetaAciOffsetOffset = 0x70;
        public const int MetaAciSizeOffset = 0x74;
        public const int MetaAcidOffsetOffset = 0x78;
        public const int MetaAcidSizeOffset = 0x7C;

        // ACI
        public const int AciSignatureOffset = 0x0;
        public const int AciReserved1Offset = 0x4;
        public const int AciReserved1Size   = 0xc;
        public const int AciProgramIdOffset = 0x10;
        public const int AciReserved2Offset = 0x18;
        public const int AciReserved2Size   = 0x8;
        public const int AciFacOffsetOffset = 0x20;
        public const int AciFacSizeOffset = 0x24;
        public const int AciSacOffsetOffset = 0x28;
        public const int AciSacSizeOffset = 0x2C;
        public const int AciKcOffsetOffset = 0x30;
        public const int AciKcSizeOffset = 0x34;
        public const int AciReserved3Offset = 0x38;
        public const int AciReserved3Size   = 0x8;

        public const int UInt32Size = 0x4;
        public const int UInt64Size = 0x8;
        public const int KernelCapabilityEntrySize = 0x4;

        public const string AciSignature = "ACI0";
        public const string MetaSignature = "META";

        // NPDM フォーマット
        public const int MetaOffset = 0x0;
        public const int MetaHeaderSize = 0x80;
        public const uint BinaryAlignSize = 0x10;
        public const int MetaHeaderOffset = 0x0;
        public const int AcidHeaderSize = 0x240;

        public string ExecuteProgram(FileManager desc, FileManager meta, FileManager outputFile, string kernelVersion, string additionalParam = "")
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            string kernelVersionParam = string.Empty;
            if (kernelVersion != null && kernelVersion.Length > 0)
            {
                kernelVersionParam = "--kernel_version " + kernelVersion;
            }

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeMeta\\MakeMeta.exe";
            process.StartInfo.Arguments = string.Format(
                "-o {0} --desc {1} --meta {2} {3} {4} --no_check_programid",
                outputFile.FilePath, desc.FilePath, meta.FilePath, kernelVersionParam, additionalParam);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();

            Assert.IsTrue(System.IO.File.Exists(desc.FilePath));
            Assert.IsTrue(System.IO.File.Exists(meta.FilePath));
            return errorMsg;
        }

        public FileManager SuccessExecute(FileManager descFile, FileManager metaFile, string kernelVersion, string additionalParam = "")
        {
            FileManager outputDesc = new FileManager();
            Assert.IsTrue(System.IO.File.Exists(descFile.FilePath));
            Assert.IsTrue(System.IO.File.Exists(metaFile.FilePath));
            string err = ExecuteProgram(descFile, metaFile, outputDesc, kernelVersion, additionalParam);
            Assert.IsTrue(err.Length == 0, err);
            Assert.IsTrue(System.IO.File.Exists(outputDesc.FilePath));
            return outputDesc;
        }

        public void FailExecute(string errMsg, FileManager descFile, FileManager metaFile, string kernelVersion, string additionalParam = "")
        {
            FileManager tmpFile = new FileManager();
            Assert.IsTrue(System.IO.File.Exists(descFile.FilePath));
            Assert.IsTrue(System.IO.File.Exists(metaFile.FilePath));
            string err = ExecuteProgram(descFile, metaFile, tmpFile, kernelVersion, additionalParam);
            Assert.IsTrue(err.Length > 0);
            // デフォルトの例外のメッセージはテストしない
            if (errMsg != null)
            {
                Assert.IsTrue(err.IndexOf(errMsg) >= 0);
            }
            Assert.IsFalse(System.IO.File.Exists(tmpFile.FilePath));
        }

        public XmlElement CreateDefaultDescParam()
        {
            XmlElement defaultElem = new XmlElement("Default");

            XmlElement is64Bit = new XmlElement("Is64BitInstruction");
            is64Bit.InnerText = DefaultIs64BitInstruction;
            defaultElem.AddChild(is64Bit);

            XmlElement processAddressSpace = new XmlElement("ProcessAddressSpace");
            processAddressSpace.InnerText = DefaultProcessAddressSpace;
            defaultElem.AddChild(processAddressSpace);

            XmlElement mainThreadPriority = new XmlElement("MainThreadPriority");
            mainThreadPriority.InnerText = DefaultMainThreadPriority;
            defaultElem.AddChild(mainThreadPriority);

            XmlElement mainThreadCoreNumber = new XmlElement("MainThreadCoreNumber");
            mainThreadCoreNumber.InnerText = DefaultMainThreadCoreNumber;
            defaultElem.AddChild(mainThreadCoreNumber);

            XmlElement mainThreadStackSize = new XmlElement("MainThreadStackSize");
            mainThreadStackSize.InnerText = DefaultMainThreadStackSize;
            defaultElem.AddChild(mainThreadStackSize);

            return defaultElem;
        }

        public XmlElement CreateDefaultMetaParam()
        {
            XmlElement defaultElem = new XmlElement("ProgramId");
            defaultElem.InnerText = DefaultProgramId;

            return defaultElem;
        }

        public void CreateDescFile(FileManager outputFile, FileManager inputFile, string[] svcHeaders, string kernelVersion, string additionalParam = "")
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);

            string svcHeaderParam = string.Empty;
            if (svcHeaders != null)
            {
                foreach (var headerName in svcHeaders)
                {
                    svcHeaderParam = svcHeaderParam + " --svc_header " + headerName;
                }
            }

            string kernelVersionParam = string.Empty;
            if (kernelVersion != null && kernelVersion.Length > 0)
            {
                kernelVersionParam = "--kernel_version " + kernelVersion;
            }

            process.StartInfo.FileName = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeDesc\\MakeDesc.exe";
            process.StartInfo.Arguments = string.Format(
                "-o {0} --desc {1} {2} {3} {4}",
                outputFile.FilePath, inputFile.FilePath, svcHeaderParam, kernelVersionParam, additionalParam);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();
            Assert.IsTrue(errorMsg.Length == 0);
            Assert.IsTrue(System.IO.File.Exists(outputFile.FilePath));
            Assert.IsTrue(System.IO.File.Exists(inputFile.FilePath));
        }

        public XmlFile CreateDefaultDescFile()
        {
            XmlFile inputDesc = new XmlFile();
            XmlFile descFile = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            elem.AddChild(defaultElem);

            inputDesc.AddLine(elem.GetXml());
            inputDesc.WriteData();

            CreateDescFile(descFile, inputDesc, null, null);
            return descFile;
        }

        public XmlFile CreateDefaultDescFileWithAdditionalTags(XmlElement targetElem)
        {
            XmlFile inputDesc = new XmlFile();
            XmlFile descFile = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            elem.AddChild(defaultElem);

            if (targetElem != null)
            {
                elem.AddChild(targetElem);
            }

            inputDesc.AddLine(elem.GetXml());
            inputDesc.WriteData();

            CreateDescFile(descFile, inputDesc, null, null);
            return descFile;
        }

        public XmlFile CreateDescFileWithAdditionalDefaultTags(XmlElement targetElem)
        {
            XmlFile inputDesc = new XmlFile();
            XmlFile descFile = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            elem.AddChild(defaultElem);

            if (targetElem != null)
            {
                defaultElem.AddChild(targetElem);
            }

            inputDesc.AddLine(elem.GetXml());
            inputDesc.WriteData();

            CreateDescFile(descFile, inputDesc, null, null);
            return descFile;
        }

        public uint RoundUp(uint target, uint align)
        {
            return (target + align - 1) & ~(align - 1);
        }

        public string GetSvcHeaderDirPath()
        {
            TestUtility.TestPath testPath = new TestUtility.TestPath(this.TestContext);
            return testPath.GetSigloRoot() + "\\Programs\\Chris\\Include\\nn\\svc\\";
        }

        public enum CapabilityType
        {
            Kernel,
            Service,
            Fs,
            None,
        };

        public XmlElement MakeDescriptorElem(XmlElement[] descs, CapabilityType type)
        {
            if (descs == null)
            {
                return null;
            }

            string descriptorName = null;
            switch (type)
            {
                case CapabilityType.Kernel:
                    descriptorName = "KernelCapabilityDescriptor";
                    break;
                case CapabilityType.Service:
                    descriptorName = "SrvAccessControlDescriptor";
                    break;
                case CapabilityType.Fs:
                    descriptorName = "FsAccessControlDescriptor";
                    break;
                default: throw new ArgumentException();
            }

            XmlElement descElem = new XmlElement(descriptorName);
            foreach (var desc in descs)
            {
                descElem.AddChild(desc);
            }

            return descElem;
        }

        public XmlElement MakeDataElem(XmlElement[] metas, CapabilityType type)
        {
            if (metas == null)
            {
                return null;
            }

            string dataName = null;
            switch (type)
            {
                case CapabilityType.Kernel:
                    dataName = "KernelCapabilityData";
                    break;
                case CapabilityType.Service:
                    dataName = "SrvAccessControlData";
                    break;
                case CapabilityType.Fs:
                    dataName = "FsAccessControlData";
                    break;
                default: throw new ArgumentException();
            }

            XmlElement metaElem = new XmlElement(dataName);
            foreach (var desc in metas)
            {
                metaElem.AddChild(desc);
            }

            return metaElem;
        }

        public XmlElement[] MakeDataElemWithMulpleElements(Dictionary<CapabilityType, XmlElement[]> metas)
        {
            if (metas == null)
            {
                return null;
            }

            XmlElement[] elems = new XmlElement[metas.Count];

            int i = 0;
            foreach (var elem in metas)
            {
                elems[i++] = MakeDataElem(elem.Value, elem.Key);
            }

            return elems;
        }

        public void MakeDesc(ref FileManager outputDesc, string kernelVersion = null)
        {
            MakeDesc(ref outputDesc, new XmlElement[0], new XmlElement[0], kernelVersion);
        }

        public void MakeDesc(ref FileManager outputDesc, XmlElement desc, XmlElement meta, string kernelVersion = null)
        {
            XmlElement[] descs = (desc != null) ? new XmlElement[] { desc } : null;
            XmlElement[] metas = (meta != null) ? new XmlElement[] { meta } : null;
            MakeDesc(ref outputDesc, descs, metas, kernelVersion);
        }

        public void MakeDesc(ref FileManager outputDesc, XmlElement[] descs, XmlElement[] metas, string kernelVersion = null)
        {
            if (descs != null && descs.Length == 0)
            {
                descs = null;
            }
            if (metas != null && metas.Length == 0)
            {
                metas = null;
            }

            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });


            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    descElem.AddChild(elem);
                }
            }

            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    defaultElem.AddChild(elem);
                }
            }

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        public void MakeDescWithMultipleElements(ref FileManager outputDesc, Dictionary<CapabilityType, XmlElement[]> descs, Dictionary<CapabilityType, XmlElement[]> metas, string kernelVersion = null)
        {
            XmlFile descFile = new XmlFile();
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultDescParam();
            descElem.AddChild(defaultElem);
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });


            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    descElem.AddChild(MakeDescriptorElem(elem.Value, elem.Key));
                }
            }

            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    defaultElem.AddChild(MakeDataElem(elem.Value, elem.Key));
                }
            }

            descFile.AddLine(descElem.GetXml());
            descFile.WriteData();

            CreateDescFile(outputDesc, descFile, new string[] { baseId }, kernelVersion);
        }

        public byte[] MakeKcEntryAci(XmlElement[] elems)
        {
            return CreateDefaultAciFile(elems, CapabilityType.Kernel);
        }

        public byte[] MakeKcEntryAci(XmlElement elem)
        {
            return MakeKcEntryAci(new XmlElement[] { elem });
        }

        public byte[] CreateDefaultAciFile(CapabilityType type, string kernelVersion = null)
        {
            return CreateDefaultAciFile(new XmlElement[0], type, kernelVersion);
        }

        public byte[] CreateDefaultAciFile(XmlElement targetElement, CapabilityType type, string kernelVersion = null)
        {
            return CreateDefaultAciFile(new XmlElement[] { targetElement }, type, kernelVersion);
        }

        public byte[] CreateDefaultAciFile(XmlElement[] targetElements, CapabilityType type, string kernelVersion = null)
        {
            if (targetElements != null && targetElements.Length == 0)
            {
                targetElements = null;
            }

            FileManager descFile = new FileManager();
            MakeDesc(ref descFile, MakeDescriptorElem(targetElements, type), null, kernelVersion);

            XmlFile meta = new XmlFile();
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            XmlElement defaultElem = CreateDefaultMetaParam();
            elem.AddChild(core);
            core.AddChild(defaultElem);

            XmlElement dataElem = MakeDataElem(targetElements, type);
            if (dataElem != null)
            {
                core.AddChild(dataElem);
            }

            meta.AddLine(elem.GetXml());
            meta.WriteData();
            FileManager outputFile = SuccessExecute(descFile, meta, kernelVersion, null);

            byte[] metaBinary = outputFile.ReadFile(MetaHeaderSize, MetaOffset);
            uint aciOffset = BitConverter.ToUInt32(metaBinary, MetaAciOffsetOffset);
            uint aciSize = BitConverter.ToUInt32(metaBinary, MetaAciSizeOffset);
            return outputFile.ReadFile((int)aciSize, aciOffset);
        }

        public byte[] CreateDefaultAciFileWithMultipleElements(Dictionary<CapabilityType, XmlElement[]> targetElements, string kernelVersion = null)
        {
            FileManager descFile = new FileManager();
            MakeDescWithMultipleElements(ref descFile, targetElements, null, kernelVersion);

            XmlFile meta = new XmlFile();
            XmlElement elem = new XmlElement("Meta");
            XmlElement core = new XmlElement("Core");
            XmlElement defaultElem = CreateDefaultMetaParam();
            elem.AddChild(core);
            core.AddChild(defaultElem);

            XmlElement[] dataElems = MakeDataElemWithMulpleElements(targetElements);
            if (dataElems != null)
            {
                foreach (var data in dataElems)
                {
                    core.AddChild(data);
                }
            }

            meta.AddLine(elem.GetXml());
            meta.WriteData();
            FileManager outputFile = SuccessExecute(descFile, meta, kernelVersion, null);

            byte[] metaBinary = outputFile.ReadFile(MetaHeaderSize, MetaOffset);
            uint aciOffset = BitConverter.ToUInt32(metaBinary, MetaAciOffsetOffset);
            uint aciSize = BitConverter.ToUInt32(metaBinary, MetaAciSizeOffset);
            return outputFile.ReadFile((int)aciSize, aciOffset);
        }


    }
}
