﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeDescTest
{
    [TestClass]
    public class TestOutputXml : TestBase
    {
        private const string AcidTagName = "<Acid>";
        private const string KernelCapabilityDescriptorBegin = "<KernelCapabilityDescriptor>";
        private const string KernelCapabilityDescriptorEnd = "</KernelCapabilityDescriptor>";
        private const string KernelCapabilityDataBegin = "<KernelCapabilityData>";
        private const string KernelCapabilityDataEnd = "</KernelCapabilityData>";
        private const string KernelVersionTagName = "<KernelVersion>";

        public List<string> ReadFile(FileManager readFile)
        {
            List<string> list = new List<string>();
            using (FileStream fs = new FileStream(readFile.FilePath, FileMode.Open, FileAccess.Read))
            {
                StreamReader sr = new StreamReader(fs);
                string line = null;
                while ((line = sr.ReadLine()) != null)
                {
                    string text = line.Trim();

                    // Bool は input と output で値が異なることがある
                    if (text.IndexOf("true", StringComparison.OrdinalIgnoreCase) >= 0
                        || text.IndexOf("false", StringComparison.OrdinalIgnoreCase) >= 0)
                    {
                        XmlElement elem = XmlElement.Convert(text);
                        elem.InnerText = elem.InnerText.ToLower();
                        list.Add(elem.GetXml());
                    }
                    else
                    {
                        list.Add(text);
                    }
                }
            }
            return list;
        }

        [TestMethod]
        public void TestSameDataWithInput()
        {
            XmlFile desc = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            elem.AddChild(defaultElem);

            desc.AddLine(elem.GetXml());
            desc.WriteData();
            FileManager result = SuccessExecute(desc, null, null);

            List<string> inputList = ReadFile(desc);
            List<string> outputList = ReadFile(result);

            foreach (var line in inputList)
            {
                Assert.IsTrue(outputList.Contains(line), string.Format("'{0}' not found", line));
            }
        }

        [TestMethod]
        public void TestFindAcid()
        {
            XmlFile desc = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            elem.AddChild(defaultElem);

            desc.AddLine(elem.GetXml());
            desc.WriteData();
            FileManager result = SuccessExecute(desc, null, null);

            List<string> outputList = ReadFile(result);

            bool check = false;
            foreach (var line in outputList)
            {
                if (line.IndexOf(AcidTagName) >= 0)
                {
                    check = true;
                    break;
                }
            }

            Assert.IsTrue(check);
        }

        [TestMethod]
        public void TestUseKernelVersion()
        {
            XmlFile desc = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            elem.AddChild(defaultElem);

            desc.AddLine(elem.GetXml());
            desc.WriteData();
            FileManager result = SuccessExecute(desc, null, "1.3");

            List<string> outputList = ReadFile(result);

            bool descriptor = false;
            bool data = false;
            bool descriptorCheck = false;
            bool dataCheck = false;
            foreach (var line in outputList)
            {
                if (line == KernelCapabilityDescriptorBegin)
                {
                    descriptor = true;
                }
                else if (descriptor && line == KernelVersionTagName)
                {
                    descriptorCheck = true;
                }
                else if (line == KernelCapabilityDescriptorEnd)
                {
                    descriptor = false;
                }
                else if (line == KernelCapabilityDataBegin)
                {
                    data = true;
                }
                else if (data && line == KernelVersionTagName)
                {
                    dataCheck = true;
                }
                else if (line == KernelCapabilityDataEnd)
                {
                    data = false;
                }
            }
            Assert.IsTrue(descriptorCheck && dataCheck);
        }

        [TestMethod]
        public void TestNotUseKernelVersion()
        {
            XmlFile desc = new XmlFile();
            XmlElement elem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            elem.AddChild(defaultElem);

            desc.AddLine(elem.GetXml());
            desc.WriteData();
            FileManager result = SuccessExecute(desc, null, null);

            List<string> outputList = ReadFile(result);

            Assert.IsTrue(outputList.Where(line => line == KernelVersionTagName).Count() == 0);
        }
    }
}
