﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using System.Threading.Tasks;

using Nintendo.HtcTools.Htclow;

namespace Nintendo.HtcTools.Htclow
{
    [TestClass]
    public class DurabilityTest
    {
        private const int ModuleId = 0;
        private const int ChannelId = 0;

        private Channel m_Channel = new Channel() { ModuleId = ModuleId, ChannelId = ChannelId };

        #region Check Memory Leak

        /// <summary>
        /// Recieve を繰り返してメモリリークが発生しないかテストする
        /// メモリ使用量は、Visual Studio の出力を目視確認する
        /// </summary>
        [TestMethod]
        public void ReceiveManyTimes()
        {
            // メモリリーク検査を実際に行う際は、LoopCount を大きく増やしてデバッグ実行する
            const int LoopCount = 10;

            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService(true))
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                var util = new HandshakeUtil(factory, service);
                util.DoHandshakeByHost(htclowChannel, m_Channel);

                for (int i = 0; i < LoopCount; i++)
                {
                    long sequenceId = i + 1;

                    var data = DataUtil.MakeRandom(Packet.MaxBodySize);
                    var sendPacket = factory.MakeDataPacket(m_Channel, data, sequenceId);

                    // データ送信
                    service.SendPacket(sendPacket);

                    // データ受信
                    var receivedData = htclowChannel.Receive(Packet.MaxBodySize);
                    CollectionAssert.AreEqual(data, receivedData);

                    // ack 受信
                    var expect = factory.MakeDataAckPacket(m_Channel, sequenceId);
                    var recvPacket = service.ReceivePacket();

                    Assert.AreEqual(expect, recvPacket);
                }
            }
        }

        /// <summary>
        /// Send を繰り返してメモリリークが発生しないかテストする
        /// メモリ使用量は、Visual Studio の出力を目視確認する
        /// </summary>
        [TestMethod]
        public void SendManyTimes()
        {
            // メモリリーク検査を実際に行う際は、LoopCount を大きく増やしてデバッグ実行する
            const int LoopCount = 10;

            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService(true))
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                var util = new HandshakeUtil(factory, service);
                util.DoHandshakeByHost(htclowChannel, m_Channel);

                for (int i = 0; i < LoopCount; i++)
                {
                    var data = DataUtil.MakeRandom(Packet.MaxBodySize);

                    // データ送信
                    htclowChannel.Send(data);

                    // データ受信
                    var dataPacket = service.ReceivePacket();
                    var expectDataPacket = factory.MakeDataPacket(m_Channel, data, dataPacket.SequenceId);

                    Assert.AreEqual(expectDataPacket, dataPacket);

                    // ack 送信
                    var ackPacket = factory.MakeDataAckPacket(m_Channel, dataPacket.SequenceId);
                    service.SendPacket(ackPacket);
                }
            }
        }

        #endregion

    }
}
