﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.HtcTools.HtcfsLibrary;

namespace Nintendo.HtcTools.HtcfsLibraryTest
{
    /// <summary>
    /// testFs_Directory に相当するテストを HtcfsLibrary 単体で行う
    /// </summary>
    [TestClass]
    public class HtcfsDirectoryTest
    {
        private void CleanDirectoryRecursively(HtcfsLibraryTestService service, string path)
        {
            CompoundResult result;

            // open
            result = service.OpenDirectory(out var handle, path, OpenDirectoryMode.All);
            Assert.IsTrue(result.IsSuccess());

            // ディレクトリのエントリをすべて削除
            result = service.GetEntryCount(out var count, handle);
            Assert.IsTrue(result.IsSuccess());

            result = service.ReadDirectory(out var entries, handle, count);
            Assert.IsTrue(result.IsSuccess());

            foreach (var entry in entries)
            {
                var entryPath = $"{path}/{entry.Name}";
                switch (entry.EntryType)
                {
                    case DirectoryEntryType.Directory:
                        result = service.DeleteDirectory(entryPath, true);
                        Assert.IsTrue(result.IsSuccess());
                        break;
                    case DirectoryEntryType.File:
                        result = service.DeleteFile(entryPath);
                        Assert.IsTrue(result.IsSuccess());
                        break;
                    default:
                        throw new Exception("Unknown entry type.");
                }
            }

            // close
            result = service.CloseDirectory(handle);
            Assert.IsTrue(result.IsSuccess());
        }

        // CleanDirectoryRecursively() が正しく動作していること
        [TestMethod]
        public void Clean()
        {
            var randomSeed = 0;

            var testService = new HtcfsLibraryTestService();
            var random = new Random(randomSeed);

            for (int count = 0; count < 5; count++)
            {
                var directoryPaths = new List<string>()
                {
                    testService.TestRootPath
                };

                // テストルートディレクトリ内に無作為な構成でエントリを作成する
                for (int i = 0; i < 10; i++)
                {
                    var parentIndex = random.Next(directoryPaths.Count);
                    string entryPath = $"{directoryPaths[parentIndex]}/{count}_{i}";

                    if (random.Next(2) != 0)
                    {
                        var result = testService.CreateFile(entryPath, count);
                        Assert.IsTrue(result.IsSuccess());
                    }
                    else
                    {
                        var result = testService.CreateDirectory(entryPath);
                        Assert.IsTrue(result.IsSuccess());

                        directoryPaths.Add(entryPath);
                    }
                }

                // CleanDirectoryRecursively() によりテストルートディレクトリが空になることを確認する
                CleanDirectoryRecursively(testService, testService.TestRootPath);

                var resultOpenDirectory = testService.OpenDirectory(out var handle, testService.TestRootPath, OpenDirectoryMode.All);
                Assert.IsTrue(resultOpenDirectory.IsSuccess());

                var resultGetEntryCount = testService.GetEntryCount(out var entryCount, handle);
                Assert.IsTrue(resultGetEntryCount.IsSuccess());

                var resultCloseDirectory = testService.CloseDirectory(handle);
                Assert.IsTrue(resultCloseDirectory.IsSuccess());
            }

            testService.Dispose();
        }
    }
}
