﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Hid
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Net.Sockets;
    using System.Text;
    using System.Threading.Tasks;
    using Nintendo.Htcs;

    public class DebugBridgeClient : IDisposable
    {
        public const string HtcsPortName = "@HidDebugBridgeService";
        private const byte ProtocolId = 0x10;

        private bool disposed = false;
        private TcpClient tcpClient = new TcpClient();
        private DebugPad debugPad = new DebugPad(0);
        private BaseModule[] modules;

        public DebugBridgeClient()
        {
            var list = new List<BaseModule>();
            list.Add(this.debugPad);
            this.modules = list.ToArray();
        }

        ~DebugBridgeClient()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        public void Connect(PortMapItem portMapItem)
        {
            this.tcpClient.Connect(portMapItem.EndPoint);
        }

        public void Close()
        {
            foreach (var m in this.modules)
            {
                m.Reset();
            }

            this.tcpClient.Close();
        }

        public void Update()
        {
            byte[] autoInputCommands = this.GenerateAutoInputCommands();
            byte[] requestCommands = this.GenerateRequestCommands();
            foreach (var m in this.modules)
            {
                m.Update();
            }

            if (autoInputCommands.Length == 0 && requestCommands.Length == 0)
            {
                return;
            }

            NetworkStream networkStream = this.tcpClient.GetStream();

            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                writer.Write((ushort)(4 + autoInputCommands.Length + requestCommands.Length));
                writer.Write((byte)0);
                writer.Write((byte)DebugBridgeClient.ProtocolId);
                writer.Write((byte[])autoInputCommands);
                writer.Write((byte[])requestCommands);
                var bytes = stream.ToArray();
                networkStream.Write(bytes, 0, bytes.Length);
            }

            if (requestCommands.Length == 0)
            {
                return;
            }

            var commandsCount = this.ReadCommandsCount(networkStream);
            var commands = this.ReadCommands(networkStream, commandsCount);
            using (var stream = new MemoryStream(commands))
            using (var reader = new NetworkBinaryReader(stream))
            {
                while (reader.BaseStream.Position < reader.BaseStream.Length)
                {
                    this.AcceptCommand(reader);
                }
            }
        }

        public DebugPad GetDebugPad()
        {
            return this.debugPad;
        }

        private void Dispose(bool disposing)
        {
            if (this.disposed)
            {
                return;
            }

            if (disposing)
            {
                this.tcpClient.Close();
            }

            this.disposed = true;
        }

        private byte[] GenerateAutoInputCommands()
        {
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                foreach (var m in this.modules)
                {
                    writer.Write((byte[])m.GenerateAutoInputCommand());
                }

                return stream.ToArray();
            }
        }

        private byte[] GenerateRequestCommands()
        {
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                foreach (var m in this.modules)
                {
                    writer.Write((byte[])m.GenerateRequestCommand());
                }

                return stream.ToArray();
            }
        }

        private ushort ReadCommandsCount(NetworkStream networkStream)
        {
            var header = new byte[4];
            var headerCount = networkStream.Read(header, 0, header.Length);
            using (var stream = new MemoryStream(header, 0, headerCount))
            using (var reader = new NetworkBinaryReader(stream))
            {
                var packetCount = reader.ReadUInt16();
                reader.ReadByte();
                var protocolId = reader.ReadByte();
                if (packetCount <= header.Length || protocolId != DebugBridgeClient.ProtocolId)
                {
                    throw new ObjectDisposedException("NetworkStream");
                }

                return (ushort)(packetCount - header.Length);
            }
        }

        private byte[] ReadCommands(NetworkStream networkStream, ushort count)
        {
            var commands = new byte[count];
            var commandsCount = networkStream.Read(commands, 0, commands.Length);
            if (commandsCount != count)
            {
                throw new ObjectDisposedException("NetworkStream");
            }

            return commands;
        }

        private void AcceptCommand(NetworkBinaryReader commandsReader)
        {
            var commandCount = commandsReader.ReadUInt16();
            commandsReader.ReadByte();
            var commandType = commandsReader.ReadByte();
            if (commandCount <= 4)
            {
                throw new ObjectDisposedException("NetworkStream");
            }

            using (var stream = new MemoryStream(commandsReader.ReadBytes(commandCount - 4)))
            using (var reader = new NetworkBinaryReader(stream))
            {
                switch (commandType)
                {
                    case (byte)CommandType.RespondInputState:
                        this.AcceptInputState(reader);
                        break;
                    default:
                        throw new ObjectDisposedException("NetworkStream");
                }

                if (reader.BaseStream.Position != reader.BaseStream.Length)
                {
                    throw new ObjectDisposedException("NetworkStream");
                }
            }
        }

        private void AcceptInputState(NetworkBinaryReader reader)
        {
            var moduleType = reader.ReadByte();
            var moduleId = reader.ReadByte();
            var module = this.modules.FirstOrDefault(m => m.ModuleTypeValue == moduleType && m.ModuleId == moduleId);
            if (module == null)
            {
                throw new ObjectDisposedException("NetworkStream");
            }

            module.AcceptInputState(reader);
        }
    }
}
