﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using GitExternalStorage;
using GitExternalStorage.Git;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TestUtility;

namespace GitExternalStorageTest
{
    public abstract class UnitTestGitRepositoryBase
    {
        public abstract string GetRepositoryPath();
        public abstract GitRepositoryBase GetGitRepositoryInstance(string workingDirectory);

        private string GetRepositoryContentFilePath()
        {
            return Path.Combine(GetRepositoryPath(), "file1.txt");
        }

        private string GetRepositoryContentFileRepositoryRelativePath()
        {
            return "file1.txt";
        }

        private string GetRepositoryContentFileContent()
        {
            return "this is a test";
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryInit()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Init();
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryOpen()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryDotGitDirectory()
        {
            // リポジトリのルートディレクトリをカレントとして動作した場合
            {
                var oldDirectory = Environment.CurrentDirectory;
                Environment.CurrentDirectory = GetRepositoryPath();

                var repo = GetGitRepositoryInstance(".");
                repo.Open();

                string repositoryPath = Path.GetFullPath(repo.GetGitDirectory()).TrimEnd('\\');

                Environment.CurrentDirectory = oldDirectory;

                string expectedPath = Path.GetFullPath(Path.Combine(GetRepositoryPath(), ".git")).TrimEnd('\\');

                Assert.AreEqual(expectedPath, repositoryPath);
            }
            // リポジトリ内のサブディレクトリをカレントとして動作した場合
            {
                var directory = Path.Combine(GetRepositoryPath(), "directory1");
                Directory.CreateDirectory(directory);

                var oldDirectory = Environment.CurrentDirectory;
                Environment.CurrentDirectory = directory;

                var repo = GetGitRepositoryInstance(".");
                repo.Open();

                string repositoryPath = Path.GetFullPath(repo.GetGitDirectory()).TrimEnd('\\');

                Environment.CurrentDirectory = oldDirectory;

                string expectedPath = Path.GetFullPath(Path.Combine(GetRepositoryPath(), ".git")).TrimEnd('\\');

                Assert.AreEqual(expectedPath, repositoryPath);
            }
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryRootDirectory()
        {
            {
                var repo = GetGitRepositoryInstance(GetRepositoryPath());
                repo.Open();

                string repositoryPath = Path.GetFullPath(repo.GetRepositoryRoot()).TrimEnd('\\');
                string expectedPath = Path.GetFullPath(GetRepositoryPath()).TrimEnd('\\');

                Assert.AreEqual(expectedPath, repositoryPath);
            }
            {
                var directory = Path.Combine(GetRepositoryPath(), "directory1");
                Directory.CreateDirectory(directory);

                var repo = GetGitRepositoryInstance(directory);
                repo.Open();

                string repositoryPath = Path.GetFullPath(repo.GetRepositoryRoot()).TrimEnd('\\');
                string expectedPath = Path.GetFullPath(GetRepositoryPath()).TrimEnd('\\');

                Assert.AreEqual(expectedPath, repositoryPath);
            }
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryIsInsideRepository()
        {
            {
                var repo = GetGitRepositoryInstance(GetRepositoryPath());
                Assert.IsTrue(repo.IsInsideRepository());
            }

            {
                var repo = GetGitRepositoryInstance("/");
                Assert.IsFalse(repo.IsInsideRepository());
            }

            {
                var repo = GetGitRepositoryInstance(Path.Combine(GetRepositoryPath(), ".git"));
                Assert.IsTrue(repo.IsInsideRepository());
            }

            {
                var directory = Path.Combine(GetRepositoryPath(), "directory1");
                Directory.CreateDirectory(directory);
                var repo = GetGitRepositoryInstance(directory);
                Assert.IsTrue(repo.IsInsideRepository());
            }
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryConfig()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();

            repo.SetConfig("foo.bar", "true");
            string value = repo.GetConfig("foo.bar");

            Assert.AreEqual("true", value);
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryAdd()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();

            File.WriteAllText(GetRepositoryContentFilePath(), GetRepositoryContentFileContent());

            repo.Add(GetRepositoryContentFileRepositoryRelativePath());
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryCommit()
        {
            // Libgit2 にて commit 操作を実装するコストを削減するため、 GitRepositoryCommand を利用します。
            var repo = new GitExternalStorage.Git.GitRepositoryCommand(GetRepositoryPath());

            repo.Commit("[TestGitRepositoryBaseRepositoryCommit] Test commit.");
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryShowHead()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();

            var content = repo.ShowHead(GetRepositoryContentFileRepositoryRelativePath());

            Assert.AreEqual(GetRepositoryContentFileContent(), content);
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryCheckout()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();

            File.WriteAllText(GetRepositoryContentFilePath(), File.ReadAllText(GetRepositoryContentFilePath()) + " append text");

            var filelist = new List<string>()
            {
                GetRepositoryContentFileRepositoryRelativePath()
            };

            repo.Checkout(filelist);

            Assert.AreEqual(File.ReadAllText(GetRepositoryContentFilePath()), GetRepositoryContentFileContent());
        }

        [TestMethod]
        public void TestGitRepositoryBaseRepositoryTree()
        {
            var repo = GetGitRepositoryInstance(GetRepositoryPath());
            repo.Open();

            var treeEntries = repo.GetHeadTree();

            var expected = new List<TreeEntry>()
            {
                new TreeEntry()
                {
                    FilePath = GetRepositoryContentFileRepositoryRelativePath(),
                    FileSize = GetRepositoryContentFileContent().Length
                },
            };

            Assert.IsTrue(Enumerable.SequenceEqual(expected, treeEntries, new TreeEntryComparar()));
        }

        private class TreeEntryComparar : IEqualityComparer<TreeEntry>
        {
            #region IEqualityComparer<TreeEntry> メンバー

            public bool Equals(TreeEntry x, TreeEntry y)
            {
                return (x.FilePath == y.FilePath) && (x.FileSize == y.FileSize);
            }

            public int GetHashCode(TreeEntry obj)
            {
                return obj.GetHashCode();
            }

            #endregion
        }
    }
}
