﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    [TestClass]
    public class SorterTest
    {
        public TestContext TestContext { get; set; }

        public string TestTempDirectory { get; set; }

        private string GetTestDataDirectory()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            return testPath.GetSigloRoot() + @"\Tests\Tools\Sources\Tests\FsAccessLogCheckerTest\FsAccessLogAnalysisLibraryTest";
        }

        public static string ReadAllText(string path, Encoding encoding)
        {
            using (StreamReader reader = new StreamReader(path, encoding))
            {
                string text = reader.ReadToEnd();
                return text;
            }
        }

        public static string ReadAllText(string path)
        {
            using (StreamReader reader = new StreamReader(path))
            {
                string text = reader.ReadToEnd();
                return text;
            }
        }

        public static int GetLineCount(string path)
        {
            using (StreamReader reader = new StreamReader(path))
            {
                int lines = 0;
                while (reader.Peek() > 0)
                {
                    reader.ReadLine();
                    ++lines;
                }
                return lines;
            }
        }

        [TestInitialize]
        public void TestInitialize()
        {
            TestTempDirectory = TestUtility.TestPath.CreateTemporaryDirectory("SIGLO_FSACCESS_TEST");
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestCleanup]
        public void TestCleanup()
        {
            TestUtility.TestPath.DeleteDirectoryIfExisted(TestTempDirectory);
        }

        [TestMethod]
        public void TestSort()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest1.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest1.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortMultiPeriod()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest2.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            var encoding = Encoding.GetEncoding(932);
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath, encoding), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortLargeFile()
        {
            string inputFilePath = TestTempDirectory + @"\largefile.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (StreamWriter sw = new StreamWriter(File.OpenWrite(inputFilePath)))
            {
                Random random = new Random();
                for (int i = 0; i < 400000; ++i)
                {
                    if (random.Next(10) == 0)
                    {
                        sw.WriteLine("ソフトウェアテスト表");
                    }
                    else
                    {
                        sw.WriteLine(string.Format("FS_ACCESS: {{ start: {0}, end: {0}, result: 0x00177A02, handle: 0x0000000000000000, function: \"CreateFile\", path: \"test:/w.txt\", size: 1 }}", random.Next(100000)));
                    }
                }
            }
            if (File.Exists(outputFilePath))
            {
                File.Delete(outputFilePath);
            }
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }
            Assert.AreEqual(GetLineCount(outputFilePath), GetLineCount(inputFilePath));
        }

        [TestMethod]
        [ExpectedException(typeof(ArgumentException))]
        public void TestInvalidTempFilePath()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt?";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }
        }

        [TestMethod]
        public void TestSortBreakFile()
        {
            // 改行コードが混ざっているファイルへのテスト
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_break.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF8File()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf8.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            // BOM なし UTF-8 は読み込みに失敗することがあります (環境依存)
            // 読み込み時に例外が発生しないことだけを確認します
            //Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF8BomFile()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf8_bom.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF16LEFile()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf16.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            // BOM なし UTF-16LE は読み込みに失敗することがあります (環境依存)
            // 読み込み時に例外が発生しないことだけを確認します
            //Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF16LEBomFile()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf16_bom.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF16BEFile()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf16be.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            // BOM なし UTF-16BE は読み込みに失敗することがあります (環境依存)
            // 読み込み時に例外が発生しないことだけを確認します
            //Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }

        [TestMethod]
        public void TestSortUTF16BEBomFile()
        {
            string inputFilePath = GetTestDataDirectory() + @"\input\sorttest_utf16be_bom.txt";
            string expectFilePath = GetTestDataDirectory() + @"\expect\sorttest2.txt";
            string outputFilePath = TestTempDirectory + @"\out.txt";
            using (FileStream stream = File.OpenRead(inputFilePath))
            {
                FsAccessLogStreamSorter sorter = new FsAccessLogStreamSorter(stream, outputFilePath);
                sorter.GetReader().Close();
            }

            Assert.AreEqual(ReadAllText(outputFilePath), ReadAllText(expectFilePath));
        }
    }
}
