﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    internal class TestListener : IFsAccessLogParserListener
    {
        public class CallCount
        {
            public int SetUp { get; internal set; }
            public int OnParseLine { get; internal set; }
            public int TearDown { get; internal set; }
        }

        public List<FsAccessLog> LogList { get; set; } = new List<FsAccessLog>();
        public CallCount Count { get; set; } = new CallCount();

        public void SetUp(FsAccessLogAnalyzer analyzer)
        {
            LogList.Clear();
            Count.SetUp++;
        }

        public void OnParseLine(string line, FsAccessLog log)
        {
            Count.OnParseLine++;
            if (log != null)
            {
                LogList.Add(log);
            }
        }

        public void TearDown()
        {
            Count.TearDown++;
        }
    }

    [TestClass]
    public class FsLogAnalysisLibraryParserTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestAllPossibleElements()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000200734060," +
                "function: \"WriteFile\", priority: Normal, offset: -1, size: -1, name: \"test\", path: \"path/to/file\", new_path: \"test\"," +
                "index: -1, userid: 0x1234567890ABCDEFABCDEF1234567890, dataid: 0xBEEF, systemdataid: 0xBEEF," +
                "applicationid: 0xBEEF, bispartitionid: 0, imagedirectoryid: 0, savedataid: 0xBEEF, savedataspaceid: 0, programid: 0xBEEF," +
                "content_type: 0, file_handle: 0x0000000200734060, proxy_type: 0, contentstorageid: 0," +
                "gamecard_handle: 0x734, gamecard_partition: 0, write_option: Flush, start_tag: true, end_tag: true," +
                "save_data_owner_id: 0xBEEF, save_data_flags: 0, save_data_size: -1, save_data_journal_size: -1, open_mode: 0xBEEF," +
                "image_data_size: 0, width: -1, height: -1, image_orientation: -1, album_report_option: 0xFFFF," +
                "model_type: -1, nfp_mount_target: -1, access_id: -1, data_size: -1, buffer_size: -1, file_size: -1, " +
                "save_data_time_stamp: -1, save_data_commit_id: 0x0000000200734060, " +
                "class_name: test" +
                "}");
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, listener.LogList.Count);
            Assert.IsNull(listener.LogList[0].ParserError);

            FsAccessLog log = listener.LogList[0];
            Assert.AreEqual(0, log.Start);
            Assert.AreEqual(0, log.End);
            Assert.AreEqual(0u, (uint)log.Result);
            Assert.AreEqual(0x0000000200734060u, (ulong)log.Handle);
            Assert.AreEqual("WriteFile", log.Function);
            Assert.AreEqual("Normal", log.Priority);
            Assert.AreEqual(ulong.MaxValue, (ulong)log.Offset);
            Assert.IsTrue(log.Offset.Invalid);
            Assert.AreEqual(ulong.MaxValue, (ulong)log.Size);
            Assert.IsTrue(log.Size.Invalid);
            Assert.AreEqual("test", log.Name);
            Assert.AreEqual("path/to/file", log.Path);
            Assert.AreEqual("test", log.NewPath);
            Assert.AreEqual(0xBEEFu, (uint)log.OpenMode);
            Assert.AreEqual(-1, log.Index);
            Assert.AreEqual(0x1234567890ABCDEFu, log.UserId.Upper);
            Assert.AreEqual(0xABCDEF1234567890u, log.UserId.Lower);
            Assert.AreEqual(0xBEEFu, (ulong)log.DataId);
            Assert.AreEqual(0xBEEFu, (ulong)log.SystemDataId);
            Assert.AreEqual(0xBEEFu, (ulong)log.ApplicationId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootPartition1Root, log.BisPartitionId);
            Assert.AreEqual(FsFunction.ImageDirectoryId.Nand, log.ImageDirectoryId);
            Assert.AreEqual(0xBEEFu, (ulong)log.SaveDataId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.System, log.SaveDataSpaceId);
            Assert.AreEqual(0xBEEFu, (ulong)log.ProgramId);
            Assert.AreEqual(FsFunction.ContentType.Meta, log.ContentType);
            Assert.AreEqual(0x0000000200734060u, (ulong)log.FileHandle);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Code, log.ProxyType);
            Assert.AreEqual(FsFunction.ContentStorageId.System, log.ContentStorageId);
            Assert.AreEqual(0x734u, (uint)log.GamecardHandle);
            Assert.AreEqual(FsFunction.GameCardPartition.Update, log.GamecardPartition);
            Assert.AreEqual(FsFunction.WriteOptionFlag.Flush, log.WriteOption);
            Assert.AreEqual(0u, log.ImageDataSize);
            Assert.AreEqual(-1, log.Width);
            Assert.AreEqual(-1, log.Height);
            Assert.AreEqual(0xFFFFu, (uint)log.AlbumReportOption);
            Assert.IsTrue(log.StartTag);
            Assert.IsTrue(log.EndTag);
            Assert.AreEqual(0, log.SaveDataFlags);
            Assert.AreEqual(0xBEEFu, (ulong)log.SaveDataOwnerId);
            Assert.AreEqual(-1, log.SaveDataSize);
            Assert.AreEqual(-1, log.SaveDataJournalSize);
            Assert.AreEqual(-1, log.SaveDataTimeStamp);
            Assert.AreEqual(0x0000000200734060u, (ulong)log.SaveDataCommitId);
            Assert.AreEqual("test", log.ClassName);
            Assert.AreEqual("test::WriteFile", log.GetFunctionFullName());
            Assert.AreEqual(-1, log.ModelType);
            Assert.AreEqual(-1, log.NfpMountTarget);
            Assert.AreEqual(-1, log.AccessId);
            Assert.AreEqual(-1, log.DataSize);
            Assert.AreEqual(-1, log.BufferSize);
            Assert.AreEqual(-1, log.FileSize);
        }

        [TestMethod]
        public void TestInvalidLog()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 280248, end: 281057, result: 0x00000000, handle: 0x0000000200734060, function: \"WriteFile\", offset: 0, size: 8388864, test: hoge }");
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            // YamlDotNet 3.1.0 未満ではマッチしなかったキーは無視できない
            Assert.AreEqual(0, analyzer.LogList.Count);
            Assert.AreEqual(1, listener.Count.SetUp);
            Assert.AreEqual(1, listener.Count.OnParseLine);
            Assert.AreEqual(1, listener.Count.TearDown);
            Assert.AreEqual(1, listener.LogList.Count);
            Assert.IsFalse(listener.LogList[0].IsAnalyzable());
            Assert.IsNotNull(listener.LogList[0].ParserError);
        }

        [TestMethod]
        public void TestInvalidFetchedLog()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 280248, end: 281057, result: 0x00000000, handle: 0x0000000200734060, fetched: [{test: hoge}] }");
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            // YamlDotNet 3.1.0 未満ではマッチしなかったキーは無視できない
            Assert.AreEqual(0, analyzer.LogList.Count);
            Assert.AreEqual(1, listener.Count.SetUp);
            Assert.AreEqual(1, listener.Count.OnParseLine);
            Assert.AreEqual(1, listener.Count.TearDown);
            Assert.AreEqual(1, listener.LogList.Count);
            Assert.IsFalse(listener.LogList[0].IsAnalyzable());
            Assert.IsNotNull(listener.LogList[0].ParserError);
        }

        [TestMethod]
        public void TestIdLogCompatibility()
        {
            int logCount = 0;
            TestStream testStream = new TestStream();
            Action<string, Array> addLog = (string element, Array ids) =>
            {
                for (int i = 0; i < ids.Length; ++i)
                {
                    testStream.Write("FS_ACCESS: { " + element + ": " + ids.GetValue(i) + " }");
                    testStream.Write("FS_ACCESS: { " + element + ": " + (int)ids.GetValue(i) + " }");
                    logCount += 2;
                }
            };
            addLog("imagedirectoryid",   Enum.GetValues(typeof(FsFunction.ImageDirectoryId)));
            addLog("contentstorageid",   Enum.GetValues(typeof(FsFunction.ContentStorageId)));
            addLog("gamecard_partition", Enum.GetValues(typeof(FsFunction.GameCardPartition)));
            addLog("savedataspaceid",    Enum.GetValues(typeof(FsFunction.SaveDataSpaceId)));
            addLog("content_type",       Enum.GetValues(typeof(FsFunction.ContentType)));
            addLog("proxy_type",         Enum.GetValues(typeof(FsFunction.FileSystemProxyType)));
            addLog("bispartitionid",     Enum.GetValues(typeof(FsFunction.BisPartitionId)));
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(logCount, listener.LogList.Count);
            int checkIndex = 0;
            Assert.AreEqual(FsFunction.ImageDirectoryId.Nand,   listener.LogList[checkIndex++].ImageDirectoryId);
            Assert.AreEqual(FsFunction.ImageDirectoryId.Nand,   listener.LogList[checkIndex++].ImageDirectoryId);
            Assert.AreEqual(FsFunction.ImageDirectoryId.SdCard, listener.LogList[checkIndex++].ImageDirectoryId);
            Assert.AreEqual(FsFunction.ImageDirectoryId.SdCard, listener.LogList[checkIndex++].ImageDirectoryId);

            Assert.AreEqual(FsFunction.ContentStorageId.System, listener.LogList[checkIndex++].ContentStorageId);
            Assert.AreEqual(FsFunction.ContentStorageId.System, listener.LogList[checkIndex++].ContentStorageId);
            Assert.AreEqual(FsFunction.ContentStorageId.User,   listener.LogList[checkIndex++].ContentStorageId);
            Assert.AreEqual(FsFunction.ContentStorageId.User,   listener.LogList[checkIndex++].ContentStorageId);
            Assert.AreEqual(FsFunction.ContentStorageId.SdCard, listener.LogList[checkIndex++].ContentStorageId);
            Assert.AreEqual(FsFunction.ContentStorageId.SdCard, listener.LogList[checkIndex++].ContentStorageId);

            Assert.AreEqual(FsFunction.GameCardPartition.Update, listener.LogList[checkIndex++].GamecardPartition);
            Assert.AreEqual(FsFunction.GameCardPartition.Update, listener.LogList[checkIndex++].GamecardPartition);
            Assert.AreEqual(FsFunction.GameCardPartition.Normal, listener.LogList[checkIndex++].GamecardPartition);
            Assert.AreEqual(FsFunction.GameCardPartition.Normal, listener.LogList[checkIndex++].GamecardPartition);
            Assert.AreEqual(FsFunction.GameCardPartition.Secure, listener.LogList[checkIndex++].GamecardPartition);
            Assert.AreEqual(FsFunction.GameCardPartition.Secure, listener.LogList[checkIndex++].GamecardPartition);

            Assert.AreEqual(FsFunction.SaveDataSpaceId.System,       listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.System,       listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.User,         listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.User,         listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.SdSystem,     listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.SdSystem,     listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.ProperSystem, listener.LogList[checkIndex++].SaveDataSpaceId);
            Assert.AreEqual(FsFunction.SaveDataSpaceId.ProperSystem, listener.LogList[checkIndex++].SaveDataSpaceId);

            Assert.AreEqual(FsFunction.ContentType.Meta,    listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Meta,    listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Control, listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Control, listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Manual,  listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Manual,  listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Logo,    listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Logo,    listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Data,    listener.LogList[checkIndex++].ContentType);
            Assert.AreEqual(FsFunction.ContentType.Data,    listener.LogList[checkIndex++].ContentType);

            Assert.AreEqual(FsFunction.FileSystemProxyType.Code,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Code,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Rom,     listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Rom,     listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Logo,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Logo,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Control, listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Control, listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Manual,  listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Manual,  listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Meta,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Meta,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Data,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Data,    listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Package, listener.LogList[checkIndex++].ProxyType);
            Assert.AreEqual(FsFunction.FileSystemProxyType.Package, listener.LogList[checkIndex++].ProxyType);

            Assert.AreEqual(FsFunction.BisPartitionId.BootPartition1Root,           listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootPartition1Root,           listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootPartition2Root,           listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootPartition2Root,           listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.UserDataRoot,                 listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.UserDataRoot,                 listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part1,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part1,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part2,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part2,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part3,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part3,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part4,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part4,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part5,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part5,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part6,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.BootConfigAndPackage2Part6,   listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.CalibrationBinary,            listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.CalibrationBinary,            listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.CalibrationFile,              listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.CalibrationFile,              listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SafeMode,                     listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SafeMode,                     listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.User,                         listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.User,                         listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.System,                       listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.System,                       listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SystemProperEncryption,       listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SystemProperEncryption,       listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SystemProperPartition,        listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.SystemProperPartition,        listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.Invalid,                      listener.LogList[checkIndex++].BisPartitionId);
            Assert.AreEqual(FsFunction.BisPartitionId.Invalid,                      listener.LogList[checkIndex++].BisPartitionId);

            Assert.AreEqual(logCount, checkIndex);
        }

        [TestMethod]
        public void TestIdLogEnumIdOutOfRange()
        {
            // 将来のバージョンで id が追加されてもパーサーエラーにならないことを確認するテスト
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { imagedirectoryid: 2 }");
            testStream.Write("FS_ACCESS: { imagedirectoryid: -1 }");
            testStream.Write("FS_ACCESS: { imagedirectoryid: Test }");
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(3, listener.LogList.Count);
            Assert.IsNull(listener.LogList[0].ParserError);
            Assert.IsNull(listener.LogList[1].ParserError);
            Assert.IsNull(listener.LogList[2].ParserError);
            Assert.AreEqual("Test", listener.LogList[2].ImageDirectoryId.ToString());
        }

        [TestMethod]
        public void TestEmptyData()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: {}");
            testStream.SetUp();

            TestListener listener = new TestListener();
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            parser.Listener = listener;
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(0, analyzer.LogList.Count);
            Assert.AreEqual(1, listener.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);

            FsAccessLog log = listener.LogList.First();
            Assert.AreEqual(0, log.Start);
            Assert.AreEqual(0, log.End);
            Assert.AreEqual(0u, (ulong)log.Offset);
            Assert.IsFalse(log.Offset.Invalid);
            Assert.AreEqual(0u, (ulong)log.Size);
            Assert.IsFalse(log.Size.Invalid);
            Assert.AreEqual(0u, (uint)log.Result);
            Assert.AreEqual(0u, (ulong)log.Handle);
            Assert.AreEqual(0, (int)log.WriteOption);
            Assert.IsFalse(log.ForSystem);
            Assert.IsNull(log.Function);
            Assert.IsNull(log.Path);
            Assert.IsNull(log.Name);
            Assert.IsNull(log.NewPath);
            Assert.IsNull(log.OpenMode);
            Assert.IsNull(log.Index);
            Assert.IsNull(log.UserId);
            Assert.IsNull(log.DataId);
            Assert.IsNull(log.SdkVersion);
            Assert.IsNull(log.Spec);
            Assert.IsNull(log.ParserError);
            Assert.IsNull(log.AnalyzerError);
            Assert.IsNull(log.BisPartitionId);
            Assert.IsNull(log.ContentStorageId);
            Assert.IsNull(log.ImageDirectoryId);
            Assert.IsNull(log.SaveDataId);
            Assert.IsNull(log.SaveDataSpaceId);
            Assert.IsNull(log.SystemDataId);
            Assert.IsNull(log.FileHandle);
            Assert.IsNull(log.ProgramId);
            Assert.IsNull(log.ContentType);
            Assert.IsNull(log.ProxyType);
            Assert.IsNull(log.GamecardHandle);
            Assert.IsNull(log.GamecardPartition);
            Assert.IsNull(log.SaveDataFlags);
            Assert.IsNull(log.SaveDataOwnerId);
            Assert.IsNull(log.SaveDataSize);
            Assert.IsNull(log.SaveDataJournalSize);
            Assert.IsNull(log.SaveDataTimeStamp);
            Assert.IsNull(log.SaveDataCommitId);
            Assert.IsNull(log.ImageDataSize);
            Assert.IsNull(log.Width);
            Assert.IsNull(log.Height);
            Assert.IsNull(log.ImageOrientation);
            Assert.IsNull(log.AlbumReportOption);
            Assert.IsNull(log.ClassName);
            Assert.AreEqual(FsApiType.Misc, log.Type);
            Assert.AreEqual(FsApiAccessType.None, log.AccessType);
            Assert.AreEqual(FsMountTarget.Unknown, log.MountTarget);
            Assert.AreEqual(0, log.MountHash);
        }

        [TestMethod]
        public void TestWinPathSeparator()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 4, end: 5, result: 0x00177A02, handle: 0x0000000000000000, function: \"CreateFile\", path: \"test:\\w.txt\", size: -1 }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);
            // パス区切りは / に正規化されている
            Assert.AreEqual("test:/w.txt", analyzer.LogList[0].Path);
            Assert.IsTrue(analyzer.IsCompleteLog);
        }

        [TestMethod]
        public void TestAccessLogStartLog()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(0, 1, "test");
            ulong handle = testStream.GetUniqueHandle();
            testStream.OpenFile(1, 2, handle, "test:/1");
            testStream.WriteFile(2, 3, handle, 0, 4);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.HasSdkVersion());
            Assert.AreEqual("NX", analyzer.LogSpec);
            Assert.IsFalse(analyzer.ForSystem);
            Assert.IsTrue(analyzer.IsCompleteLog);
        }

        [TestMethod]
        public void TestAccessLogStartLogForSystem()
        {
            TestStream testStream = new TestStream();
            testStream.StartLogForSystem();
            testStream.MountRom(0, 1, "test");
            ulong handle = testStream.GetUniqueHandle();
            testStream.OpenFile(1, 2, handle, "test:/1");
            testStream.WriteFile(2, 3, handle, 0, 4);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.HasSdkVersion());
            Assert.IsFalse(analyzer.HasSystemAccess());
            Assert.AreEqual("NX", analyzer.LogSpec);
            Assert.IsTrue(analyzer.ForSystem);
        }

        [TestMethod]
        public void TestAccessLogStartLogMulti()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(0, 1, "test");
            ulong handle = testStream.GetUniqueHandle();
            testStream.OpenFile(1, 2, handle, "test:/1");
            testStream.WriteFile(2, 3, handle, 0, 4);
            testStream.StartLog();
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<FsAccessLogAnalyzer> analyzers = parser.Parse();
            Assert.AreEqual(2, analyzers.Count);
            Assert.AreEqual(3, analyzers[0].LogList.Count);
            Assert.IsTrue(analyzers[0].IsCompleteLog);
            Assert.IsTrue(analyzers[0].HasSdkVersion());
            Assert.AreEqual("NX", analyzers[0].LogSpec);
            Assert.IsTrue(analyzers[1].IsCompleteLog);
            Assert.IsTrue(analyzers[1].HasSdkVersion());
            Assert.AreEqual("NX", analyzers[1].LogSpec);
        }

        [TestMethod]
        public void TestAccessLogStartLogMidstream()
        {
            TestStream testStream = new TestStream();
            {
                testStream.MountRom(0, 1, "test");
                ulong handle = testStream.GetUniqueHandle();
                testStream.OpenFile(1, 2, handle, "test:/1");
                testStream.WriteFile(2, 3, handle, 0, 4);
            }
            testStream.StartLog();
            {
                testStream.MountRom(0, 1, "test");
                ulong handle = testStream.GetUniqueHandle();
                testStream.OpenFile(1, 2, handle, "test:/1");
                testStream.WriteFile(2, 3, handle, 0, 4);
            }
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<FsAccessLogAnalyzer> analyzers = parser.Parse();
            Assert.AreEqual(2, analyzers.Count);
            for (int i = 0; i < 2; ++i)
            {
                Assert.AreEqual(3, analyzers[i].LogList.Count);
            }
            Assert.IsTrue(analyzers[0].IsCompleteLog);
            Assert.IsFalse(analyzers[0].HasSdkVersion());
            Assert.IsNull(analyzers[0].LogSpec);
            Assert.IsTrue(analyzers[1].IsCompleteLog);
            Assert.IsTrue(analyzers[1].HasSdkVersion());
            Assert.AreEqual("NX", analyzers[1].LogSpec);
        }

        [TestMethod]
        public void TestExpectedSdkVersion()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog("0.16.17");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);

            Assert.IsTrue(analyzer.IsExpectedSdkVersion(new Version(0, 16, 17)));
            Assert.IsFalse(analyzer.IsExpectedSdkVersion(new Version(0, 16, 18)));

            Assert.IsTrue(analyzer.IsExpectedSdkVersion(new Version(0, 0, 0)));
            Assert.IsFalse(analyzer.IsExpectedSdkVersion(new Version(1, 0, 0)));

            Assert.IsTrue(analyzer.IsExpectedSdkVersion(new Version(0, 16, 0)));
            Assert.IsFalse(analyzer.IsExpectedSdkVersion(new Version(0, 17, 0)));
        }

        [TestMethod]
        public void TestExpectedSdkVersionEmptyLog()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: {}");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsFalse(analyzer.IsExpectedSdkVersion(new Version(0, 0, 0)));
        }

        [TestMethod]
        public void TestExpectedSdkVersionInvalidVesion()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { sdk_version: Invalid, spec: NX }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsFalse(analyzer.IsExpectedSdkVersion(new Version(0, 0, 0)));
        }

        [TestMethod]
        public void TestMountSubstitution()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 53364, end: 53367, result: 0x00000000, handle: 0x0000000000000000, function: \"MountHost\", name: \"rom\" }");
            testStream.Write("FS_ACCESS: { start: 53347, end: 53367, result: 0x00000000, handle: 0x0000000000000000, function: \"MountRom\", name: \"rom\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList.All(log => log.MountTarget == FsMountTarget.Rom));
        }

        [TestMethod]
        public void TestMountFailure()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 0, end: 0, result: 0x00000001, handle: 0x0000000000000000, function: \"MountHost\", name: \"rom\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 1, result: 0x00000000, handle: 0x0000000000000000, function: \"MountRom\", name: \"rom\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.AreEqual(FsMountTarget.Host, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Rom, analyzer.LogList[1].MountTarget);
        }

        [TestMethod]
        public void TestUserId()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 61025, end: 61224, result: 0x00000000, handle: 0x0000000000000000, function: \"MountSaveData\", name: \"save\", userid: 0xFEDCBA98765432100123456789ABCDEF }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.AreEqual(new UserId(0xFEDCBA9876543210, 0x0123456789ABCDEF), analyzer.LogList[0].UserId);
        }

        [TestMethod]
        public void TestWriteFileWriteCount()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 61231, end: 61234, result: 0x00000000, handle: 0x00000000085d0090, function: \"WriteFile\", offset: 0, size: 4 }");
            testStream.Write("FS_ACCESS: { start: 61231, end: 61234, result: 0x00000000, handle: 0x00000000085d0090, function: \"WriteFile\", offset: 0, size: 4, write_option: Flush }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.AreEqual(1, analyzer.LogList[0].WriteCount);
            Assert.AreEqual(2, analyzer.LogList[1].WriteCount);
        }

        [TestMethod]
        public void TestCommitSaveData()
        {
            TestStream testStream = new TestStream();
            testStream.MountRom(0, 1, "rom");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"rom\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.AreEqual(FsMountTarget.Rom, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Rom, analyzer.LogList[1].MountTarget);
        }

        [TestMethod]
        public void TestTotalTime()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(1000 * 60 * 10, 1000 * 60 * 10 + 1, "rom");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.AreEqual(1, analyzer.GetTotalMilliseconds());
        }

        [TestMethod]
        public void TestWriteSize()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(0, 1, "test");
            ulong handle = testStream.GetUniqueHandle();
            testStream.CreateFile(1, 2, "test:/1", 1024);
            testStream.OpenFile(2, 3, handle, "test:/1");
            testStream.SetFileSize(3, 4, handle, 4096);
            testStream.WriteFile(4, 5, handle, 0, 4);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(4u, FsAccessLogAnalyzer.GetTotalWriteSize(analyzer.LogList));
        }

        [TestMethod]
        public void TestMinusSize()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 4, end: 5, result: 0x00177A02, handle: 0x0000000000000000, function: \"CreateFile\", path: \"test:/test\", size: -1 }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsTrue(analyzer.LogList[0].Size.Invalid);
        }

        [TestMethod]
        public void TestMinusOffset()
        {
            TestStream testStream = new TestStream();
            testStream.Write("FS_ACCESS: { start: 4, end: 5, result: 0x00177A02, handle: 0x0000000000000000, function: \"WriteFile\", offset: -1, size: 12 }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsTrue(analyzer.LogList[0].Offset.Invalid);
        }

        [TestMethod]
        public void TestRewindTime()
        {
            TestStream testStream = new TestStream();
            ulong handle = testStream.GetUniqueHandle();
            testStream.ReadFile(20000, 20002, handle, 0, 4);
            testStream.ReadFile(19999, 20001, handle, 0, 4);    // 1つ前の期間と重なる
            testStream.ReadFile(19998, 19999, handle, 0, 4);    // 1つ前の期間と境界で重なる
            testStream.ReadFile(10001, 10002, handle, 0, 4);    // 巻き戻りが発生しているが許容範囲内の時間差
            testStream.ReadFile(0, 1, handle, 0, 4);            // 巻き戻りが発生し許容範囲外の時間差
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            // ソート対応により巻戻りはなくなった
            Assert.AreEqual(5, analyzer.LogList.Count);
            Assert.AreEqual(20002, FsAccessLogAnalyzer.GetTotalMilliseconds(analyzer.LogList));
        }

        [TestMethod]
        public void TestResultFailure()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(0, 1, "test");
            ulong handle = testStream.GetUniqueHandle();
            testStream.CreateFile(1, 2, "test:/1", 1024);
            testStream.OpenFile(2, 3, handle, "test:/1");
            testStream.WriteFile(4, 5, handle, 0, 4);
            testStream.WriteFile(5, 6, handle, 0, 1024, 0x1);
            testStream.ReadFile(6, 7, handle, 0, 4);
            testStream.ReadFile(7, 8, handle, 0, 1024, 0x1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(4u, FsAccessLogAnalyzer.GetTotalWriteSize(analyzer.LogList));
            Assert.AreEqual(4u, FsAccessLogAnalyzer.GetTotalReadSize(analyzer.LogList));
            Assert.AreEqual(1, analyzer.LogList[3].WriteCount);
            // 失敗時は計上されない
            Assert.AreEqual(0, analyzer.LogList[4].WriteCount);
        }

        [TestMethod]
        public void TestSdCardOutputLog()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.HasSdkVersion());
            Assert.AreEqual("NX", analyzer.LogSpec);
            Assert.IsTrue(analyzer.IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputLogIncomplateEmpty()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(0, analyzer.LogList.Count);
            Assert.IsFalse(analyzer.IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputLogIncomplateNotEmpty()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);
            Assert.IsFalse(analyzer.IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLog()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.CreateFile(40726, 40790, "test:/1", 1024);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(1, analyzer[0].LogList.Count);
            Assert.AreEqual(2, analyzer[1].LogList.Count);
            Assert.IsTrue(analyzer[0].IsCompleteLog);
            Assert.IsTrue(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLogTwoStartTag()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.CreateFile(40726, 40790, "test:/2", 1024);
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.CreateFile(40726, 40790, "test:/2", 1024);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(2, analyzer[0].LogList.Count);
            Assert.AreEqual(2, analyzer[1].LogList.Count);
            Assert.IsTrue(analyzer[0].IsCompleteLog);
            Assert.IsTrue(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLogFormarEmpty()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(0, analyzer[0].LogList.Count);
            Assert.AreEqual(1, analyzer[1].LogList.Count);
            Assert.IsTrue(analyzer[0].IsCompleteLog);
            Assert.IsTrue(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLogLatterEmpty()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.StartLog();
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(1, analyzer[0].LogList.Count);
            Assert.AreEqual(0, analyzer[1].LogList.Count);
            Assert.IsTrue(analyzer[0].IsCompleteLog);
            Assert.IsTrue(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLogFormerImcomplete()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.CreateFile(40726, 40790, "test:/1", 1024);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(1, analyzer[0].LogList.Count);
            Assert.AreEqual(2, analyzer[1].LogList.Count);
            Assert.IsFalse(analyzer[0].IsCompleteLog);
            Assert.IsTrue(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputMultipleLogLatterImcomplete()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { end_tag: true }");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.CreateFile(40726, 40790, "test:/1", 1024);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(1, analyzer[0].LogList.Count);
            Assert.AreEqual(2, analyzer[1].LogList.Count);
            Assert.IsTrue(analyzer[0].IsCompleteLog);
            Assert.IsFalse(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestSdCardOutputImcompleteMultipleLog()
        {
            TestStream testStream = new TestStream("0x000000000000007E");
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.Write("FS_ACCESS: { start_tag: true }");
            testStream.StartLog();
            testStream.MountSaveData(40711, 40726, "test", 0);
            testStream.CreateFile(40726, 40790, "test:/1", 1024);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            List<Nintendo.FsAccessLogAnalysis.FsAccessLogAnalyzer> analyzer = parser.Parse();
            Assert.AreEqual(2, analyzer.Count);
            Assert.AreEqual("NX", analyzer[0].LogSpec);
            Assert.AreEqual("NX", analyzer[1].LogSpec);
            Assert.AreEqual(1, analyzer[0].LogList.Count);
            Assert.AreEqual(2, analyzer[1].LogList.Count);
            Assert.IsFalse(analyzer[0].IsCompleteLog);
            Assert.IsFalse(analyzer[1].IsCompleteLog);
        }

        [TestMethod]
        public void TestMountContentStorage()
        {
            TestStream testStream = new TestStream();
            testStream.MountContentStorage(0, 1, "system", FsFunction.ContentStorageId.System);
            testStream.MountContentStorage(0, 1, "user", FsFunction.ContentStorageId.User);
            testStream.MountContentStorage(0, 1, "sdcard", FsFunction.ContentStorageId.SdCard);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsNotNull(analyzer.LogList[0].ContentStorageId);
            Assert.IsTrue(analyzer.HasSystemAccess());
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.SdCard, analyzer.LogList[2].MountTarget);
        }

        [TestMethod]
        public void TestMountImageDirectory()
        {
            TestStream testStream = new TestStream();
            testStream.MountImageDirectory(0, 0, 1, "nand", FsFunction.ImageDirectoryId.Nand);
            testStream.MountImageDirectory(0, 0, 1, "sdcard", FsFunction.ImageDirectoryId.SdCard);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsTrue(analyzer.HasSystemAccess());
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("ImageDirectoryId: "));
            Assert.IsNotNull(analyzer.LogList[0].ImageDirectoryId);
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.SdCard, analyzer.LogList[1].MountTarget);
        }

        [TestMethod]
        public void TestMountResultFailureSameName()
        {
            // 失敗したマウントの解決が正しくできているかどうかテストする
            TestStream testStream = new TestStream();
            testStream.MountImageDirectory(0x00320002, 0, 1, "test", FsFunction.ImageDirectoryId.Nand);
            testStream.MountImageDirectory(0x00320002, 0, 1, "test", FsFunction.ImageDirectoryId.SdCard);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(2, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("ImageDirectoryId: "));
            Assert.IsNotNull(analyzer.LogList[0].ImageDirectoryId);
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.SdCard, analyzer.LogList[1].MountTarget);
        }

        [TestMethod]
        public void TestMountSystemSaveData()
        {
            TestStream testStream = new TestStream();
            testStream.MountSystemSaveData(0, 1, "system", FsFunction.SaveDataSpaceId.System, 0, 0);
            testStream.MountSystemSaveData(0, 1, "user", FsFunction.SaveDataSpaceId.User, 0, 0);
            testStream.MountSystemSaveData(0, 1, "sdsystem", FsFunction.SaveDataSpaceId.SdSystem, 0, 0);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.IsTrue(analyzer.HasSystemAccess());
            Assert.IsNotNull(analyzer.LogList[0].SaveDataSpaceId);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("SaveDataId: "));
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("SaveDataSpaceId: "));
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Nand, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.SdCard, analyzer.LogList[2].MountTarget);
        }

        [TestMethod]
        public void TestOpenDirectory()
        {
            TestStream testStream = new TestStream();
            testStream.MountRom(0, 1, "test");
            testStream.Write("FS_ACCESS: { start: 16199, end: 16201, result: 0x00000000, handle: 0x000000194ae0c1e8, function: \"OpenDirectory\", path: \"test:/d\" }");
            testStream.Write("FS_ACCESS: { start: 16201, end: 16201, result: 0x00000000, handle: 0x000000194ae0c1e8, function: \"CloseDirectory\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.IsTrue(analyzer.IsCompleteLog);
            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.AreEqual(FsMountTarget.Rom, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Rom, analyzer.LogList[2].MountTarget);
        }

        [TestMethod]
        public void TestSaveDataGroupAccess()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.ExtendSaveData(1, 2, 100, 100, 0);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(1, analyzer.LogList.Count);

            FsAccessLog extendSaveData = analyzer.LogList[0];
            Assert.AreEqual("ExtendSaveData", extendSaveData.Function);
            Assert.AreEqual(FsMountTarget.Nand, extendSaveData.MountTarget);
            Assert.AreEqual(1, extendSaveData.WriteCount);
            Assert.AreEqual(0u, extendSaveData.ActuallyWrittenSize);
        }

        [TestMethod]
        public void TestAlbumApi()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.SaveScreenshot(0, 1, 3686400, 3);
            testStream.SaveAndShareScreenshot(0, 1, 3686400, 3, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(2, analyzer.LogList.Count);

            FsAccessLog saveScreenshot = analyzer.LogList[0];
            Assert.AreEqual("SaveScreenshot", saveScreenshot.Function);
            Assert.AreEqual(FsMountTarget.Nand, saveScreenshot.MountTarget);
            Assert.AreEqual(6, saveScreenshot.WriteCount);
            Assert.AreEqual(500 * 1024u, saveScreenshot.ActuallyWrittenSize);

            FsAccessLog saveAndShareScreenshot = analyzer.LogList[1];
            Assert.AreEqual("SaveAndShareScreenshot", saveAndShareScreenshot.Function);
            Assert.AreEqual(FsMountTarget.Nand, saveAndShareScreenshot.MountTarget);
            Assert.AreEqual(6, saveAndShareScreenshot.WriteCount);
            Assert.AreEqual(500 * 1024u, saveAndShareScreenshot.ActuallyWrittenSize);
        }

        [TestMethod]
        public void TestAlbumApiUnknownImageSize()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.SaveScreenshot(0, 1, 3686400, 3, 0, 0);
            testStream.SaveAndShareScreenshot(0, 1, 3686400, 3, 1, 1920, 1080);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(0, analyzer.LogList.Count);
            Assert.AreEqual(2, analyzer.ErrorLogList.Count);

            FsAccessLog saveScreenshot = analyzer.ErrorLogList[0];
            Assert.AreEqual(FsMountTarget.Unknown, saveScreenshot.MountTarget);

            FsAccessLog saveAndShareScreenshot = analyzer.ErrorLogList[1];
            Assert.AreEqual(FsMountTarget.Unknown, saveAndShareScreenshot.MountTarget);

            // プロパティが例外送出しないことを確認（値は不定）
            Func<object, bool> alwaysTrue = (object value) => { return true; };
            Assert.IsTrue(alwaysTrue(saveScreenshot.WriteCount));
            Assert.IsTrue(alwaysTrue(saveScreenshot.ActuallyWrittenSize));
            Assert.IsTrue(alwaysTrue(saveAndShareScreenshot.WriteCount));
            Assert.IsTrue(alwaysTrue(saveAndShareScreenshot.ActuallyWrittenSize));
        }
    }

    [TestClass]
    public class FsLogAnalysisLibraryLogHashTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestUserId()
        {
            // MountSaveData
            {
                TestStream testStream = new TestStream();
                testStream.MountSaveData(0, 1, "TestA", 1);
                testStream.MountSaveData(0, 1, "TestB", 2);
                testStream.MountSaveData(0, 1, "TestC", 1);
                testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
                testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
                testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(6, analyzer.LogList.Count);
                Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("UserId: "));
                // 異なる UserId の場合、ハッシュは一致しない
                Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
                // 同じ UserId の場合、ハッシュは一致する
                Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
            }

            // GetSaveDataSize
            {
                TestStream testStream = new TestStream();
                testStream.GetSaveDataSize(0, 1, 0);
                testStream.GetSaveDataSize(0, 1, 1);
                testStream.GetSaveDataSize(0, 1, 0);
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(3, analyzer.LogList.Count);

                // 異なる UserId の場合、ハッシュは一致しない
                Assert.AreNotEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[1].GetHashCode());
                // 同じ UserId の場合、ハッシュは一致する
                Assert.AreEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[2].GetHashCode());
            }
        }

        [TestMethod]
        public void TestDataId()
        {
            TestStream testStream = new TestStream();
            testStream.MountSystemSaveData(0, 1, "TestA", FsFunction.SaveDataSpaceId.System, 1, 0);
            testStream.MountSystemSaveData(0, 1, "TestB", FsFunction.SaveDataSpaceId.System, 2, 0);
            testStream.MountSystemSaveData(0, 1, "TestC", FsFunction.SaveDataSpaceId.System, 1, 0);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("DataId: "));
            // 異なる DataId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            // 同じ DataId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
        }

        [TestMethod]
        public void TestIndexForAoc()
        {
            TestStream testStream = new TestStream();
            testStream.MountAddOnContent(0, 1, "TestA", 1);
            testStream.MountAddOnContent(0, 1, "TestB", 2);
            testStream.MountAddOnContent(0, 1, "TestC", 1);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("index: "));
            // 異なる Index の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            // 同じ Index の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
        }

        [TestMethod]
        public void TestIndexForCacheStorage()
        {
            // index 指定されたら外側で必ず有効になるので、
            // false の場合はテストしない
            FsGuideline.IsCacheStorageIndexUsed = true;

            // MountCacheStorage
            {
                TestStream testStream = new TestStream();
                testStream.MountCacheStorage(0, 1, "TestA");
                testStream.MountCacheStorage(0, 1, "TestB", 1);
                testStream.MountCacheStorage(0, 1, "TestC", 0);
                testStream.OpenFile(0, 1, 1, "TestA:/test.txt");
                testStream.OpenFile(0, 1, 2, "TestB:/test.txt");
                testStream.OpenFile(0, 1, 3, "TestC:/test.txt");
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(6, analyzer.LogList.Count);
                Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("index: "));
                // 異なる Index の場合、ハッシュは一致しない
                Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
                // 同じ Index の場合、ハッシュは一致する
                Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
            }

            // GetCacheStorageSize
            {
                TestStream testStream = new TestStream();
                testStream.GetCacheStorageSize(0, 1);
                testStream.GetCacheStorageSize(0, 1, 1);
                testStream.GetCacheStorageSize(0, 1, 0);
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(3, analyzer.LogList.Count);

                // 異なる Index の場合、ハッシュは一致しない
                Assert.AreNotEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[1].GetHashCode());
                // 同じ Index の場合、ハッシュは一致する
                Assert.AreEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[2].GetHashCode());
            }
        }

        [TestMethod]
        public void TestApplicationId()
        {
            TestStream testStream = new TestStream();
            testStream.MountBcatSaveData(0, 1, "TestA", 1);
            testStream.MountBcatSaveData(0, 1, "TestB", 2);
            testStream.MountBcatSaveData(0, 1, "TestC", 1);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("ApplicationId: "));
            // 異なる ApplicationId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            // 同じ ApplicationId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
        }

        [TestMethod]
        public void TestBisPartitionId()
        {
            TestStream testStream = new TestStream();
            testStream.MountBis(0, 1, "TestA", FsFunction.BisPartitionId.BootPartition1Root);
            testStream.MountBis(0, 1, "TestB", FsFunction.BisPartitionId.BootPartition2Root);
            testStream.MountBis(0, 1, "TestC", FsFunction.BisPartitionId.BootPartition1Root);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("BisPartitionId: "));
            // 異なる BisPartitionId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            // 同じ BisPartitionId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
        }

        [TestMethod]
        public void TestContentStorageId()
        {
            TestStream testStream = new TestStream();
            testStream.MountContentStorage(0, 1, "TestA", FsFunction.ContentStorageId.System);
            testStream.MountContentStorage(0, 1, "TestB", FsFunction.ContentStorageId.User);
            testStream.MountContentStorage(0, 1, "TestC", FsFunction.ContentStorageId.System);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("ContentStorageId: "));
            // 異なる ContentStorageId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[4].MountTarget);
            // 同じ ContentStorageId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[5].MountTarget);
        }

        [TestMethod]
        public void TestSaveDataSpaceId()
        {
            TestStream testStream = new TestStream();
            testStream.MountSystemSaveData(0, 1, "TestA", FsFunction.SaveDataSpaceId.System, 0, 0);
            testStream.MountSystemSaveData(0, 1, "TestB", FsFunction.SaveDataSpaceId.User, 0, 0);
            testStream.MountSystemSaveData(0, 1, "TestC", FsFunction.SaveDataSpaceId.System, 0, 0);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("SaveDataSpaceId: "));
            // 異なる SaveDataSpaceId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[4].MountTarget);
            // 同じ SaveDataSpaceId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[5].MountTarget);
        }

        [TestMethod]
        public void TestSaveDataId()
        {
            TestStream testStream = new TestStream();
            testStream.MountSystemSaveData(0, 1, "TestA", FsFunction.SaveDataSpaceId.System, 0, 0);
            testStream.MountSystemSaveData(0, 1, "TestB", FsFunction.SaveDataSpaceId.System, 1, 0);
            testStream.MountSystemSaveData(0, 1, "TestC", FsFunction.SaveDataSpaceId.System, 0, 0);
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestA\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestB\" }");
            testStream.Write("FS_ACCESS: { start: 1, end: 2, result: 0x00000000, handle: 0x00000000085d0090, function: \"CommitSaveData\", name: \"TestC\" }");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].GetMountArgumentsString().Contains("SaveDataSpaceId: "));
            // 異なる SaveDataId の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[4].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[4].MountTarget);
            // 同じ SaveDataId の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[3].GetHashCode(), analyzer.LogList[5].GetHashCode());
            Assert.AreEqual(analyzer.LogList[3].MountTarget, analyzer.LogList[5].MountTarget);
        }

        [TestMethod]
        public void TestPath()
        {
            TestStream testStream = new TestStream();
            testStream.StartLog();
            testStream.MountRom(0, 1, "test");
            testStream.OpenFile(1, 2, 1, "test:/1");
            testStream.OpenFile(2, 3, 2, "test:/2");
            testStream.OpenFile(3, 4, 3, "test:/1");
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(4, analyzer.LogList.Count);
            // 異なる path の場合、ハッシュは一致しない
            Assert.AreNotEqual(analyzer.LogList[1].GetHashCode(), analyzer.LogList[2].GetHashCode());
            // 同じ path の場合、ハッシュは一致する
            Assert.AreEqual(analyzer.LogList[1].GetHashCode(), analyzer.LogList[3].GetHashCode());
        }

        [TestMethod]
        public void TestSameHashMount()
        {
            // 同じ関数でマウント名違い
            {
                TestStream testStream = new TestStream();
                testStream.MountRom(0, 1, "TestA", 0);
                testStream.MountRom(0, 1, "TestB", 0);
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(2, analyzer.LogList.Count);
                Assert.AreEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[1].GetHashCode());
            }

            // 異なる関数 (同一対象) でマウント名違い
            {
                TestStream testStream = new TestStream();
                testStream.MountSaveData(0, 1, "TestA", 1, 1);
                testStream.MountSaveDataReadOnly(0, 1, "TestB", 1, 1);
                testStream.SetUp();

                FsAccessLogParser parser = new FsAccessLogParser(testStream);
                FsAccessLogAnalyzer analyzer = parser.Parse()[0];
                Assert.AreEqual(2, analyzer.LogList.Count);
                Assert.AreEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[1].GetHashCode());
            }
        }

        [TestMethod]
        public void TestSameArgumentMount()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeviceSaveData(0, 1, "TestA", 1);
            testStream.MountCacheStorage(0, 1, "TestB", (ulong)1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(2, analyzer.LogList.Count);
            // 同じ引数でもマウント関数が違う場合は、異なるハッシュ
            Assert.AreNotEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[1].GetHashCode());
        }

        private static void TestBcatImpl(TestStream testStream, Tuple<string, string, int>[] expected)
        {
            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            var groups = parser.Parse()[0].LogList
                .GroupBy(v => v.GetHashCode())
                .OrderBy(v => v.First().Start);
            var index = 0;
            foreach (var group in groups)
            {
                System.Diagnostics.Debug.WriteLine(index);

                var expectedValue = expected[index];
                Assert.AreEqual(expectedValue.Item1, group.First().GetFunctionFullName());
                Assert.AreEqual(expectedValue.Item2, group.First().Path);
                Assert.AreEqual(expectedValue.Item3, group.Count());

                foreach (var item in group)
                {
                    System.Diagnostics.Debug.WriteLine("  " + item.GetFunctionFullName() + " " + item.Path);
                }

                ++index;
            }
        }

        [TestMethod]
        public void TestBcat()
        {
            long time = 0;
            const string Directory1 = "test";
            const string Directory2 = "test2";
            const string File1 = "test/test.txt";
            const string File2 = "test/test2.txt";
            const ulong DirectoryHandle1 = 1;
            const ulong DirectoryHandle2 = 2;
            const ulong FileHandle1 = 3;
            const ulong FileHandle2 = 4;

            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(time, time);
            ++time;

            testStream.DeliveryCacheDirectoryOpen(time, time, DirectoryHandle1, Directory1);
            ++time;
            testStream.DeliveryCacheDirectoryOpen(time, time, DirectoryHandle2, Directory2);
            ++time;
            testStream.DeliveryCacheFileOpen(time, time, FileHandle1, File1);
            ++time;
            testStream.DeliveryCacheFileOpen(time, time, FileHandle2, File2);
            ++time;

            testStream.EnumerateDeliveryCacheDirectory(time, time);
            ++time;
            testStream.EnumerateDeliveryCacheDirectory(time, time);
            ++time;
            testStream.DeliveryCacheDirectoryRead(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheDirectoryRead(time, time, DirectoryHandle2);
            ++time;
            testStream.DeliveryCacheFileRead(time, time, FileHandle1, 0, 16);
            ++time;
            testStream.DeliveryCacheFileRead(time, time, FileHandle2, 0, 16);
            ++time;

            testStream.DeliveryCacheDirectoryClose(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheDirectoryDestructor(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheDirectoryClose(time, time, DirectoryHandle2);
            ++time;
            testStream.DeliveryCacheDirectoryDestructor(time, time, DirectoryHandle2);
            ++time;
            testStream.DeliveryCacheFileClose(time, time, FileHandle1);
            ++time;
            testStream.DeliveryCacheFileDestructor(time, time, FileHandle1);
            ++time;
            testStream.DeliveryCacheFileClose(time, time, FileHandle2);
            ++time;
            testStream.DeliveryCacheFileDestructor(time, time, FileHandle2);
            ++time;

            testStream.UnmountDeliveryCacheStorage(time, time);
            ++time;

            testStream.SetUp();

            var expected = new Tuple<string, string, int>[]
            {
                new Tuple<string, string, int>("MountDeliveryCacheStorage", null, 2),
                new Tuple<string, string, int>("DeliveryCacheDirectory::Open", Directory1, 3),
                new Tuple<string, string, int>("DeliveryCacheDirectory::Open", Directory2, 3),
                new Tuple<string, string, int>("DeliveryCacheFile::Open", File1, 3),
                new Tuple<string, string, int>("DeliveryCacheFile::Open", File2, 3),
                new Tuple<string, string, int>("EnumerateDeliveryCacheDirectory", null, 2),
                new Tuple<string, string, int>("DeliveryCacheDirectory::~DeliveryCacheDirectory", null, 4),
            };

            TestBcatImpl(testStream, expected);
        }

        [TestMethod]
        public void TestBcatWithZeroHandle()
        {
            long time = 0;
            const string Directory1 = "test";
            const string File1 = "test/test.txt";
            ulong DirectoryHandle1 = 0;
            const ulong FileHandle1 = 0;

            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(time, time);
            ++time;
            testStream.DeliveryCacheDirectoryOpen(time, time, DirectoryHandle1, Directory1);
            ++time;
            testStream.DeliveryCacheFileOpen(time, time, FileHandle1, File1);
            ++time;
            testStream.EnumerateDeliveryCacheDirectory(time, time);
            ++time;
            testStream.DeliveryCacheDirectoryRead(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheFileRead(time, time, FileHandle1, 0, 16);
            ++time;
            testStream.DeliveryCacheDirectoryClose(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheDirectoryDestructor(time, time, DirectoryHandle1);
            ++time;
            testStream.DeliveryCacheFileClose(time, time, FileHandle1);
            ++time;
            testStream.DeliveryCacheFileDestructor(time, time, FileHandle1);
            ++time;
            testStream.UnmountDeliveryCacheStorage(time, time);
            ++time;
            testStream.SetUp();

            // パスが null になるため、 直感的でないグループに分かれる
            var expected = new Tuple<string, string, int>[]
            {
                new Tuple<string, string, int>("MountDeliveryCacheStorage", null, 2),
                new Tuple<string, string, int>("DeliveryCacheDirectory::Open", Directory1, 1),
                new Tuple<string, string, int>("DeliveryCacheFile::Open", File1, 1),
                new Tuple<string, string, int>("EnumerateDeliveryCacheDirectory", null, 1),
                new Tuple<string, string, int>("DeliveryCacheDirectory::Read", null, 1),
                new Tuple<string, string, int>("DeliveryCacheFile::Read", null, 5),
            };

            TestBcatImpl(testStream, expected);
        }

        [TestMethod]
        public void TestNfp()
        {
            TestStream testStream = new TestStream();
            testStream.NfpMount(0, 1, 0);
            testStream.NfpMount(0, 1, 1);
            testStream.NfpCreateApplicationArea(0, 1, 0, 1);
            testStream.NfpCreateApplicationArea(0, 1, 1, 1);
            testStream.NfpFlush(0, 1);
            testStream.NfpRestore(0, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];
            Assert.AreEqual(6, analyzer.LogList.Count);

            for (int i = 1; i < 5; ++i)
            {
                Assert.AreEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[i].GetHashCode());
            }

            // Restore だけ異なる
            Assert.AreNotEqual(analyzer.LogList[0].GetHashCode(), analyzer.LogList[5].GetHashCode());
        }
    }

    [TestClass]
    public class FsLogAnalysisLibraryParserDeliveryCacheDirectoryTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestEnumerateDeliveryCacheDirectory()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            testStream.EnumerateDeliveryCacheDirectory(1, 1);
            testStream.UnmountDeliveryCacheStorage(1, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            Assert.AreEqual(3, analyzer.LogList.Count);
            Assert.IsTrue(analyzer.LogList[0].AccessType.HasFlag(FsApiAccessType.Read));
            Assert.IsTrue(analyzer.LogList[1].AccessType.HasFlag(FsApiAccessType.Read));

            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[2].MountTarget);
        }

        [TestMethod]
        public void TestDeliveryCacheDirectory()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            testStream.DeliveryCacheDirectoryOpen(0, 0, 1, "test");
            testStream.DeliveryCacheDirectoryRead(0, 0, 1);
            testStream.DeliveryCacheDirectoryClose(0, 0, 1);
            testStream.UnmountDeliveryCacheStorage(1, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            Assert.AreEqual(5, analyzer.LogList.Count);

            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[2].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[3].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[4].MountTarget);

            var hash = analyzer.LogList[0].MountHash;
            Assert.IsTrue(analyzer.LogList.All(log => log.MountHash == hash));
        }

        [TestMethod]
        public void TestDeliveryCacheDirectoryWithDestructor()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            testStream.DeliveryCacheDirectoryOpen(0, 0, 1, "test");
            testStream.DeliveryCacheDirectoryRead(0, 0, 1);
            testStream.DeliveryCacheDirectoryClose(0, 0, 1);
            testStream.DeliveryCacheDirectoryDestructor(0, 0, 1);
            testStream.UnmountDeliveryCacheStorage(1, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            Assert.AreEqual(6, analyzer.LogList.Count);

            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[2].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[3].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[4].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[5].MountTarget);

            var hash = analyzer.LogList[0].MountHash;
            Assert.IsTrue(analyzer.LogList.All(log => log.MountHash == hash));
        }

        [TestMethod]
        public void TestDeliveryCacheFile()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            testStream.DeliveryCacheFileOpen(0, 0, 1, "test/test.txt");
            testStream.DeliveryCacheFileRead(0, 0, 1, 0, 16);
            testStream.DeliveryCacheFileClose(0, 0, 1);
            testStream.UnmountDeliveryCacheStorage(1, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            Assert.AreEqual(5, analyzer.LogList.Count);

            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[2].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[3].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[4].MountTarget);

            var hash = analyzer.LogList[0].MountHash;
            Assert.IsTrue(analyzer.LogList.All(log => log.MountHash == hash));
        }

        [TestMethod]
        public void TestDeliveryCacheFileWithDestructor()
        {
            TestStream testStream = new TestStream();
            testStream.MountDeliveryCacheStorage(0, 0);
            testStream.DeliveryCacheFileOpen(0, 0, 1, "test/test.txt");
            testStream.DeliveryCacheFileRead(0, 0, 1, 0, 16);
            testStream.DeliveryCacheFileClose(0, 0, 1);
            testStream.DeliveryCacheFileDestructor(0, 0, 1);
            testStream.UnmountDeliveryCacheStorage(1, 1);
            testStream.SetUp();

            FsAccessLogParser parser = new FsAccessLogParser(testStream);
            FsAccessLogAnalyzer analyzer = parser.Parse()[0];

            Assert.AreEqual(6, analyzer.LogList.Count);

            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[0].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[1].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[2].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[3].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[4].MountTarget);
            Assert.AreEqual(FsMountTarget.Bcat, analyzer.LogList[5].MountTarget);

            var hash = analyzer.LogList[0].MountHash;
            Assert.IsTrue(analyzer.LogList.All(log => log.MountHash == hash));
        }
    }
}
