﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Input;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Input;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.EffectMaker.UIControls.TestTypes;

namespace Test.EffectMaker.UIControls.UnitTests
{
    /// <summary>
    /// Unit tests related to IExecutable and IExecutionSource implementations.
    /// </summary>
    [TestClass]
    public class ExecutableUnitTests
    {
        /// <summary>
        /// Test a executable with a button.
        /// </summary>
        [TestMethod]
        public void TestExecutable01()
        {
            var counter = 0;

            var button = new UIButton();
            button.Executable = new AnonymousExecutable(() => counter++);

            button.PerformClick();

            Assert.AreEqual(1, counter);

            button.Executable = null;

            button.PerformClick();

            Assert.AreEqual(1, counter);
        }

        /// <summary>
        /// Test the button availability update with an executable.
        /// </summary>
        [TestMethod]
        public void TestExecutable02()
        {
            var counter = 0;

            var button = new UIButton();
            var executable = new AnonymousExecutable(() => counter++);
            button.Executable = executable;

            Assert.IsTrue(button.Enabled);

            button.PerformClick();

            Assert.AreEqual(1, counter);

            executable.IsEnabled = false;

            Assert.IsFalse(button.Enabled);

            executable.IsEnabled = true;

            Assert.IsTrue(button.Enabled);
        }

        /// <summary>
        /// Test the button availability update with an executable.
        /// </summary>
        [TestMethod]
        public void TestExecutable03()
        {
            var counter = 0;

            var button = new UIButton();
            button.Executable = new AnonymousExecutable(
                p => counter++,
                p =>
                {
                    if ((p is bool) == false)
                    {
                        return true;
                    }

                    return (bool)p;
                });

            button.PerformClick();

            Assert.AreEqual(1, counter);

            button.ExecutionParameter = DateTime.Now;

            button.PerformClick();

            Assert.AreEqual(2, counter);

            button.ExecutionParameter = true;

            button.PerformClick();

            Assert.AreEqual(3, counter);

            button.ExecutionParameter = false;

            button.PerformClick();

            Assert.AreEqual(3, counter);
        }

        /// <summary>
        /// Test button with executable through binding.
        /// </summary>
        [TestMethod]
        public void TestExecutable04()
        {
            var button = new UIButton();

            var binder1 = new Binder(button, "Executable", "IncrementCounterExecutable")
            {
                Mode = BindingMode.OneTime,
            };
            var binder2 = new Binder(button, "Text", "Counter");

            button.DataContext = new TestExecutableViewModel();

            Assert.AreEqual("0", button.Text);

            button.PerformClick();

            Assert.AreEqual("1", button.Text);
        }
    }
}
