﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/types.h>

namespace nw      {
namespace eftdemo {


//---------------------------------------------------------------------------
//  DisplayList クラス
//---------------------------------------------------------------------------
class DisplayList
{
public:
    // コンストラクタ
    DisplayList()
    {
        mDispListBuffer     = NULL;
        mDispListSize       = 0;
        mDispListUsedSize   = 0;
        mAllocator          = NULL;
    }

    //-------------------------------------------
    // デストラクタ
    //-------------------------------------------
    ~DisplayList()
    {
        Finalize( mAllocator );
    }

    //-------------------------------------------
    // ディスプレイリストの初期化
    //-------------------------------------------
    void Initialize( nw::ut::IAllocator* allocator, u32 dlBufferSize ) volatile
    {
#if defined(NW_PLATFORM_CAFE)
        mDispListBuffer = allocator->Alloc( dlBufferSize, GX2_DISPLAY_LIST_ALIGNMENT );
        GX2Invalidate( GX2_INVALIDATE_CPU, mDispListBuffer, mDispListSize );
#endif
        mDispListSize   = dlBufferSize;
        mAllocator      = allocator;
    }

    //-------------------------------------------
    // ディスプレイリストの初期化
    //-------------------------------------------
    void Finalize( nw::ut::IAllocator* allocator ) volatile
    {
#if defined(NW_PLATFORM_CAFE)
        if ( mDispListBuffer )
        {
            allocator->Free( mDispListBuffer );
            mDispListBuffer = NULL;
        }
#endif
    }

    //-------------------------------------------
    // ディスプレイリストを有効にする
    //-------------------------------------------
    void BindDisplayList( bool invalidate = true ) volatile
    {
#if defined(NW_PLATFORM_CAFE)
        if ( invalidate )
        {
            GX2Invalidate( GX2_INVALIDATE_CPU, mDispListBuffer, mDispListSize
        );
        GX2BeginDisplayList( mDispListBuffer, mDispListSize );
#endif
    }

    //-------------------------------------------
    // ディスプレイリストを無効にする
    //-------------------------------------------
    u32 UnbindDisplayList() volatile
    {
#if defined(NW_PLATFORM_CAFE)
        mDispListUsedSize = GX2EndDisplayList( mDispListBuffer );
        return mDispListUsedSize;
#else
        return 0;
#endif
    }

    //-------------------------------------------
    // ディスプレイリストをコールする
    //-------------------------------------------
    void CallDisplayList() volatile
    {
        if ( mDispListUsedSize == 0 )
        {
            return;
        }

#if defined(NW_PLATFORM_CAFE)
        // エフェクトランタイム内部でディスプレイリストを呼び出す為、
        // GX2DirectCallDisplayListを利用する。
        GX2DirectCallDisplayList( mDispListBuffer, mDispListUsedSize );
#endif
    }


    //-------------------------------------------
    // ディスプレイリストの内容をダンプする
    //-------------------------------------------
    void Dump() volatile
    {
#if !defined( NW_RELEASE )
        if ( mDispListUsedSize == 0 )
        {
            return;
        }

        u32* ptr = (u32*)mDispListBuffer;

        for( u32 i = 0; i < mDispListUsedSize / 32; i++ )
        {
            u32 idx = 8 * i;
            NW_LOG( "[%08P] %08X %08X %08X %08X %08X %08X %08X %08X\n", &ptr[idx],
                ptr[idx], ptr[idx + 1], ptr[idx + 2], ptr[idx + 3], ptr[idx + 4], ptr[idx + 5], ptr[idx + 6], ptr[idx + 7] );
        }
#endif
    }

    //-------------------------------------------
    // ディスプレイリストの内容を取得する
    //-------------------------------------------
    u32*    GetDisplayListBuffer() const
    {
        return (u32*)mDispListBuffer;
    }


    //-------------------------------------------
    // ディスプレイリストのサイズを取得する
    //-------------------------------------------
    u32     GetDisplayListBufferSize() const
    {
        return mDispListUsedSize;
    }

private:
    void*               mDispListBuffer;
    u32                 mDispListSize;
    u32                 mDispListUsedSize;
    nw::ut::IAllocator* mAllocator;
};


} // namespace eftdemo
} // namespace nw
