﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>
//#include <nw/effect.h>
//#include <ut/eftut_Heap.h>

#include <nw/eft/eft2_System.h>
#include <nw/eft/eftvw2_ViewerSystem.h>
#include <nw/eft/eftut2_Heap.h>

#include <eftdemo_Camera.h>
#include <eftdemo_Layer.h>
#include <eftdemo_System.h>
#include <eftdemo_Texture.h>
#include <eftdemo_Menu.h>
#include <eftdemo_ModelUtility.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/hio.h>
#if defined(NW_USE_CPU_PROFILER)
#include <cafe/profiler.h>
#endif
#endif

//------------------------------------------------------------------------------
//      Effect デモクラス
//------------------------------------------------------------------------------
class EffectDemo
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //---------------------------------------------------------------------------
    EffectDemo()
    {
        mEftSystem                  = NULL;
        mHandle.Invalidate();
        mEmitterSetIndex            = 0;
        mEffectBinaryResource       = NULL;
        mVwrCamera                  = NULL;
        mNwAllocator                = NULL;
        mEnableDisplayList          = true;
        mDisplayWindow              = false;
        mDebugMenuPosX              = 32.f;
        mDebugMenuPosY              = 32.f;
        mFrameRate                  = 60.f;
        mBackgroundColor.Set( 0.0f, 0.0f, 0.0f, 0.0f );
        mDebugMenu                  = NULL;        // Initializeで設定されます。
        mBackgroundTextureRes       = NULL;
        mPlaySpeed                  = 1.0f;
        mGridType                   = 0;
        mIsLinearMode               = true;

        InitDemoValue();
    };

    //---------------------------------------------------------------------------
    //! @brief        デモごとの変数設定
    //---------------------------------------------------------------------------
    void InitDemoValue();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    ~EffectDemo() {};

    //char mArgs[256];

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの初期化処理
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator, nw::eftdemo::DebugMenu* debugMenu );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの終了処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモの計算処理
    //---------------------------------------------------------------------------
    void ProcCalc( nw::dev::Pad* pad, nw::eftdemo::DrawParam& drawParam  );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトの描画処理
    //---------------------------------------------------------------------------
    void ProcDraw( nw::eftdemo::DrawParam& drawParam, u32 drawPathFlag = 0 );

    //---------------------------------------------------------------------------
    //! @brief        モデルの描画処理
    //---------------------------------------------------------------------------
    void ProcModelDraw( nw::eftdemo::DrawParam& drawParam, nw::eftdemo::SimpleShader::ViewId id = nw::eftdemo::SimpleShader::VIEW_ID_CAMERA );

    //---------------------------------------------------------------------------
    //! @brief        エフェクトデモのデバッグ文字描画
    //---------------------------------------------------------------------------
    void DebugDraw( nw::dev::DevTextWriter* writer );

    //---------------------------------------------------------------------------
    //! @brief        Heapを取得
    //---------------------------------------------------------------------------
    nw::eftut2::NwHeap* GetHeap()
    {
        return &mEftHeap;
    }

    //---------------------------------------------------------------------------
    //! @brief        カメラをセットする
    //---------------------------------------------------------------------------
    void SetCamera( nw::eftdemo::Camera* camera )
    {
        mVwrCamera = camera;
    }

    //---------------------------------------------------------------------------
    //! @brief        プロジェクションマトリクスをセットする
    //---------------------------------------------------------------------------
    void SetProjectionMatrix( nw::math::MTX44* projMatrix )
    {
        mProjectionMatrix = projMatrix;
    }

#ifdef NW_PLATFORM_CAFE
    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetDepthBufferTexture( const GX2Texture* depthBufferTexture )
    {
        mEftSystem->SetDepthTexture( const_cast<GX2Texture*>( depthBufferTexture ) );
    }
    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetFrameBufferTexture( const GX2Texture* frameBufferTexture )
    {
        mEftSystem->SetFrameBufferTexture( const_cast<GX2Texture*>( frameBufferTexture ) );
    }
    //---------------------------------------------------------------------------
    //! @brief        各種テクスチャをセットする
    //---------------------------------------------------------------------------
//  void SetTextureSlot( nw::eft2::TextureSlot slot, const GX2Texture* texture )
//  {
//      mEftSystem->SetTextureSlot( slot, texture );
//  }
#endif
#ifdef NW_PLATFORM_WIN32
    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetDepthBufferTexture( GLuint depthBufferTexture )
    {
        mEftSystem->SetDepthTexture( depthBufferTexture );
    }
    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャをセットする
    //---------------------------------------------------------------------------
    void SetFrameBufferTexture( GLuint frameBufferTexture )
    {
        mEftSystem->SetFrameBufferTexture( frameBufferTexture );
    }
    //---------------------------------------------------------------------------
    //! @brief        各種テクスチャをセットする
    //---------------------------------------------------------------------------
//  void SetTextureSlot( nw::eft::TextureSlot slot, GLuint texture )
//  {
//      mEftSystem->SetTextureSlot( slot, texture );
//  }
#endif

    //---------------------------------------------------------------------------
    //! @brief        デバッグメニューの表示位置(X)。
    //---------------------------------------------------------------------------
    f32 GetDebugMenuPosX() const
    {
        return mDebugMenuPosX;
    }

    //---------------------------------------------------------------------------
    //! @brief        デバッグメニューの表示位置(X)。
    //---------------------------------------------------------------------------
    f32 GetDebugMenuPosY() const
    {
        return mDebugMenuPosY;
    }

    //---------------------------------------------------------------------------
    //! @brief        フレームレートを設定します。
    //---------------------------------------------------------------------------
    void SetFrameRate( nw::eftdemo::INIT_SETTINGS_FRAME_RATE frameRate )
    {
        switch( frameRate )
        {
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_60:
            mFrameRate = 60.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_30:
            mFrameRate = 30.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_20:
            mFrameRate = 20.f;
            break;
        case nw::eftdemo::INIT_SETTINGS_FRAME_RATE_15:
            mFrameRate = 15.f;
            break;
        default:
            mFrameRate = 60.f;
            break;
        }
    }

    void SetFrameRate( f32 frameRate )
    {
        mFrameRate = frameRate;
    }

    //---------------------------------------------------------------------------
    //! @brief        フレームレートを取得します。
    //---------------------------------------------------------------------------
    void GetFrameRate( f32* frameRate ) const
    {
        *frameRate = mFrameRate;
    }


    //---------------------------------------------------------------------------
    //! @brief        描画すべきエミッタが存在するか。
    //---------------------------------------------------------------------------
    bool IsHasRenderingEmitter( u8 groupID, u32 drawPath )
    {
        return mEftSystem->IsHasRenderingEmitter( groupID, drawPath );
    }

    //---------------------------------------------------------------------------
    //! @brief        シェーダタイプを切り替える。
    //---------------------------------------------------------------------------
    void SetShaderType( nw::eft2::ShaderType type )
    {
        return mEftSystem->SetShaderType( type );
    }

    //---------------------------------------------------------------------------
    //! @brief        背景カラーを設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundColor( nw::math::VEC4 color )
    {
        mBackgroundColor = color;
    }

    //---------------------------------------------------------------------------
    //! @brief        カメラを設定します。
    //---------------------------------------------------------------------------
    void SetCameraSettings(
        float posX, float posY, float posZ,
        float lookAtX, float lookAtY, float lookAtZ,
        float rotation,
        int projType,
        float aspect,
        float fov,
        float znear, float zfar );

    //---------------------------------------------------------------------------
    //! @brief        背景カラーを設定します。
    //---------------------------------------------------------------------------
    const nw::math::VEC4 GetBackgroundColor() const
    {
        return mBackgroundColor;
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundImage( u32 width, u32 height, const u8* pixels )
    {
        mBackgroundTexture.SetImage( mNwAllocator, width, height, pixels );
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundImageRes( const void* resource )
    {
        mBackgroundTextureRes = resource;
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像を取得します。
    //---------------------------------------------------------------------------
    nw::eftdemo::Texture& GetBackgroundTexture()
    {
        return mBackgroundTexture;
    }

    //---------------------------------------------------------------------------
    //! @brief        背景画像リソースを取得します。
    //---------------------------------------------------------------------------
    const void* GetBackgroundTextureRes()
    {
        return mBackgroundTextureRes;
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトシステムを取得します。
    //---------------------------------------------------------------------------
    nw::eft2::System* GetEffectSystem()
    {
        return mEftSystem;
    }

    //---------------------------------------------------------------------------
    //! @brief        エフェクトビューアシステムを取得します。
    //---------------------------------------------------------------------------
    nw::eftvw2::ViewerSystem* GetEffectViewerSystem()
    {
        return mEftVwrSystem;
    }


    //---------------------------------------------------------------------------
    //! @brief        ディスプレイリスト設定の有効/無効
    //---------------------------------------------------------------------------
    bool EnabledDisplayList()
    {
        return mEnableDisplayList;
    }

    //---------------------------------------------------------------------------
    //! @brief        ウィンドウの有効/無効
    //---------------------------------------------------------------------------
    bool DisplayWindow()
    {
        return mDisplayWindow;
    }

    //---------------------------------------------------------------------------
    //! @brief        再生速度を設定します。
    //!
    //! @param[in]    playSpeed 再生速度。
    //---------------------------------------------------------------------------
    void SetPlaySpeed( float playSpeed)
    {
        mPlaySpeed = playSpeed;
    }

    //---------------------------------------------------------------------------
    //! @brief        再生速度を取得します。
    //!
    //! @return       再生速度を返します。
    //---------------------------------------------------------------------------
    float GetPlaySpeed()
    {
        return mPlaySpeed;
    }

    //---------------------------------------------------------------------------
    //! @brief        グリッドタイプを設定します。
    //!
    //! @param[in]    gridType グリッドタイプ。
    //---------------------------------------------------------------------------
    void SetGridType( int gridType )
    {
        mGridType = gridType;
    }

    //---------------------------------------------------------------------------
    //! @brief        グリッドタイプを取得します。
    //!
    //! @return       グリッドタイプを返します。
    //---------------------------------------------------------------------------
    int GetGridType()
    {
        return mGridType;
    }

    //---------------------------------------------------------------------------
    //! @brief        リニアモードの有効/無効を設定します。
    //!
    //! @param[in]    isLinearMode リニアモードの有効/無効。
    //---------------------------------------------------------------------------
    void SetLinearMode( bool isLinearMode )
    {
        mIsLinearMode = isLinearMode;
    }


    //---------------------------------------------------------------------------
    //! @brief        リニアモードの有効/無効を取得します。
    //!
    //! @return       リニアモードの有効/無効を返します。
    //---------------------------------------------------------------------------
    bool GetLinearMode()
    {
        return mIsLinearMode;
    }

private:
    //---------------------------------------------------------------------------
    //! @brief        ファイルをロードします
    //---------------------------------------------------------------------------
    void* LoadBinary( const char* fileName, s32 alignment )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path      = fileName;
        loadArg.allocator = mNwAllocator;
        loadArg.alignment = alignment;
        u8* pPtcl = fileSystem->Load( loadArg );

        return pPtcl;
    }

    //---------------------------------------------------------------------------
    //! @brief        ロードされたファイルを破棄します。
    //---------------------------------------------------------------------------
    void UnLoadBinary( const char* fileName, void* binary, s32 alignment )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path      = fileName;
        loadArg.allocator = mNwAllocator;
        loadArg.alignment = alignment;

        fileSystem->Unload( loadArg, static_cast<u8*>( binary ) );
    }


private:
    nw::eftut2::NwHeap                  mEftHeap;                       //!< エフェクトヒープクラス
    nw::eftut2::NwHeap                  mEftDynamicHeap;                //!< エフェクトダイナミックヒープクラス
    nw::eft2::System*                   mEftSystem;                     //!< エフェクトシステム
    nw::eftvw2::ViewerSystem*           mEftVwrSystem;                  //!< エフェクトビューアシステム
    nw::eft2::Handle                    mHandle;                        //!< エフェクトハンドル
    s32                                 mEmitterSetIndex;               //!< エミッタセットID
    void*                               mEffectBinaryResource;          //!< エフェクトバイナリリソース
    f32                                 mFrameRate;                     //!< フレームレート
    nw::math::VEC4                      mBackgroundColor;               //!< 背景カラー
    nw::eftdemo::BackgroundTexture      mBackgroundTexture;             //!< 背景テクスチャ
    const void*                               mBackgroundTextureRes;          //!< 背景テクスチャリソース
    nw::eftdemo::Camera*                mVwrCamera;                     //!< カメラ
    nw::math::MTX44*                    mProjectionMatrix;              //!< プロジェクション
    nw::ut::MemoryAllocator*            mNwAllocator;                   //!< アロケータ―
    bool                                mEnableDisplayList;             //!< ディスプレイリスト設定の有効/無効
    f32                                 mDebugMenuPosX;                 //!<
    f32                                 mDebugMenuPosY;                 //!<
    nw::eftdemo::DebugMenu*             mDebugMenu;                     //!<
    nw::eftdemo::DebugMenuPage          mDebugMenuPage;                 //!<
    bool                                mDisplayWindow;                 //!< ログウインドウの表示/非表示
    float                               mPlaySpeed;                     //!< 再生速度
    int                                 mGridType;                      //!< グリッドタイプ
    bool                                mIsLinearMode;                  //!< リニアモードの有効/無効
};
