﻿using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestGameCard : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key mount_gamecard_result_value 0x0", // fs::ResultGameCardCardAccessFailure
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key try_gamecard_access_result_value 0x0", // fs::ResultGameCardCardAccessFailure
                "application uninstall --all --ignore-running-check",
                "gamecard erase",
                "application uninstall --all",
                "application reset-required-version --all",
            });
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void AccessFailedOnMount()
        {
            var appv0 = m_Maker.MakeApplication(0x0100394000059000, 0);
            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard write " + appv0.Path
            }));
            // gamecard write で一旦マウントに失敗してしまうので、再起動する
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": true",
                "\\\"lastMountFailureResult\\\": \\\"0x00000000\\\""
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key mount_gamecard_result_value 0x0013c402", // fs::ResultGameCardCardAccessFailure
            }));
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": false",
                "\\\"lastMountFailureResult\\\": \\\"0x0013c402\\\""
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key mount_gamecard_result_value 0x0", // ResultSuccess
            }));
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": true",
                "\\\"lastMountFailureResult\\\": \\\"0x00000000\\\""
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void AccessFailedOnTryAccess()
        {
            var appv0 = m_Maker.MakeApplication(0x0100394000059000, 0);
            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard write " + appv0.Path
            }));
            // gamecard write で一旦マウントに失敗してしまうので、再起動する
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": true",
                "\\\"lastMountFailureResult\\\": \\\"0x00000000\\\""
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard ensure-access",
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key try_gamecard_access_result_value 0x0013c402", // fs::ResultGameCardCardAccessFailure
            }));
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": true",
                "\\\"lastMountFailureResult\\\": \\\"0x00000000\\\""
            }));
            Assert.IsTrue(command.FailureExecute("gamecard ensure-access", "Unexpected error. result = 0x0013c402"));
            Assert.IsTrue(command.Execute(new string[] {
                "debug set-hexadecimal-fwdbg --name ns.gamecard --key try_gamecard_access_result_value 0x0", // ResultSuccess
            }));
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard mount-status",
            }, new string[]
            {
                "\\\"isAttached\\\": true",
                "\\\"lastMountFailureResult\\\": \\\"0x00000000\\\""
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard ensure-access",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestDeleteApplicationCompletelyOnGameCard()
        {
            var app0 = m_Maker.MakeApplication(0x0100394000059000, 0);
            var app1 = m_Maker.MakeApplication(0x010039400005a000, 0);
            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard write " + app0.Path,
                "application install " + app1.Path
            }));

            Assert.IsTrue(command.FailureExecute(
                string.Format("application uninstall 0x{0}", app0.Id),
                expect: string.Format("Failed to execute the command. Game Card is inserted.")
            ));

            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application uninstall 0x{0}", app1.Id),
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "gamecard write " + app1.Path,
            }));

            // アプリケーション記録は GameCard で、ゲームカードは刺さっているけど違うゲームカードのアプリを消せるか
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application uninstall 0x{0}", app0.Id),
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestAutoBootApplication()
        {
            var app0 = m_Maker.MakeApplication(0x0100394000059000, 0);
            var patchv1 = m_Maker.MakePatch(0x0100394000059000, 1, app0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("gamecard write {0} --on-card-patch {1} --auto", app0.Path, patchv1.Path),
            }));

            Reboot(command);

            // アプリが起動するのを待つ
            Thread.Sleep(5 * 1000);

            Assert.IsTrue(command.Execute(new string[] {
                "gamecard mount-status"
            }));
            var status = GameCardMountStatusForJson.Deserialize(command.LastOutput);
            Assert.AreEqual(0, status.sequence);
            Assert.AreEqual(0u, status.lastMountFailureResult);

            Assert.IsTrue(command.Execute(new string[] {
                "gamecard erase",
            }));
            Reboot(command);
        }

        public void MakePathListFile(string filename, IEnumerable<string> appList, IEnumerable<string> patchList)
        {
            using (var fs = File.CreateText(filename))
            {
                foreach (var path in appList)
                {
                    fs.WriteLine(path);
                }
                foreach (var path in patchList)
                {
                    fs.WriteLine(path);
                }
            }
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(1200 * 1000)]
        public void TestMultiApplicationCard()
        {
            UInt64 id = 0x0100abcd00000000;
            UInt64 cardId = 0x0100abcdf0000000;
            string cardIdString = string.Format("0x{0:x16}", cardId);
            var appList = new List<NspInfo>();
            var patchList = new List<NspInfo>();

            Func<int, UInt64> makeId = (int index) => { return id + 0x1000 * (UInt64)index; };

            // TORIAEZU: パスが長すぎてテストが実行できないので、7 にしないでおく
            //           下のテストで利用する関係上最大 7 にしておく
            const int MaxApplicationCountForLongTest = 3;
            const int MaxApplicationCountForNormalTest = 3;
            int maxApplicationCount = IsLongTest ? MaxApplicationCountForLongTest : MaxApplicationCountForNormalTest;
            for (int i = 0; i < maxApplicationCount; i++)
            {
                appList.Add(m_Maker.MakeApplication(makeId(i), 0));
                patchList.Add(m_Maker.MakePatch(makeId(i), 1, appList[i].Path));
            }

            appList = appList.OrderBy(i => Guid.NewGuid()).ToList();
            patchList = patchList.OrderBy(i => Guid.NewGuid()).ToList();

            var multiAppPath = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "mac.nsp");

            var appPathList = appList.Select(x => x.Path.Replace("\"", "")).ToList();
            var patchPathList = patchList.Select(x => x.Path.Replace("\"", "")).ToList();
            var pathListFilePath = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "pathlist.txt");

            MakePathListFile(pathListFilePath, appPathList, patchPathList);
            AuthoringTool authoringTool = new AuthoringTool(this.TestContext);
            var executeResult = authoringTool.Execute(string.Format("bundleup --id {0} -o {1} --nsp-list-file {2}", cardIdString, multiAppPath, pathListFilePath));
            Assert.IsTrue(executeResult.Item1);

            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, pathListFilePath),
            }));

            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application list-view")
            }));
            var viewList = ApplicationViewForJson.Deserialize(command.LastOutput);
            Assert.AreEqual(maxApplicationCount, viewList.Count);

            // 書き込んだ順に並んでいる
            for (int i = 0; i < maxApplicationCount; i++)
            {
                var view = viewList[i];
                var idString = string.Format("0x{0}", appList[i].Id);

                Assert.AreEqual(idString.ToLower(), view.id);
                Assert.IsTrue(view.isGameCard);
                Assert.IsTrue(view.hasGameCardEntity);
                Assert.IsTrue(view.isLaunchable);
                Assert.IsTrue(view.hasPatchEntity);

                // 起動チェック
                SuccessLaunchApplication(command, Convert.ToUInt64(idString, 16));

                // 整理は出来る
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application delete-entity {0}", idString)
                }));

                // 削除は出来ない
                Assert.IsTrue(command.FailureExecute(
                    string.Format("application uninstall {0}", idString),
                    expect: "Game Card is inserted."
                ));
            }

            // MultiApplicationCardIndicator.nsp で指定した順番と違う順番には書き込めない
            {
                string reorderedPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "reorderedPathList.txt");
                var reorderedList = appPathList;
                var tmp = reorderedList[0];
                reorderedList[0] = reorderedList[1];
                reorderedList[1] = tmp;
                MakePathListFile(reorderedPathList, reorderedList, patchPathList);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, reorderedPathList)
                ));
            }

            // コンテンツが足りないと失敗する
            {
                var missingAppList = appPathList;
                missingAppList.RemoveAt(0);
                string missingAppPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "missingAppPathList.txt");
                MakePathListFile(missingAppPathList, missingAppList, patchPathList);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, missingAppPathList)
                ));

                var missingPatchList = patchPathList;
                missingPatchList.RemoveAt(0);
                string missingPatchPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "missingPatchPathList.txt");
                MakePathListFile(missingPatchPathList, appPathList, missingPatchList);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, missingPatchPathList)
                ));
            }

            // 不要なコンテンツが混じっていると失敗する
            {
                var additionalApp = m_Maker.MakeApplication(makeId(maxApplicationCount + 1), 0);
                var additionalAppList = appPathList;
                additionalAppList.Add(additionalApp.Path);
                string additionalAppPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "additionalAppPathList.txt");
                MakePathListFile(additionalAppPathList, additionalAppList, patchPathList);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, additionalAppPathList)
                ));

                var additionalPatch = m_Maker.MakePatch(makeId(maxApplicationCount + 1), 1, additionalApp.Path);
                var additionalPatchList = patchPathList;
                additionalPatchList.Add(additionalPatch.Path);
                string additionalPatchPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "additionalPatchPathList.txt");
                MakePathListFile(additionalPatchPathList, additionalPatchList, patchPathList);

                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, additionalPatchPathList)
                ));
            }

            // 重複するコンテンツが混じっていると失敗する
            {
                var duplicatedAppList = appPathList;
                duplicatedAppList[1] = duplicatedAppList[0];
                string duplicatedAppPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "duplicatedAppPathList.txt");
                MakePathListFile(duplicatedAppPathList, duplicatedAppList, patchPathList);


                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-application-card {0} --nsp-list-file {1}", multiAppPath, duplicatedAppPathList)
                ));

                var duplicatedPatchList = patchPathList;
                duplicatedPatchList[1] = duplicatedPatchList[0];
                string duplicatedPatchPathList = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "duplicatedPatchPathList.txt");
                MakePathListFile(duplicatedPatchPathList, duplicatedPatchList, patchPathList);
                Assert.IsTrue(command.FailureExecute(
                    string.Format("gamecard write --multi-patchlication-card {0} --nsp-list-file {1}", multiAppPath, duplicatedPatchPathList)
                ));
            }
        }
    }
}
