﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Script.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;


namespace DevMenuCommandTest
{
    [TestClass]
    public class SystemUpdateTaskTest : TestBase
    {
        private const ulong TestAppId = 0x0100ccccaaaa0000;
        private const string TestAppIdString = "0x0100ccccaaaa0000";

        private static SystemProgramList LocalSystemUpdateMeta;
        private static SystemProgramList NetworkSystemUpdateMeta;

        private static List<NspInfo> AppList;
        private static NspInfo SystemUpdateV1;
        private static NspInfo OldSystemUpdate;
        private static NspInfo CurrentSystemUpdate;
        private static NspInfo NewSystemUpdate;
        private static NspInfo SystemUpdateForRecovery;

        private static bool IsHostNetworkEnabled;
        private static bool IsClientNetworkEnabled;
        private static bool IsNetworkEnabled;

        private enum TestVersion
        {
            ReceiverVersion = 0,
            OldVersion = 1,
            CurrentVersion = 2,
            NewVersion = 3,
        }

        private static void DownloadLatestSystemUpdate(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("systemprogram uninstall {0}", LocalSystemUpdateMeta.id)
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "systemupdate check-latest"
            }));
            Assert.IsFalse(command.LastOutput.Contains("UpToDate"));
            if (command.LastOutput.Contains("NeedsDownload"))
            {
                Assert.IsTrue(command.Execute(new string[]
                {
                    "systemupdate download-latest"
                }));
            }

            Assert.IsTrue(command.Execute(new string[]
            {
                "systemupdate apply-downloaded"
            }));

            ForceReset(command);
        }

        private static void RecoverSystemUpdate(DevMenuCommandSystem command)
        {
            command.Execute(new string[]
            {
                string.Format("systemprogram uninstall {0}", LocalSystemUpdateMeta.id),
                string.Format("systemprogram install {0}", SystemUpdateForRecovery.Path)
            });
            ForceReset(command);
        }

        [ClassInitialize]
        public static void ClassInitialize(TestContext context)
        {
            SetupTestEnvironment(context);

            var hostTask = Task.Run(() =>
            {
                IsHostNetworkEnabled = VerifyNetworkEnvironment(HostCommand);
            });

            var clientTask = Task.Run(() =>
            {
                IsClientNetworkEnabled = VerifyNetworkEnvironment(ClientCommand);
            });

            hostTask.Wait();
            clientTask.Wait();

            IsNetworkEnabled = IsHostNetworkEnabled && IsClientNetworkEnabled;

            if (!IsNetworkEnabled)
            {
                return;
            }

            CleanupBase(context);

            // Client 側の一旦 SystemUpdate をアンインストールして、最新のシステムを落とす
            {
                var command = ClientCommand;

                var systemProgramList = GetSystemProgramList(command);
                // 一個しかないはず
                LocalSystemUpdateMeta = systemProgramList.Single(x => x.type == "SystemUpdate");

                DownloadLatestSystemUpdate(command);

                systemProgramList = GetSystemProgramList(command);
                NetworkSystemUpdateMeta = systemProgramList.Single(x => x.type == "SystemUpdate");
            }

            var systemUpdateMetaId = Convert.ToUInt64(NetworkSystemUpdateMeta.id, 16);
            SystemUpdateV1 = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, 1, new MakeTestSystemContent.ContentMetaInfo[] { });
            OldSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, NetworkSystemUpdateMeta.version - 1, new MakeTestSystemContent.ContentMetaInfo[] { });
            CurrentSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, NetworkSystemUpdateMeta.version, new MakeTestSystemContent.ContentMetaInfo[] { });
            NewSystemUpdate = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, NetworkSystemUpdateMeta.version + 1, new MakeTestSystemContent.ContentMetaInfo[] { });
            SystemUpdateForRecovery = SysMaker.MakeTestSystemUpdateMeta(systemUpdateMetaId, LocalSystemUpdateMeta.version, new MakeTestSystemContent.ContentMetaInfo[] { });

            AppList = new List<NspInfo>();

            AppList.Add(AppMaker.MakeApplication(TestAppId, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 1, AppList[0].Path, requiredSystemVersion: NetworkSystemUpdateMeta.version - 1, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 2, AppList[0].Path, requiredSystemVersion: NetworkSystemUpdateMeta.version, additionalOptions: "--small-code"));
            AppList.Add(AppMaker.MakePatch(TestAppId, 3, AppList[0].Path, requiredSystemVersion: NetworkSystemUpdateMeta.version + 1, additionalOptions: "--small-code"));
        }

        [TestCleanup]
        public void TestCleanup()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var cleanupCommand = new string[]
            {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            };

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                command.Execute(cleanupCommand, ignoreError: true);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [ClassCleanup]
        public static void TestClassCleanup()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }

            var hostTask = Task.Run(() =>
            {
                RecoverSystemUpdate(HostCommand);
            });

            var clientTask = Task.Run(() =>
            {
                RecoverSystemUpdate(ClientCommand);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        private void InstallTestApplication(DevMenuCommandSystem command, int version, string appStorage = null, string patchStorage = null)
        {
            InstallTestApplicationImpl(command, version, AppList, appStorage, patchStorage);
        }

        private void InstallTestSystem(DevMenuCommandSystem command, NspInfo nsp)
        {
            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("systemprogram uninstall 0x{0}", nsp.Id),
                string.Format("systemprogram install {0}", nsp.Path)
            }));
            ForceReset(command);
        }

        private NspInfo GetNspInfo(TestVersion version)
        {
            switch(version)
            {
            case TestVersion.ReceiverVersion: return SystemUpdateV1;
            case TestVersion.OldVersion: return OldSystemUpdate;
            case TestVersion.CurrentVersion: return CurrentSystemUpdate;
            case TestVersion.NewVersion: return NewSystemUpdate;
            }
            return null;
        }

        private string GetExpectedLatestSystemUpdateState(TestVersion version)
        {
            switch(version)
            {
            case TestVersion.OldVersion: return "NeedsDownload";
            case TestVersion.CurrentVersion: return "UpToDate";
            case TestVersion.NewVersion: return "UpToDate";
            }
            return null;
        }

        private void SendTest(DevMenuCommandSystem command, Barrier barrier, TestVersion version)
        {
            InstallTestSystem(command, GetNspInfo(version));
            InstallTestApplication(command, (int)version);

            barrier.SignalAndWait();

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", TestAppId)
            }));
            {
                var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, output.state);
            }
        }

        private void ReceiveTest(DevMenuCommandSystem command, Barrier barrier, TestVersion version)
        {
            InstallTestSystem(command, GetNspInfo(TestVersion.ReceiverVersion));
            InstallTestApplication(command, (int)TestVersion.ReceiverVersion);

            var resumeFile = Path.Combine(SysMaker.OutputDirPath, "resume.txt");
            if (File.Exists(resumeFile))
            {
                File.Delete(resumeFile);
            }

            barrier.SignalAndWait();

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("localcontentshare receive-application {0} --output-resume-file \"{1}\" --timeout 300", TestAppIdString, resumeFile)
            }));
            {
                var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Suspened, output.state);
                Assert.AreEqual((UInt32)LcsSuspendedReason.RebootRequired, output.reason);
                Assert.IsTrue(!string.IsNullOrEmpty(output.context));
                Assert.IsTrue(File.Exists(resumeFile));
            }

            var systemProgramList = GetSystemProgramList(command);
            Assert.IsTrue(systemProgramList.Any(
                x => x.id.ToLower() == NetworkSystemUpdateMeta.id &&
                x.version == (NetworkSystemUpdateMeta.version - ((int)TestVersion.CurrentVersion - (int)version))
                ));

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                string.Format("localcontentshare receive-application {0} --resume \"{1}\" --timeout 300", TestAppIdString, resumeFile)
            }));
            {
                var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                Assert.AreEqual((UInt32)LcsState.Completed, output.state);
            }

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list-view"
            }));
            var view = ApplicationViewForJson.Deserialize(command.LastOutput);
            Assert.AreEqual(1, view.Count);
            Assert.AreEqual((UInt32)version << 16, view[0].version);

            Assert.IsTrue(command.Execute(new string[]
            {
                "systemupdate check-latest"
            },
                GetExpectedLatestSystemUpdateState(version)
            ));
        }

        private void TestImpl(TestVersion version)
        {
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                SendTest(HostCommand, barrier, version);
            });

            var clientTask = Task.Run(() =>
            {
                ReceiveTest(ClientCommand, barrier, version);
            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000), Ignore]
        public void SendSameSystemWithNetwork()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }
            TestImpl(TestVersion.CurrentVersion);
        }

        [TestMethod]
        [Timeout(600 * 1000), Ignore]
        public void SendOlderSystemThanNetwork()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }
            TestImpl(TestVersion.OldVersion);
        }

        [TestMethod]
        [Timeout(600 * 1000), Ignore]
        public void SendNewerSystemThanNetwork()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }
            TestImpl(TestVersion.NewVersion);
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void ReceivePatchWithNetworkSystemUpdateTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                InstallTestApplication(command, (int)TestVersion.OldVersion);

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", TestAppId)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }
            });

            var clientTask = Task.Run(() =>
            {
                var command = ClientCommand;
                InstallTestSystem(command, GetNspInfo(TestVersion.OldVersion));
                InstallTestApplication(command, (int)TestVersion.ReceiverVersion);

                while (true)
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "systemupdate get-background-state"
                    }));
                    if (command.LastOutput.Contains("InProgress") || command.LastOutput.Contains("Ready"))
                    {
                        break;
                    }
                }

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare receive-application 0x{0:X16} --timeout 300", TestAppId)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "systemupdate get-background-state"
                }));
                Assert.IsTrue(command.LastOutput.Contains("InProgress") || command.LastOutput.Contains("Ready"));

            });

            hostTask.Wait();
            clientTask.Wait();
        }

        [TestMethod]
        [Timeout(600 * 1000), Ignore]
        public void SendPatchWithNetworkSystemUpdateTask()
        {
            if (!IsNetworkEnabled)
            {
                return;
            }
            var barrier = new Barrier(2);

            var hostTask = Task.Run(() =>
            {
                var command = HostCommand;
                InstallTestSystem(command, GetNspInfo(TestVersion.OldVersion));
                InstallTestApplication(command, (int)TestVersion.OldVersion);

                while (true)
                {
                    Assert.IsTrue(command.Execute(new string[]
                    {
                        "systemupdate get-background-state"
                    }));
                    if (command.LastOutput.Contains("InProgress") || command.LastOutput.Contains("Ready"))
                    {
                        break;
                    }
                }

                barrier.SignalAndWait();

                Assert.IsTrue(command.Execute(new string[]
                {
                    string.Format("localcontentshare send-application 0x{0:X16} --timeout 300", TestAppId)
                }));
                {
                    var output = LocalContentShareHandlingForJson.Deserialize(command.LastOutput);
                    Assert.AreEqual((UInt32)LcsState.Completed, output.state);
                }

                Assert.IsTrue(command.Execute(new string[]
                {
                    "systemupdate get-background-state"
                }));
                Assert.IsTrue(command.LastOutput.Contains("InProgress") || command.LastOutput.Contains("Ready"));

            });

            var clientTask = Task.Run(() =>
            {
                ReceiveTest(ClientCommand, barrier, TestVersion.OldVersion);
            });

            hostTask.Wait();
            clientTask.Wait();

            TestImpl(TestVersion.NewVersion);
        }
    }
}
