﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    public class MakeTestSystemContent
    {
        public class ContentMetaInfo
        {
            public string Id;
            public string Type;
            public UInt32 Version;
            public byte Attribute;

            public ContentMetaInfo(ulong id, string type, UInt32 version, byte attribute = 0)
            {
                Id = string.Format("{0:X16}", id);
                Type = type;
                Version = version;
                Attribute = attribute;
            }

            public ContentMetaInfo(NspInfo info, UInt32 version, byte attribute = 0)
            {
                Id = info.Id;
                Type = info.Type;
                Version = version;
                Attribute = attribute;
            }
        }

        private AuthoringTool m_AuthoringTool;
        public string OutputDirPath { get; private set; }

        public MakeTestSystemContent(TestContext context)
        {
            m_AuthoringTool = new AuthoringTool(context);
            OutputDirPath = Path.Combine(new TestUtility.TestPath(context).GetSigloRoot(), "Tests/Outputs", context.TestName);

            Cleanup();
        }

        public void Cleanup()
        {
            if (Directory.Exists(OutputDirPath))
            {
                Directory.Delete(OutputDirPath, true);
            }

            // Directory.Delete は非同期なので、終了を待つ
            const int Timeout = 10 * 1000; // 10 秒
            const int Term = 100; // 100 ミリ秒
            for (var current = 0; current < Timeout; current += Term)
            {
                if (Directory.Exists(OutputDirPath))
                {
                    break;
                }
                System.Threading.Thread.Sleep(Term);
            }

            Directory.CreateDirectory(OutputDirPath);
        }

        public NspInfo MakeTestSystemUpdateMeta(ulong id, UInt32 version, IEnumerable<ContentMetaInfo> list)
        {
            var fileName = string.Format("SystemUpdateMeta_0x{0:X16}_v{1}", id, version);
            var nmetaPath = Path.Combine(OutputDirPath, string.Format("{0}.nmeta", fileName));
            var nspPath = Path.Combine(OutputDirPath, string.Format("{0}.nsp", fileName));

            var metaContents = new List<string>();
            metaContents.AddRange(new string[]
                {
                    string.Format("<NintendoSdkMeta>"),
                    string.Format("    <SystemUpdate>"),
                    string.Format("        <ContentMeta>"),
                    string.Format("            <Id>0x{0:X16}</Id>", id),
                    string.Format("            <Version>{0}</Version>", version),
                    string.Format("            <Type>{0}</Type>", "SystemUpdate"),
                });

            foreach (var info in list)
            {
                metaContents.AddRange(new string[]
                    {
                        string.Format("                <ContentMeta>"),
                        string.Format("                    <Id>0x{0:X16}</Id>", info.Id),
                        string.Format("                    <Version>{0}</Version>", info.Version),
                        string.Format("                    <Type>{0}</Type>", info.Type),
                        string.Format("                    <ContentMetaAttribute>{0}</ContentMetaAttribute>", info.Attribute),
                        string.Format("                </ContentMeta>"),
                    });
            }

            metaContents.AddRange(new string[]
                {
                    string.Format("        </ContentMeta>"),
                    string.Format("    </SystemUpdate>"),
                    string.Format("</NintendoSdkMeta>")
                });

            CreateFile(nmetaPath, metaContents);

            var outputs = m_AuthoringTool.Execute(string.Join(" ", new string[]
                {
                    "creatensp",
                    "--type", "SystemUpdate",
                    "-o", nspPath,
                    "--meta", nmetaPath,
                    "--keyconfig", m_AuthoringTool.KeyConfigFilePath
                }));

            if (!outputs.Item1)
            {
                throw new Exception(outputs.Item2.ToString());
            }

            return new NspInfo(nspPath, id, "SystemUpdate");
        }

        private void CreateFile(string filePath, IEnumerable<string> contents)
        {
            using (var fs = File.CreateText(filePath))
            {
                foreach (var line in contents)
                {
                    fs.WriteLine(line);
                }
            }
        }
    }
}
