﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    public class MakeNsData
    {
        public class BlackListRecord
        {
            public ulong Id { get; set; }
            public UInt32 Version { get; set; }

            public BlackListRecord(ulong id, UInt32 version)
            {
                Id = id;
                Version = version;
            }
        }

        private string m_MakeNsDataPath;
        private AuthoringTool m_AuthoringTool;

        public MakeNsData(TestContext context)
        {
            m_MakeNsDataPath = System.IO.Path.Combine(new TestUtility.TestPath(context).GetSigloRoot(), "Tools/CommandLineTools/MakeNsData/MakeNsData.exe");
            m_AuthoringTool = new AuthoringTool(context);
        }

        public Tuple<bool, StringBuilder, StringBuilder> Execute(string args)
        {
            using (var process = new Process())
            {
                var outputs = ProcessManager.SetupProcess(process, m_MakeNsDataPath, args);
                ProcessManager.RunAndWaitProcess(process);

                return new Tuple<bool, StringBuilder, StringBuilder>(process.ExitCode == 0, outputs.Item1, outputs.Item2);
            }
        }

        public void WriteBlackListCsv(string path, IEnumerable<BlackListRecord> list = null)
        {
            using (StreamWriter sw = new StreamWriter(path, false, Encoding.ASCII))
            {
                if (list != null)
                {
                    foreach (var v in list)
                    {
                        sw.WriteLine($"0x{v.Id:X16},{v.Version}");
                    }
                }
            }
        }

        public void MakeBlackListMeta(string path, string id, UInt32 version)
        {
            MakeBlackListMeta(path, Convert.ToUInt64(id, 16), version);
        }

        public void MakeBlackListMeta(string path, ulong id, UInt32 version)
        {
            using (StreamWriter sw = new StreamWriter(path, false, Encoding.UTF8))
            {
                sw.WriteLine($"<NintendoSdkMeta>");
                sw.WriteLine($"  <SystemData>");
                sw.WriteLine($"    <ContentMetaAttribute>Rebootless</ContentMetaAttribute>");
                sw.WriteLine($"    <Id>0x{id:X16}</Id>");
                sw.WriteLine($"    <Name>ApplicationBlackList</Name>");
                sw.WriteLine($"    <Version>{version}</Version>");
                sw.WriteLine($"  </SystemData>");
                sw.WriteLine($"</NintendoSdkMeta>");
            }
        }

        public NspInfo MakeBlackListNsp(string output, string name, string id, UInt32 version, IEnumerable<BlackListRecord> list = null)
        {
            return MakeBlackListNsp(output, name, Convert.ToUInt64(id, 16), version, list);
        }

        public NspInfo MakeBlackListNsp(string output, string name, ulong id, UInt32 version, IEnumerable<BlackListRecord> list = null)
        {
            var root = Path.Combine(output, name);
            Directory.CreateDirectory(root);

            // convert csv to binary
            {
                var csv = Path.Combine(output, $"{name}.csv");
                WriteBlackListCsv(csv, list);

                var binary = Path.Combine(root, "blacklist.dat");
                var result = Execute($"blacklist -i {csv} -o {binary} -v");
                if (!result.Item1)
                {
                    throw new Exception(result.Item2.ToString());
                }
            }

            // make blacklist nsp
            var nsp = Path.Combine(output, $"{name}.nsp");
            {
                var nmeta = Path.Combine(output, $"{name}.nmeta");
                MakeBlackListMeta(nmeta, id, version);

                var result = m_AuthoringTool.Execute(string.Join(" ", new string[]
                    {
                        "creatensp",
                        "--type", "SystemData",
                        "--meta", nmeta,
                        "--data", root,
                        "-o", nsp,
                        "--keyconfig", m_AuthoringTool.KeyConfigFilePath
                    }));
                if (!result.Item1)
                {
                    throw new Exception(result.Item2.ToString());
                }
            }
            return new NspInfo(nsp, id, "SystemData");
        }
    }
}
