﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    public class DevMenuCommandBase
    {
        public string TargetName { get; private set; }

        private RunOnTarget m_RunOnTarget;

        public class RunOnTargetOptions
        {
            public bool NoWait { get; set; }
            public bool SuppressAutoKill { get; set; }
        }

        public string LastOutput
        {
            get;
            private set;
        }

        public void SetVerbose(bool isVerbose)
        {
            m_RunOnTarget.IsVerboseLogEnabled = isVerbose;
        }

        public DevMenuCommandBase(TestContext context, string exePath, string target)
        {
            m_RunOnTarget = new RunOnTarget(context, exePath, target);
            TargetName = target;
        }

        public bool Execute(IEnumerable<string> commandList, string expect = null, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] expects = string.IsNullOrEmpty(expect) ? null : new string[] { expect };
            return SuccessExecuteImpl(commandList, expects, false, ignoreError, options);
        }

        public bool Execute(IEnumerable<string> commandList, IEnumerable<string> expects, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            return SuccessExecuteImpl(commandList, expects, false, ignoreError, options);
        }

        public bool ResetAndExecute(IEnumerable<string> commandList, string expect = null, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] expects = string.IsNullOrEmpty(expect) ? null : new string[] { expect };
            return SuccessExecuteImpl(commandList, expects, true, ignoreError, options);
        }

        public bool ResetAndExecute(IEnumerable<string> commandList, IEnumerable<string> expects, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            return SuccessExecuteImpl(commandList, expects, true, ignoreError, options);
        }

        private bool SuccessExecuteImpl(IEnumerable<string> commandList, IEnumerable<string> expects, bool doReset, bool ignoreError, RunOnTargetOptions options)
        {
            return ExecuteImpl(commandList, expects, doReset, true, ignoreError, options);
        }

        // DevMenuCommand は途中で Fail すると終わってしまうので、複数実行を受け付けないようにする
        public bool FailureExecute(string command, string expect = null, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] expects = string.IsNullOrEmpty(expect) ? null : new string[] { expect };
            return FailureExecute(command, expects, ignoreError, options);
        }

        public bool FailureExecute(string command, string[] expects, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] commandList = new string[] { command };
            return FailureExecuteImpl(commandList, expects, false, ignoreError, options);
        }

        public bool ResetAndFailureExecute(string command, string expect = null, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] expects = string.IsNullOrEmpty(expect) ? null : new string[] { expect };
            return ResetAndFailureExecute(command, expects, ignoreError, options);
        }

        public bool ResetAndFailureExecute(string command, string[] expects, bool ignoreError = false, RunOnTargetOptions options = null)
        {
            string[] commandList = new string[] { command };
            return FailureExecuteImpl(commandList, expects, true, ignoreError, options);
        }

        private bool FailureExecuteImpl(IEnumerable<string> commandList, IEnumerable<string> expects, bool doReset, bool ignoreError, RunOnTargetOptions options)
        {
            return ExecuteImpl(commandList, expects, doReset, false, ignoreError, options);
        }

        private string GetJoinString(bool ignoreError)
        {
            return ignoreError ? " THEN " : " AND ";
        }

        private bool ExecuteImpl(IEnumerable<string> commandList, IEnumerable<string> expects, bool doReset, bool expectsSuccess, bool ignoreError, RunOnTargetOptions options)
        {
            var args = string.Join(GetJoinString(ignoreError), commandList);

            if (options == null)
            {
                options = new RunOnTargetOptions();
            }

            var optionList = new string[]
            {
                expects != null ? string.Join("", expects.Select(x => " --pattern-not-found-failure \"" + x + "\"")) : "",
                expectsSuccess ? " --pattern-failure-exit \\[FAILURE\\] " : " --pattern-failure-exit \\[SUCCESS\\] ",
                " --filter \"FileName =~ /DevMenuCommand/\" ",
                options.NoWait ? "--no-wait " : "",
                options.SuppressAutoKill ? "--suppress-auto-kill " : ""
            };

            var outputs = m_RunOnTarget.Execute(args, doReset, optionList);
            var stdout = outputs.Item2.ToString();
            if (!options.NoWait)
            {
                var lastIndex = stdout.LastIndexOf("[SUCCESS]");
                if (lastIndex < 0)
                {
                    lastIndex = stdout.LastIndexOf("[FAILURE]");
                }
                LastOutput = stdout.Remove(lastIndex);
            }

            return outputs.Item1;
        }
    }

    public class DevMenuCommand : DevMenuCommandBase
    {
        public DevMenuCommand(TestContext context, string targetName = null) : base(
            context,
            System.IO.Path.Combine(new TestUtility.TestPath(context).GetSigloRoot(),
                "Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommand/Release/DevMenuCommand.nsp"),
            (targetName != null) ? targetName : (string)context.Properties["TargetName"])
        {}
    }

    public class DevMenuCommandSystem : DevMenuCommandBase
    {
        public DevMenuCommandSystem(TestContext context, string targetName = null) : base(
            context,
            System.IO.Path.Combine(new TestUtility.TestPath(context).GetSigloRoot(),
                "Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Release/DevMenuCommandSystem.nsp"),
            (targetName != null) ? targetName : (string)context.Properties["TargetName"])
        {}
    }
}
