﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    [TestClass]
    public class ExcecutionTest
    {
        public TestContext TestContext { get; set; }

        [TestMethod]
        [Timeout(90 * 1000)]
        public void TestBasic()
        {
            Console.WriteLine("TestBasic start.");

            var maker = new MakeTestApplication(this.TestContext);
            var app1 = maker.MakeApplication(0x0100394000059000);
            var app2 = maker.MakeApplication(0x0100394000059001);

            var command = new DevMenuCommand(this.TestContext);
            command.SetVerbose(true);

            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
                "application list",
                "application install " + app1.Path + " --measure",
                "application verify " + app1.Id,
                "application install " + app2.Path,
                "application verify " + app2.Id,
                "application list",
                "application uninstall " + app1.Id,
                "application uninstall " + app2.Id,
                "application list"
            }));

            maker.Cleanup();
        }

        [TestMethod]
        [Timeout(90 * 1000)]
        public void TestBasicSystem()
        {
            Console.WriteLine("TestBasicSystem start.");

            var testPath = new TestUtility.TestPath(this.TestContext);

            var maker = new MakeTestApplication(this.TestContext);
            var app1 = maker.MakeApplication(0x0100394000059000);
            var app2 = maker.MakeApplication(0x0100394000059001);

            var command = new DevMenuCommandSystem(this.TestContext);
            command.SetVerbose(true);

            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
                "application list",
                "application install " + app1.Path + " --measure",
                "application verify " + app1.Id,
                "application install " + app2.Path,
                "application verify " + app2.Id,
                "application list",
                "application uninstall " + app1.Id,
                "application uninstall " + app2.Id,
                "application list",
                "application update-version-list " + testPath.GetSigloRoot() + "\\Tests\\Ns\\Sources\\Tests\\VersionList\\DummyVersionList_Full.json"
            }));

            var systemData1 = maker.MakeSystemData(0x1000000000000003);
            var systemData2 = maker.MakeSystemData(0x1000000000000004);
            var systemProgram = maker.MakeSystemData(0x0100000000001000);

            Assert.IsTrue(command.Execute(new string[] {
                "systemprogram list",
                "systemprogram install " + systemData1.Path,
                "systemprogram install " + systemData2.Path,
                "systemprogram uninstall 0x1000000000000003",
                "systemprogram uninstall 0x1000000000000004",
                "systemprogram list",
                "applet redirect-program --id 0x0100000000001000 --path " + systemProgram.Path,
                "applet list-redirect-programs",
                "batch " + testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\DevMenuCommandTest\\batch.txt",
                "error dump-all"
            }));

            maker.Cleanup();
        }

        [TestMethod]
        [Timeout(90 * 1000)]
        public void TestEntity()
        {
            Console.WriteLine("TestEntity start.");
            var maker = new MakeTestApplication(this.TestContext);
            var app = maker.MakeApplication(0x0100394000059000);
            var aoc = maker.MakeAddOnContent(0x0100394000059000, 1);

            var command = new DevMenuCommandSystem(this.TestContext);
            command.SetVerbose(true);

            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
                "application install " + app.Path + " --measure",
                "addoncontent install " + aoc.Path,
                "application list-record " + app.Id,
                "application list-content-meta-database",
                "application verify " + app.Id,
                "application delete-entity " + app.Id,
            }));

            maker.Cleanup();
        }

        [TestMethod]
        [Timeout(90 * 1000)]
        public void TestCleanupRedundant()
        {
            Console.WriteLine("TestCleanupRedundant start.");
            var maker = new MakeTestApplication(this.TestContext);
            var aocv0 = maker.MakeAddOnContent(0x0100394000059000, 1, 0);
            var aocv1 = maker.MakeAddOnContent(0x0100394000059000, 1, 1);

            var command = new DevMenuCommandSystem(this.TestContext);
            command.SetVerbose(true);

            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
                "application install " + aocv1.Path + " --measure",
                "application install-redundant-entity " + aocv0.Path,
                "application list-record " + aocv1.Id,
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database",
            },
                @"\""version\"": 0,"
            ));
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database",
            },
                @"\""version\"": 65536,"
            ));

            Assert.IsTrue(command.Execute(new string[] {
                "application delete-redundant-entity",
            }));

            Assert.IsFalse(command.Execute(new string[] {
                "application list-content-meta-database",
            },
                @"\""version\"": 0,"
            ));
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database",
            },
                @"\""version\"": 65536,"
            ));

            maker.Cleanup();
        }

        [TestMethod]
        [Timeout(150 * 1000)]
        public void TestPatchAddOnContent()
        {
            Console.WriteLine("TestBasic start.");

            var maker = new MakeTestApplication(this.TestContext);
            var app = maker.MakeApplication(0x0100394000059000);
            var patch1 = maker.MakePatch(0x0100394000059000, 1, app.Path);
            var patch2 = maker.MakePatch(0x0100394000059000, 2, app.Path, patch1.Path);
            var aoc = maker.MakeAddOnContent(0x0100394000059000, 1, 17);

            var command = new DevMenuCommand(this.TestContext);
            command.SetVerbose(true);

            Assert.IsTrue(command.Execute(new string[]
            {
                "application uninstall --all",
                "application install " + app.Path + " --measure",
                "patch install " + patch1.Path,
                "addoncontent install " + aoc.Path
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list"
            },
                "1.0"
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "patch list"
            },
                "1.0"
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            },
                "17"
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "patch install " + patch2.Path
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list"
            },
                "2.0"
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "patch list"
            },
                "2.0"
            ));
        }

    }
}
