﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace GitHookTest
{
    [TestClass]
    public class PreCommitFileChangeTest
    {
        public TestContext TestContext { get; set; }
        private GitHookTestContext m_GitHookTestContext { get; set; }
        private const string JIRA_KEY = "SIGLO-0";

        [TestInitialize]
        public void Initialize()
        {
            m_GitHookTestContext = new GitHookTestContext(TestContext);
        }

        [TestCleanup]
        public void CleanUp()
        {
            m_GitHookTestContext.Dispose();
        }

        [TestMethod]
        public void UpdateFileInIndexTest()
        {
            string testFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "test.cpp");
            File.WriteAllText(testFile, "test.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test\"");

            // git 管理下にあるファイルを更新してコミット
            File.WriteAllText(testFile, "test2.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test2\"");
        }

        [TestMethod]
        public void DeleteFileFromIndexTest()
        {
            string testFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "test.cpp");
            File.WriteAllText(testFile, "test.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test\"");

            // git 管理下にあるファイルを削除してコミット
            m_GitHookTestContext.ExecuteGit($"rm \"{testFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test2\"");
        }

        [TestMethod]
        public void RenameFileInIndexTest()
        {
            string testFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "test.cpp");
            File.WriteAllText(testFile, "test.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test\"");

            // git 管理下にあるファイルをリネームしてコミット
            string newTestFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "newtest.cpp");
            m_GitHookTestContext.ExecuteGit($"mv \"{testFile}\" \"{newTestFile}\"");
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test2\"");
        }

        [TestMethod]
        public void ChangeFileOnDiskTest()
        {
            string testFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "test.cpp");
            File.WriteAllText(testFile, "test.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");

            // ディスク上のファイルに Stage されていない変更がある状態でコミット
            File.WriteAllText(testFile, "test2.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test\"");

            // ディスク上には Stage されていない変更内容が残り、かつ自動整形されていることを確認
            Assert.AreEqual("test2.\r\n", File.ReadAllText(testFile));
        }

        [TestMethod]
        public void DeleteFileOnDiskTest()
        {
            string testFile = Path.Combine(m_GitHookTestContext.TestLocalRepositoryDirectory, "test.cpp");
            File.WriteAllText(testFile, "test.   ", new UTF8Encoding(true));
            m_GitHookTestContext.ExecuteGit($"add \"{testFile}\"");

            // ディスク上ではファイルを削除した状態でコミット
            File.Delete(testFile);
            m_GitHookTestContext.ExecuteGit($"commit -m \"({JIRA_KEY}) test\"");
        }
    }
}
