﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CodingCheckerUtilTest
{
    [TestClass]
    public class FormatterUtilTest
    {
        public TestContext TestContext { get; set; }

        private const string LineBreakLf = "\nHello\nWorld\n";
        private const string LineBreakCr = "\rHello\rWorld\r";
        private const string LineBreakCrlf = "\r\nHello\r\nWorld\r\n";
        private const string LineBreakMixed = "\nHello\rWorld\r\n";


        [TestMethod]
        public void CheckBOMTest()
        {
            string text = "test.";

            // BOM 付きの場合
            {
                var encoding = new UTF8Encoding(true);
                Assert.IsTrue(FormatterUtil.CheckBOM(encoding.GetPreamble().Concat(encoding.GetBytes(text)).ToArray()));
            }

            // BOM なしの場合
            {
                var encoding = new UTF8Encoding(false);
                Assert.IsFalse(FormatterUtil.CheckBOM(encoding.GetBytes(text)));
            }

            // 空のバイト列の場合
            Assert.IsFalse(FormatterUtil.CheckBOM(new byte[] { }));
        }

        [TestMethod]
        public void CheckLineBreakTest()
        {
            // 改行コードの変換が必要かどうかの判定
            Assert.IsTrue(FormatterUtil.CheckLineBreak(LineBreakLf));
            Assert.IsTrue(FormatterUtil.CheckLineBreak(LineBreakCr));
            Assert.IsFalse(FormatterUtil.CheckLineBreak(LineBreakCrlf));
            Assert.IsTrue(FormatterUtil.CheckLineBreak(LineBreakMixed));
        }

        [TestMethod]
        public void ConvertLineBreakToLfTest()
        {
            // 改行コード LF への変換
            string actual;
            actual = string.Copy(LineBreakLf);
            FormatterUtil.ConvertLineBreakToLf(ref actual);
            Assert.AreEqual(LineBreakLf, actual);
            actual = string.Copy(LineBreakCr);
            FormatterUtil.ConvertLineBreakToLf(ref actual);
            Assert.AreEqual(LineBreakLf, actual);
            actual = string.Copy(LineBreakCrlf);
            FormatterUtil.ConvertLineBreakToLf(ref actual);
            Assert.AreEqual(LineBreakLf, actual);
            actual = string.Copy(LineBreakMixed);
            FormatterUtil.ConvertLineBreakToLf(ref actual);
            Assert.AreEqual(LineBreakLf, actual);
        }

        [TestMethod]
        public void ConvertLfToCrlfTest()
        {
            // 改行コード LF から CRLF への変換
            string actual;
            actual = string.Copy(LineBreakLf);
            FormatterUtil.ConvertLfToCrlf(ref actual);
            Assert.AreEqual(LineBreakCrlf, actual);
        }

        [TestMethod]
        public void FormatTabToSpaceTest()
        {
            // 行頭タブのスペースへの変換
            var target = "\tLine1\n\t\tLine2";
            var expected = "    Line1\n        Line2";
            Assert.IsTrue(FormatterUtil.FormatTabToSpace(ref target));
            Assert.AreEqual(expected, target);

            // 行頭タブのみ変換し、行中タブは変換しない
            target = "Line\t1\n\tLine2\t\n";
            expected = "Line\t1\n    Line2\t\n";
            Assert.IsTrue(FormatterUtil.FormatTabToSpace(ref target));
            Assert.AreEqual(expected, target);

            // 行頭タブがなければ何も行わない
            target = "value1\tvalue2\tvalue3\n    Line2\n";
            expected = "value1\tvalue2\tvalue3\n    Line2\n";
            Assert.IsFalse(FormatterUtil.FormatTabToSpace(ref target));
            Assert.AreEqual(expected, target);
        }

        [TestMethod]
        public void FormatFootSpaceTest()
        {
            // 行末のスペースを取り除く
            var target = "Line1    \nLine2\n";
            var expected = "Line1\nLine2\n";
            Assert.IsTrue(FormatterUtil.FormatFootSpace(ref target));
            Assert.AreEqual(expected, target);

            // スペースのみの行は空行になる
            target = "Line1\n        \nLine3\n";
            expected = "Line1\n\nLine3\n";
            Assert.IsTrue(FormatterUtil.FormatFootSpace(ref target));
            Assert.AreEqual(expected, target);

            // 行頭、行中スペースは変換しない
            target = "    Line1    \nLine2    WithTail\n";
            expected = "    Line1\nLine2    WithTail\n";
            Assert.IsTrue(FormatterUtil.FormatFootSpace(ref target));
            Assert.AreEqual(expected, target);

            // 行末スペースがなければ何も行わない
            target = "    Line1\n\nLine3    WithTail\nLine4\n";
            expected = "    Line1\n\nLine3    WithTail\nLine4\n";
            Assert.IsFalse(FormatterUtil.FormatFootSpace(ref target));
            Assert.AreEqual(expected, target);
        }

        [TestMethod]
        public void FormatFootLineBreakTest()
        {
            // 終端が改行でなければ付加する1
            var target = "Line1\nLine2\nLine3";
            var expected = "Line1\nLine2\nLine3\n";
            Assert.IsTrue(FormatterUtil.FormatFootLineBreak(ref target));
            Assert.AreEqual(expected, target);

            // 終端が改行でなければ付加する2
            target = "Line1\nLine2\n\nLine3";
            expected = "Line1\nLine2\n\nLine3\n";
            Assert.IsTrue(FormatterUtil.FormatFootLineBreak(ref target));
            Assert.AreEqual(expected, target);

            // 終端が改行ならば何も行わない1
            target = "Line1\nLine2\nLine3\n";
            expected = "Line1\nLine2\nLine3\n";
            Assert.IsFalse(FormatterUtil.FormatFootLineBreak(ref target));
            Assert.AreEqual(expected, target);

            // 終端が改行ならば何も行わない2
            target = "Line1\nLine2\nLine3\n\n";
            expected = "Line1\nLine2\nLine3\n\n";
            Assert.IsFalse(FormatterUtil.FormatFootLineBreak(ref target));
            Assert.AreEqual(expected, target);
        }
    }
}
