﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Bin2ObjTest
{
    using System;
    using System.IO;
    using ElfLibrary;
    using ElfLibrary.Elf32;
    using ElfLibrary.Elf64;
    using Microsoft.VisualStudio.TestTools.UnitTesting;

    [TestClass]
    public class UnitTest1
    {
        private const string ResourceDir = "..\\..\\..\\..\\..\\Tools\\Resources\\BinaryData\\";
        private const string OutputDir = "";

        [TestMethod]
        // デフォルトの挙動を確認します。
        public void CheckDefault()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "DefaultObject");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.ARM == header32.Machine, "ElfMachineType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf32SectionHeader section = new Elf32SectionHeader();
                    fs.Seek((long)(header32.SectionOffset + (ulong)header32.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC | (uint)ElfSectionAttributeFlags.WRITE) == section.Flags,
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // ビッグエンディアンのオブジェクトファイルが生成できることを確認します。
        public void CheckBigEndianFlag()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "BigEndianObject");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "--big-endian", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.BigEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.PowerPC == header32.Machine, "ElfMachineType が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // Compatible フラグが有効であることを確認します。
        public void CheckCompatibleFlag()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "CompatibleObject");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "-c", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.ARM == header32.Machine, "ElfMachineType が一致しませんでした。");

                    Elf32SectionHeader section = new Elf32SectionHeader();
                    long offset = (long)header32.SectionOffset + header32.SectionEntrySize * 3; // シンボル名テーブル
                    fs.Seek(offset, SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    offset = (long)section.Offset;
                    fs.Seek(offset, SeekOrigin.Begin);
                    byte[] array = reader.ReadBytes((int)section.Size);
                    string symbol_name = "_binary_Dummy_png";
                    char[] tmp = symbol_name.ToCharArray();
                    // シンボル名テーブルの中にある 1つ目のシンボル名を比較する
                    for (int i = 1; i <= symbol_name.Length; i++)
                    {
                        Assert.IsTrue(array[i] == (byte)tmp[i - 1], "シンボル名が一致しませんでした。");
                    }

                    offset = (long)header32.SectionOffset + header32.SectionEntrySize * 4;
                    fs.Seek(offset, SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC | (uint)ElfSectionAttributeFlags.WRITE) == section.Flags,
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // 64-bit のオブジェクトファイルを生成できることを確認します。
        public void Check64bitBinary()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "64bitObject");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "-m", "aarch64", "--little-endian", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf64 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf64Header header64 = (Elf64Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.AArch64 == header64.Machine, "ElfMachineType が一致しませんでした。");

                    Elf64SectionHeader section = new Elf64SectionHeader();
                    fs.Seek((long)(header64.SectionOffset + (ulong)header64.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC | (uint)ElfSectionAttributeFlags.WRITE) == section.Flags,
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // アライン正常系のチェック
        public void CheckValidAlign()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "ValidAlignObject");
            uint align = 8;

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "--align", align.ToString(), binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.ARM == header32.Machine, "ElfMachineType が一致しませんでした。");

                    Elf32SectionHeader section = new Elf32SectionHeader();
                    fs.Seek((long)(header32.SectionOffset + (uint)header32.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(section.Offset % align == 0, "Offset の Align が揃っていませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // アライン異常系のチェック
        public void CheckInvalidAlign()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "InvalidAlignObject");
            uint align = 7;

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "--align", align.ToString(), binaryFile, objectFile });

            Assert.IsFalse(System.IO.File.Exists(objectFile), "Align が 2のべき乗ではないので、{0}が生成されています。", objectFile);
        }

        [TestMethod]
        // ReadOnly のフラグが有効であることを確認します。
        public void CheckReadOnly()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "ReadOnlyObject");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "-r", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.ARM == header32.Machine, "ElfMachineType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf32SectionHeader section = new Elf32SectionHeader();
                    fs.Seek((long)(header32.SectionOffset + (uint)header32.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC == section.Flags),
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // AArch64 lp64 のバイナリが生成できることを確認します。
        public void CheckAArch64Lp64()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "64bitLp64Object");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "-m", "aarch64", "--lp64", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf64 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf64Header header64 = (Elf64Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.AArch64 == header64.Machine, "ElfMachineType が一致しませんでした。");

                    Elf64SectionHeader section = new Elf64SectionHeader();
                    fs.Seek((long)(header64.SectionOffset + (ulong)header64.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC | (uint)ElfSectionAttributeFlags.WRITE) == section.Flags,
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }

        [TestMethod]
        // AArch64 ilp32 のバイナリが生成できることを確認します。
        public void CheckAArch64Ilp32()
        {
            string binaryFile = Path.Combine(ResourceDir, "Dummy.png");
            string objectFile = Path.Combine(OutputDir, "64bitIlp32Object");

            try
            {
                if (File.Exists(objectFile))
                {
                    File.Delete(objectFile);
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            Bin2Obj.Program.Main(new string[] { "-m", "aarch64", "--ilp32", binaryFile, objectFile });

            Assert.IsTrue(System.IO.File.Exists(objectFile), "{0} が存在しません", objectFile);

            try
            {
                using (FileStream fs = new FileStream(objectFile, FileMode.Open, FileAccess.Read))
                {
                    ElfHeader header = ElfHeader.MakeHeader(fs);

                    Assert.IsTrue(ElfArchitectureType.Elf32 == header.Architecture, "ElfArchitectureType が一致しませんでした。");
                    Assert.IsTrue(ElfByteOrderType.LittleEndian == header.ByteOrder, "ElfByteOrderType が一致しませんでした。");

                    BinaryReader binaryReader = new BinaryReader(fs);
                    ElfBinaryReader reader = new ElfBinaryReader(binaryReader);
                    Elf32Header header32 = (Elf32Header)header;
                    Assert.IsTrue((ushort)ElfMachineType.AArch64 == header32.Machine, "ElfMachineType が一致しませんでした。");

                    Elf64SectionHeader section = new Elf64SectionHeader();
                    fs.Seek((long)(header32.SectionOffset + (ulong)header32.SectionEntrySize * 4), SeekOrigin.Begin);
                    section.ReadElfFile(reader);
                    Assert.IsTrue(((uint)ElfSectionAttributeFlags.ALLOC | (uint)ElfSectionAttributeFlags.WRITE) == section.Flags,
                        "ElfSectionAttributeFlags が一致しませんでした。");
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }
        }
    }
}
