﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <chrono>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "..\Common\LogReaderWrapper.h"

/*
*   単体テスト
*   事前に Target Manager に接続し、ログが出力されない状態である必要があります。
*   実行時には引数で対象ターゲットのシリアル番号を指定してください。
*/

namespace
{
    SerialNumberString g_SerialNumber;
    LogReaderWrapper g_LogReaderWrapper;

    class NoLog : public ::testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            ASSERT_TRUE(g_LogReaderWrapper.TryLoadLibrary());
            ASSERT_TRUE(g_LogReaderWrapper.TryLoadFunctions());
        }
        virtual void SetUp()
        {
        }
        virtual void TearDown()
        {
        }
    };
}

TEST_F(NoLog, GetLogFailed)
{
    const size_t bufferSize = 128;
    char buffer[bufferSize];

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 0));

    size_t logSize;
    EXPECT_EQ(LogReaderResult_NotFound, g_LogReaderWrapper.m_GetLogLineSizeFunction(&logSize));
    EXPECT_EQ(LogReaderResult_NotFound, g_LogReaderWrapper.m_GetLogLineFunction(buffer, bufferSize));

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(NoLog, SearchFailed)
{
    long long waitTimeMs = 10000;

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 0));
    g_LogReaderWrapper.m_MoveToNewestLineFunction();
    g_LogReaderWrapper.m_MoveToFormerLineByTimeFunction(1000);

    EXPECT_EQ(LogReaderResult_Timeout, g_LogReaderWrapper.m_WaitUntilMatchedFunction(".*", 0));

    auto before = std::chrono::system_clock::now();
    EXPECT_EQ(LogReaderResult_Timeout, g_LogReaderWrapper.m_WaitUntilMatchedFunction(".*", (int)waitTimeMs));
    auto after = std::chrono::system_clock::now();
    auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(after - before).count();
    EXPECT_LT(waitTimeMs - 1000, diff);
    EXPECT_GT(waitTimeMs + 1000, diff);

    EXPECT_EQ(LogReaderResult_NotFound, g_LogReaderWrapper.m_SearchFormerLogFunction(".*"));
    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(NoLog, MoveFailed)
{
    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 0));
    EXPECT_EQ(LogReaderResult_CannotMoveLine, g_LogReaderWrapper.m_MoveToNextLineFunction());
    EXPECT_EQ(LogReaderResult_CannotMoveLine, g_LogReaderWrapper.m_MoveToPreviousLineFunction());
    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(NoLog, WaitTimeout)
{
    long long waitTimeMs = 10000;

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 0));

    // 指定時間でタイムアウトするかをテスト
    auto before = std::chrono::system_clock::now();
    EXPECT_EQ(LogReaderResult_Timeout, g_LogReaderWrapper.m_WaitForNextLineFunction((int)waitTimeMs));
    auto after = std::chrono::system_clock::now();
    auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(after - before).count();
    EXPECT_LT(waitTimeMs - 1000, diff);
    EXPECT_GT(waitTimeMs + 1000, diff);

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc >= 2)
    {
        strncpy_s(g_SerialNumber.value, argv[1], sizeof(g_SerialNumber.value));
        printf("#######################################\n");
        printf("  Specified serial: %s\n", g_SerialNumber.value);
        printf("#######################################\n");
    }
    else
    {
        printf("Please specify serial number by argv[1]\n");
        nnt::Exit(1);
    }

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();


    nnt::Exit(exitCode);
}

