﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include "HidInputerWrapper.h"
#include "TestUtil.h"

using namespace hidInputer;

namespace
{
    struct VirtualControllerDeviceInfoCase
    {
        ControllerDeviceType     deviceType;
        ControllerInterfaceType  interfaceType;
    };

    VirtualControllerDeviceInfoCase g_DeviceTypeCase[] = {
        {
            ControllerDeviceType::ControllerDeviceType_SwitchProController,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth
        },
        {
            ControllerDeviceType::ControllerDeviceType_JoyConLeft,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth
        },
        {
            ControllerDeviceType::ControllerDeviceType_JoyConRight,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth
        },
        {
            ControllerDeviceType::ControllerDeviceType_JoyConLeft,
            ControllerInterfaceType::ControllerInterfaceType_Rail
        },
        {
            ControllerDeviceType::ControllerDeviceType_JoyConRight,
            ControllerInterfaceType::ControllerInterfaceType_Rail
        }
    };

    VirtualControllerDeviceInfoCase g_InvalidDeviceTypeCase[] = {
        {
            ControllerDeviceType::ControllerDeviceType_SwitchProController,
            ControllerInterfaceType::ControllerInterfaceType_Rail,
        }
    };

    Button g_AllVirtualControllerButton[] = {
        Button::Button_A,
        Button::Button_B,
        Button::Button_X,
        Button::Button_Y,
        Button::Button_StickL,
        Button::Button_StickR,
        Button::Button_L,
        Button::Button_R,
        Button::Button_ZL,
        Button::Button_ZR,
        Button::Button_Plus,
        Button::Button_Minus,
        Button::Button_Left,
        Button::Button_Up,
        Button::Button_Right,
        Button::Button_Down,
        Button::Button_SL,
        Button::Button_SR,
        Button::Button_Home,
        Button::Button_Capture,
    };

    Stick g_AllVirtualControllerStick[] = {
        Stick::Stick_L,
        Stick::Stick_R
    };

    void VirtualControllerConnectTestHelper(
        HidInputerWrapper &hidInputerWrapper,
        ControllerDeviceInfo deviceInfo,
        ControllerResult expectResult)
    {
        int id;
        ControllerResult result = hidInputerWrapper.m_AddControllerWithDeviceInfoFunction(&id, deviceInfo);

        EXPECT_EQ(expectResult, result);

        if (expectResult == ControllerResult::ControllerResult_Success && result == ControllerResult::ControllerResult_Success)
        {
            ControllerDeviceInfo connectedDeviceInfo;
            hidInputerWrapper.m_GetConnectedControllerDeviceInfoFunction(&connectedDeviceInfo, id);

            EXPECT_EQ(deviceInfo.deviceType, connectedDeviceInfo.deviceType);
            EXPECT_EQ(deviceInfo.interfaceType, connectedDeviceInfo.interfaceType);
            for (int i = 0; i < 3; i++)
            {
                EXPECT_EQ(deviceInfo.mainColor[i], connectedDeviceInfo.mainColor[i]);
                EXPECT_EQ(deviceInfo.subColor[i], connectedDeviceInfo.subColor[i]);
            }
        }
    }

    void VirtualControllerButtonTestHelper(
        HidInputerWrapper &hidInputerWrapper,
        int controllerCount,
        ControllerDeviceType deviceType,
        ControllerInterfaceType interfaceType,
        Button buttonList[],
        int buttonCount,
        Button invalidButtonList[],
        int invalidButtonCount)
    {
        for (int i = 0; i < controllerCount; i++)
        {
            int id;
            ControllerDeviceInfo deviceInfo = {};
            deviceInfo.deviceType = deviceType;
            deviceInfo.interfaceType = interfaceType;

            EXPECT_EQ(ControllerResult::ControllerResult_Success,
                hidInputerWrapper.m_AddControllerWithDeviceInfoFunction(&id, deviceInfo));


            // ボタン操作テスト
            for (int j = 0; j < buttonCount; j++)
            {
                ControllerResult expectResult = ControllerResult::ControllerResult_Success;
                Button targetButton = buttonList[j];

                for (int k = 0; k < invalidButtonCount; k++)
                {
                    if (targetButton == invalidButtonList[k])
                    {
                        expectResult = ControllerResult::ControllerResult_InvalidButton;
                        break;
                    }
                }

                EXPECT_EQ(expectResult, hidInputerWrapper.m_PressButtonFunction( id, targetButton));

                EXPECT_EQ(expectResult, hidInputerWrapper.m_ReleaseButtonFunction( id, targetButton));

                EXPECT_EQ(expectResult, hidInputerWrapper.m_PushButtonFunction(id, targetButton, 10));
            }
        }

        // コントローラー切断
        ReleaseAllController(&hidInputerWrapper);
    }

    void VirtualControllerStickTestHelper(
        HidInputerWrapper &hidInputerWrapper,
        int controllerCount,
        ControllerDeviceType deviceType,
        ControllerInterfaceType interfaceType,
        Stick stickList[],
        int stickCount,
        Stick invalidStickList[],
        int invalidStickCount)
    {
        // 接続
        for (int i = 0; i < controllerCount; i++)
        {
            int id;
            ControllerDeviceInfo deviceInfo = {};
            deviceInfo.deviceType = deviceType;
            deviceInfo.interfaceType = interfaceType;

            EXPECT_EQ(ControllerResult::ControllerResult_Success,
                hidInputerWrapper.m_AddControllerWithDeviceInfoFunction(&id, deviceInfo));

            for (int j = 0; j < stickCount; j++)
            {
                ControllerResult expectResult = ControllerResult::ControllerResult_Success;
                Stick targetButton = stickList[j];

                for (int k = 0; k < invalidStickCount; k++)
                {
                    if (targetButton == invalidStickList[k])
                    {
                        expectResult = ControllerResult::ControllerResult_InvalidStick;
                        break;
                    }
                }

                EXPECT_EQ(expectResult,
                    hidInputerWrapper.m_HoldAnalogStickFunction(
                        id, targetButton, i * 40.0f, 1.0));

                EXPECT_EQ(expectResult,
                    hidInputerWrapper.m_ReleaseAnalogStickFunction(
                        id, targetButton));

                EXPECT_EQ(expectResult,
                    hidInputerWrapper.m_TiltAnalogStickFunction(
                        id, targetButton, -i * 40.0f, 1.0, 100));
            }
        }

        // コントローラー切断
        ReleaseAllController(&hidInputerWrapper);
    }
}

// 仮想コントローラー接続
TEST(HidInputerSuite, VirtualControllerConnectionTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);


    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// デバイス情報設定
TEST(HidInputerSuite, ConnectControllerWithDeviceInfoTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);


    //-------------------------------------
    // 仮想デバイスの情報設定
    //-------------------------------------
    // 仮想コントローラー接続前のデバイス情報更新
    const int testCaseCount = sizeof(g_DeviceTypeCase) / sizeof(VirtualControllerDeviceInfoCase);
    const int invalidTestCaseCount = sizeof(g_InvalidDeviceTypeCase) / sizeof(VirtualControllerDeviceInfoCase);


    // 不正なデバイス種別とインターフェース種別の組み合わせ
    for (int i = 0; i < invalidTestCaseCount; i++ )
    {
        for (int j = 0; j < g_ControllerIndexCount; j++)
        {
            ControllerDeviceInfo deviceInfo = {};
            deviceInfo.deviceType = g_InvalidDeviceTypeCase[i].deviceType;
            deviceInfo.interfaceType = g_InvalidDeviceTypeCase[i].interfaceType;

            VirtualControllerConnectTestHelper(
                hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_InvalidDeviceInfo);
        }
        ReleaseAllController(&hidInputerWrapper);
    }


    // 正常なデバイス種別とインターフェース種別の組み合わせ
    for (int i = 0; i < testCaseCount; i++)
    {
        int testConnectCount = g_ControllerIndexCount;

        // レール接続デバイスは１個のみ接続テスト
        if (g_DeviceTypeCase[i].interfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail)
        {
            testConnectCount = 1;
        }
        for (int j = 0; j < testConnectCount; j++)
        {
            ControllerDeviceInfo deviceInfo = {};

            deviceInfo.deviceType = g_DeviceTypeCase[i].deviceType;
            deviceInfo.interfaceType = g_DeviceTypeCase[i].interfaceType;

            VirtualControllerConnectTestHelper(
                hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_Success);
        }
        ReleaseAllController(&hidInputerWrapper);
    }

    // 本体レール最大接続数チェック
    {
        ControllerDeviceInfo deviceInfo = {};
        deviceInfo.deviceType = ControllerDeviceType::ControllerDeviceType_JoyConLeft;
        deviceInfo.interfaceType = ControllerInterfaceType::ControllerInterfaceType_Rail;

        // 左レールへ２台接続
        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_Success);

        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_ControllerCountOver);

        // 右レールへ２台接続
        deviceInfo.deviceType = ControllerDeviceType::ControllerDeviceType_JoyConRight;
        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_Success);

        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_ControllerCountOver);

        ReleaseAllController(&hidInputerWrapper);
    }

    // 8 台まで接続
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        ControllerDeviceInfo deviceInfo = {};

        deviceInfo.deviceType = ControllerDeviceType::ControllerDeviceType_SwitchProController;
        deviceInfo.interfaceType = ControllerInterfaceType::ControllerInterfaceType_Bluetooth;

        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_Success);
    }

    // 最大接続数を超えて接続
    for (int i = 0; i < testCaseCount; i++)
    {
        ControllerDeviceInfo deviceInfo = {};

        deviceInfo.deviceType = g_DeviceTypeCase[i].deviceType;
        deviceInfo.interfaceType = g_DeviceTypeCase[i].interfaceType;

        VirtualControllerConnectTestHelper(
            hidInputerWrapper, deviceInfo, ControllerResult::ControllerResult_ControllerCountOver);
    }


    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

TEST(HidInputerSuite, GetConnectedControllerDeviceInfoTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    // TIPS: Npad のデバイス情報取得は、Npad 接続テストと合わせて ConnectControllerWithDeviceInfoTest 内で行う

    // Xpad のデバイス情報取得
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        EXPECT_EQ(ControllerResult::ControllerResult_Success,
            hidInputerWrapper.m_AddControllerForIdFunction(i));

        ControllerDeviceInfo deviceInfo;
        EXPECT_EQ(ControllerResult::ControllerResult_Success,
            hidInputerWrapper.m_GetConnectedControllerDeviceInfoFunction(&deviceInfo, i));

        EXPECT_EQ(deviceInfo.deviceType, ControllerDeviceType::ControllerDeviceType_SwitchProController);
        EXPECT_EQ(deviceInfo.interfaceType, ControllerInterfaceType::ControllerInterfaceType_Bluetooth);
        for (int j = 0; j < 3; j++)
        {
            EXPECT_EQ(deviceInfo.mainColor[j], 130);
            EXPECT_EQ(deviceInfo.subColor[j], 15);
        }
    }

    // Debug Pad のデバイス情報取得
    for (int i = 0; i < g_DebugControllerCount; i++)
    {
        int id;
        EXPECT_EQ(ControllerResult::ControllerResult_Success,
            hidInputerWrapper.m_AddDebugControllerFunction(&id));

        ControllerDeviceInfo deviceInfo;
        EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
            hidInputerWrapper.m_GetConnectedControllerDeviceInfoFunction(&deviceInfo, id));
    }

    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

TEST(HidInputerSuite, VirtualControllerButtonTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);


    // 接続前の確認
    for (int id = 0; id < g_ControllerIndexCount; id++)
    {
        const int buttonCount = sizeof(g_AllVirtualControllerButton) / sizeof(Button);
        for (int i = 0; i < buttonCount; i++)
        {
            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_PressButtonFunction(id,
                g_AllVirtualControllerButton[i]));

            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_ReleaseButtonFunction(id,
                g_AllVirtualControllerButton[i]));

            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_PushButtonFunction(id,
                g_AllVirtualControllerButton[i], 10));
        }
    }

    // Switch Pro コントローラー
    {
        Button invalidButtonList[] = {
            Button::Button_SL, Button::Button_SR
        };
        VirtualControllerButtonTestHelper(hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_SwitchProController,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerButton,
            sizeof(g_AllVirtualControllerButton) / sizeof(Button),
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button));
    }

    // 右ジョイコン
    {
        Button invalidButtonList[] = {
            Button::Button_Capture, Button::Button_StickL,
            Button::Button_L, Button::Button_ZL,
            Button::Button_Down, Button::Button_Left,
            Button::Button_Right, Button::Button_Up,
            Button::Button_Minus,
        };

        VirtualControllerButtonTestHelper(hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_JoyConRight,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerButton,
            sizeof(g_AllVirtualControllerButton) / sizeof(Button),
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button));
    }
    // 右ジョイコン（レール接続）
    {
        Button invalidButtonList[] = {
            Button::Button_SL, Button::Button_SR
        };
        VirtualControllerButtonTestHelper(hidInputerWrapper,
            1,
            ControllerDeviceType::ControllerDeviceType_JoyConRight,
            ControllerInterfaceType::ControllerInterfaceType_Rail,
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button),
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button));
    }
    // 左ジョイコン
    {
        Button invalidButtonList[] = {
            Button::Button_A, Button::Button_B,
            Button::Button_X, Button::Button_Y,
            Button::Button_Home, Button::Button_Plus,
            Button::Button_StickR, Button::Button_R,
            Button::Button_ZR,
        };

        VirtualControllerButtonTestHelper(hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_JoyConLeft,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerButton,
            sizeof(g_AllVirtualControllerButton) / sizeof(Button),
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button));
    }
    // 左ジョイコン（レール接続）
    {
        Button invalidButtonList[] = {
            Button::Button_SL, Button::Button_SR
        };
        VirtualControllerButtonTestHelper(hidInputerWrapper,
            1,
            ControllerDeviceType::ControllerDeviceType_JoyConLeft,
            ControllerInterfaceType::ControllerInterfaceType_Rail,
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button),
            invalidButtonList,
            sizeof(invalidButtonList) / sizeof(Button));
    }

    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

TEST(HidInputerSuite, VirtualControllerStickTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);


    // 接続前の確認
    for (int id = 0; id < g_ControllerIndexCount; id++)
    {
        const int stickCount = sizeof(g_AllVirtualControllerStick) / sizeof(Stick);
        for (int i = 0; i < stickCount; i++)
        {
            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_HoldAnalogStickFunction(
                    id, g_AllVirtualControllerStick[i], i * 40.0f, 1.0f));

            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_ReleaseAnalogStickFunction(
                    id, g_AllVirtualControllerStick[i]));

            EXPECT_EQ(ControllerResult::ControllerResult_InvalidControllerId,
                hidInputerWrapper.m_TiltAnalogStickFunction(
                    id, g_AllVirtualControllerStick[i], -i * 40.0f, 1.0f, 100));
        }
    }

    // Switch Pro コントローラー
    {
        VirtualControllerStickTestHelper(
            hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_SwitchProController,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerStick,
            sizeof(g_AllVirtualControllerStick) / sizeof(Stick),
            NULL,
            0);
    }

    // 右ジョイコン
    {
        Stick invalidStickList[] = {
            Stick::Stick_L
        };

        VirtualControllerStickTestHelper(
            hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_JoyConRight,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerStick,
            sizeof(g_AllVirtualControllerStick) / sizeof(Stick),
            invalidStickList,
            sizeof(invalidStickList) / sizeof(Stick));
    }

    // 左ジョイコン
    {
        Stick invalidStickList[] = {
            Stick::Stick_R
        };

        VirtualControllerStickTestHelper(
            hidInputerWrapper,
            g_ControllerIndexCount,
            ControllerDeviceType::ControllerDeviceType_JoyConLeft,
            ControllerInterfaceType::ControllerInterfaceType_Bluetooth,
            g_AllVirtualControllerStick,
            sizeof(g_AllVirtualControllerStick) / sizeof(Stick),
            invalidStickList,
            sizeof(invalidStickList) / sizeof(Stick));
    }

    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}
