﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text.RegularExpressions;
using System.Threading;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace ProdTestTool
{
    public class Download
    {
        [XmlRoot("VersionInfo")]
        public class VersionInfoModel
        {
            [XmlElement("Type")]
            public string Type { get; set; }

            [XmlElement("Id")]
            public string Id { get; set; }

            [XmlElement("Version")]
            public string Version { get; set; }

            [XmlIgnore]
            public int LatestVersion { get; set; }
        }

        [XmlRoot("RomInfo")]
        public class RomInfoModel
        {
            [XmlElement("RomId")]
            public string RomId { get; set; }

            [XmlElement("VersionInfo")]
            public List<VersionInfoModel> VersionInfos { get; set; }
        }

        public class VersionInfoWithRomId : VersionInfoModel
        {
            public string RomId { get; set; }

            public VersionInfoWithRomId(VersionInfoModel versionInfo, string romId)
            {
                this.Type = versionInfo.Type;
                this.Id = versionInfo.Id;
                this.Version = versionInfo.Version;
                this.RomId = romId;
            }
        }

        public class GroupedVersionInfo
        {
            public List<VersionInfoWithRomId> Application { get; set; }
            public List<VersionInfoWithRomId> Patch { get; set; }
            public List<VersionInfoWithRomId> AddOnContent { get; set; }
            public List<VersionInfoWithRomId> OnCardPatch { get; set; }
            public List<VersionInfoWithRomId> OnCardAoc { get; set; }

            public bool HasPatch { get { return Patch.Count != 0; } }
            public bool HasAddOnContent { get { return AddOnContent.Count != 0; } }
            public bool HasOnCardPatch { get { return OnCardPatch.Count != 0; } }
            public bool HasOnCardAoc { get { return OnCardAoc.Count != 0; } }

            public GroupedVersionInfo()
            {
                Application = new List<VersionInfoWithRomId>();
                Patch = new List<VersionInfoWithRomId>();
                AddOnContent = new List<VersionInfoWithRomId>();
                OnCardPatch = new List<VersionInfoWithRomId>();
                OnCardAoc = new List<VersionInfoWithRomId>();
            }
        }

        internal static ModelType ReadXml<ModelType>(string xmlPath)
        {
            var serializer = new XmlSerializer(typeof(ModelType));
            ModelType model;
            using (var fs = new FileStream(xmlPath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                model = (ModelType)serializer.Deserialize(fs);
            }
            return model;
        }

        public static List<RomInfoModel> GetRomInfoList()
        {
            var romInfos = new List<RomInfoModel>();
            foreach (var romId in Directory.EnumerateDirectories(Constant.OutDir + "/info").Select(x => Path.GetFileName(x)))
            {
                var romInfoXmlPath = Common.MakeInfoDir(romId) + "/dump/rominfo.nsp.xml";
                var romInfo = ReadXml<RomInfoModel>(romInfoXmlPath);
                bool isOnCardPatch = File.Exists(Common.MakeInfoDir(romId) + "/thisIsOnCardPatch");
                bool isOnCardAoc = File.Exists(Common.MakeInfoDir(romId) + "/dump/oncardaoc.xml");

                int latestVersion = 0;
                if (isOnCardAoc)
                {
                    var infoxml = XDocument.Load(Common.MakeInfoDir(romId) + "/dump/oncardaoc.xml");
                    var patchVersion = infoxml.Descendants("Patch").SelectMany(x => x.Descendants("Version"));
                    if (patchVersion.Any())
                    {
                        latestVersion = patchVersion.Select(x => Convert.ToInt32(x.Value, 10)).Max();
                    }
                }

                for (int i = 0; i < romInfo.VersionInfos.Count; i++)
                {
                    var versionInfo = romInfo.VersionInfos[i];
                    if (versionInfo.Type == "Patch" && isOnCardPatch)
                    {
                        versionInfo.Type = "OnCardPatch";
                    }
                    else if (versionInfo.Type == "AddOnContent" && isOnCardAoc)
                    {
                        versionInfo.Type = "OnCardAoc";
                        versionInfo.LatestVersion = latestVersion;
                    }
                    romInfo.VersionInfos[i] = versionInfo;
                }

                romInfos.Add(romInfo);
            }
            return romInfos;
        }

        private static string ToApplicationId(string type, string id)
        {
            if (type == "Application")
            {
                return id;
            }

            var idValue = Convert.ToUInt64(id, 16);
            if (type == "Patch" || type == "OnCardPatch")
            {
                idValue -= 0x800;
            }
            else if (type == "AddOnContent" || type == "OnCardAoc")
            {
                idValue -= 0x1000;
                idValue &= ~(0x1000UL - 1);
            }
            else
            {
                throw new NotImplementedException();
            }
            return "0x" + idValue.ToString("x16");
        }

        public static Dictionary<string, GroupedVersionInfo> GroupRomInfoList(List<RomInfoModel> romInfos)
        {
            var groupedVersionInfos = new Dictionary<string, GroupedVersionInfo>();
            foreach (var romInfo in romInfos)
            {
                foreach (var versionInfo in romInfo.VersionInfos)
                {
                    var appId = ToApplicationId(versionInfo.Type, versionInfo.Id);
                    GroupedVersionInfo groupedVersionInfo;
                    if (groupedVersionInfos.ContainsKey(appId))
                    {
                        groupedVersionInfo = groupedVersionInfos[appId];
                    }
                    else
                    {
                        groupedVersionInfo = new GroupedVersionInfo();
                    }

                    switch (versionInfo.Type)
                    {
                        case "Application":
                            groupedVersionInfo.Application.Add(new VersionInfoWithRomId(versionInfo, romInfo.RomId));
                            break;
                        case "Patch":
                            groupedVersionInfo.Patch.Add(new VersionInfoWithRomId(versionInfo, romInfo.RomId));
                            break;
                        case "AddOnContent":
                            groupedVersionInfo.AddOnContent.Add(new VersionInfoWithRomId(versionInfo, romInfo.RomId));
                            break;
                        case "OnCardPatch":
                            groupedVersionInfo.OnCardPatch.Add(new VersionInfoWithRomId(versionInfo, romInfo.RomId));
                            break;
                        case "OnCardAoc":
                            groupedVersionInfo.OnCardAoc.Add(new VersionInfoWithRomId(versionInfo, romInfo.RomId));
                            break;
                        default:
                            throw new NotImplementedException();
                    }

                    groupedVersionInfos[appId] = groupedVersionInfo;
                }
            }
            return groupedVersionInfos;
        }

        public static void MakeDataBase()
        {
            var romInfos = GetRomInfoList();
            var outPath = Constant.OutDir + "/db.txt";

            using (var fs = new FileStream(outPath, FileMode.Create))
            using (var sw = new StreamWriter(fs))
            {
                var groupedVersionInfos = GroupRomInfoList(romInfos);
                Action<List<VersionInfoWithRomId>> writeRomId = (versionInfos) =>
                {
                    foreach (var versionInfo in versionInfos)
                    {
                        sw.WriteLine("    - id {0}, version {1, -8}, romId {2}{3}", versionInfo.Id, versionInfo.Version, versionInfo.RomId, versionInfo.LatestVersion != 0 ? versionInfo.LatestVersion.ToString() : string.Empty);
                    }
                };
                foreach (var appId in groupedVersionInfos.Keys)
                {
                    var groupedVersionInfo = groupedVersionInfos[appId];
                    sw.Write("{0} :", appId);
                    if (groupedVersionInfo.Application.Count != 0)
                    {
                        sw.WriteLine(" romId {0}", groupedVersionInfo.Application.First().RomId);
                    }
                    else
                    {
                        sw.WriteLine(string.Empty);
                    }
                    if (groupedVersionInfo.HasPatch)
                    {
                        sw.WriteLine("    {0} Patches", groupedVersionInfo.Patch.Count);
                        writeRomId(groupedVersionInfo.Patch);
                    }
                    if (groupedVersionInfo.HasAddOnContent)
                    {
                        sw.WriteLine("    {0} Aocs", groupedVersionInfo.AddOnContent.Count);
                        writeRomId(groupedVersionInfo.AddOnContent);
                    }
                    if (groupedVersionInfo.HasOnCardPatch)
                    {
                        sw.WriteLine("    {0} Patches (Card)", groupedVersionInfo.OnCardPatch.Count);
                        writeRomId(groupedVersionInfo.OnCardPatch);
                    }
                    if (groupedVersionInfo.HasOnCardAoc)
                    {
                        sw.WriteLine("    {0} Aocs (Card)", groupedVersionInfo.OnCardAoc.Count);
                        writeRomId(groupedVersionInfo.OnCardAoc);
                    }
                    sw.WriteLine(string.Empty);
                }
            }
        }

        private static string GetNspName(string romId)
        {
            var dirPath = Constant.RomRootDir + string.Format("/{0}", romId);
            var args = string.Format("shares browse -v -i -H{0} -u{1} -p{2} -T{3} -P\"{4}\"", Constant.HostName, Constant.UserName, Constant.PassWord, Constant.Port, dirPath);
            var result = Common.ExecuteCommand(Constant.Aspera, args);
            using (var sr = new StringReader(result))
            {
                while (sr.Peek() > -1)
                {
                    var line = sr.ReadLine();
                    if (!line.StartsWith("-v--"))
                    {
                        continue;
                    }
                    var pattern = @"(-v--)(.*)(\s+)(?<nspName>.*)(\.nsp$)";
                    if (Regex.IsMatch(line, pattern))
                    {
                        var match = Regex.Match(line, pattern);
                        var nspName = match.Groups["nspName"].Value;
                        return nspName + ".nsp";
                    }
                    else
                    {
                        throw new InvalidOperationException();
                    }
                }
                throw new InvalidOperationException();
            }
        }

        private static void ExecuteDownload(string srcPath, string dstPath)
        {
            var nspName = Path.GetFileName(srcPath);
            if (File.Exists(dstPath + "/" + nspName))
            {
                return;
            }
            var args = string.Format("shares download -v -i -H{0} -u{1} -p{2} -T{3} -s\"{4}\" -d\"{5}\"", Constant.HostName, Constant.UserName, Constant.PassWord, Constant.Port, srcPath, dstPath);
            Common.ExecuteCommand(Constant.Aspera, args);
        }

        private static void ExecuteDownloadDummy(string srcPath, string dstPath)
        {
            var nspName = Path.GetFileName(srcPath);
            if (File.Exists(dstPath + "/" + nspName))
            {
                return;
            }
            Directory.CreateDirectory(dstPath);
            File.Create(dstPath + "/" + nspName);
        }

        public static void DownloadNsp(string[] appIds)
        {
            var romInfos = GetRomInfoList();
            var groupedVersionInfos = GroupRomInfoList(romInfos);

            Func<string, string> getSrcPath = (romId) =>
            {
                // GetNspName で失敗することがあるのでリトライする
                const int RetryCountMax = 10;
                const int RetryWaitMilliSec = 120000;
                int retryCount = 0;
                string ret = string.Empty;
                while (retryCount < RetryCountMax)
                {
                    try
                    {
                        ret = Constant.RomRootDir + string.Format("/{0}/{1}", romId, GetNspName(romId));
                        break;
                    }
                    catch
                    {
                        Console.WriteLine("Retry #{0} DownloadNsp() after 120 sec.", retryCount);
                        Thread.Sleep(RetryWaitMilliSec);
                        retryCount++;
                    }
                }
                if (retryCount == RetryCountMax)
                {
                    throw new InvalidOperationException("Failed DownloadNsp()");
                }
                return ret;
            };

            Func<string, string, string> getDstPath = (appId, type) =>
            {
                return Constant.OutDir + string.Format("/dev/{0}/{1}", appId, type);
            };

            foreach (var appId in appIds)
            {
                var groupedVersionInfo = groupedVersionInfos[appId];

                if (groupedVersionInfo.Application.Count == 0)
                {
                    // 主タイトルが存在しない。テストデータと思われるので Skip
                    Console.WriteLine("Skip {0} because it has no application.", appId);
                    return;
                }

                // アプリは 1 つだけしか使わない（UPP 差し替え等で複数エントリあることがある）
                {
                    var app = groupedVersionInfo.Application.First();
                    var srcPath = getSrcPath(app.RomId);
                    var dstPath = getDstPath(appId, "app");
                    ExecuteDownload(srcPath, dstPath);
                }
                foreach (var patch in groupedVersionInfo.Patch)
                {
                    var srcPath = getSrcPath(patch.RomId);
                    var dstPath = getDstPath(appId, "patch");
                    ExecuteDownload(srcPath, dstPath);
                }
                foreach (var aoc in groupedVersionInfo.AddOnContent)
                {
                    var srcPath = getSrcPath(aoc.RomId);
                    var dstPath = getDstPath(appId, "aoc");
                    ExecuteDownload(srcPath, dstPath);
                }
                foreach (var oncardpatch in groupedVersionInfo.OnCardPatch)
                {
                    // OnCardPatch はダウンロードせず空ファイル作成
                    var srcPath = getSrcPath(oncardpatch.RomId);
                    var dstPath = getDstPath(appId, "oncardpatch");
                    ExecuteDownloadDummy(srcPath, dstPath);
                }
                foreach (var oncardaoc in groupedVersionInfo.OnCardAoc)
                {
                    var srcPath = getSrcPath(oncardaoc.RomId);
                    var dstPath = getDstPath(appId, "oncardaoc");
                    ExecuteDownload(srcPath, dstPath);
                }
            }
        }
    }
}
