﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Pages
{
    public class PageValidationsTest : SharedTextContextBase
    {
        public PageValidationsTest(TestContextFixture fixture) : base(fixture)
        {

        }

        [Fact]
        public void DefaultCtor()
        {
            _context.DiContainer.GetInstance<PageValidations>();
        }

        public static IEnumerable<object> ApplicationIdTestData
        {
            get
            {
                var rangeOver =
                    string.Format(Resources.ApplicationId_Validate_RangeOver,
                        Constants.ProgramIdMinimum.ToUpperHex(), Constants.ProgramIdMaximum.ToUpperHex(), 0, false);

                yield return new object[] {null, 0, true};
                yield return new object[] {rangeOver, 0, false};

                yield return new object[] {rangeOver, Constants.ProgramIdMinimum - 1, false};
                yield return new object[] {null, Constants.ProgramIdMinimum - 1, true};

                yield return new object[] {null, Constants.ProgramIdMinimum, false};
                yield return new object[] {null, Constants.ProgramIdMinimum, true};

                yield return new object[] {null, Constants.ProgramIdMinimum + 1, false};
                yield return new object[] {null, Constants.ProgramIdMinimum + 1, true};

                yield return new object[] {null, Constants.ProgramIdMaximum - 1, false};
                yield return new object[] {null, Constants.ProgramIdMaximum - 1, true};

                yield return new object[] {null, Constants.ProgramIdMaximum, false};
                yield return new object[] {null, Constants.ProgramIdMaximum, true};

                yield return new object[] {rangeOver, Constants.ProgramIdMaximum + 1, false};
                yield return new object[] {null, Constants.ProgramIdMaximum + 1, true};
            }
        }

        [Theory]
        [MemberData(nameof(ApplicationIdTestData))]
        public void ApplicationId(string actual, ulong id, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta {IsReadOnly = isReadOnly};

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.ApplicationId(id));
        }

        public static IEnumerable<object> DisplayVersionTestData
        {
            get
            {
                var maxLength =
                    string.Format(Resources.DisplayVersion_Validate_Length,
                        Application.MaxDisplayVersionLength);

                yield return new object[] {null, "a", true};
                yield return new object[] {null, "a", false};

                yield return new object[] {null, string.Empty, true};
                yield return new object[] {Resources.DisplayVersion_Validate_Empty, string.Empty, false};

                yield return new object[] {null, null, true};
                yield return new object[] {Resources.DisplayVersion_Validate_Empty, null, false};

                yield return new object[] {null, "01234567890123456789", true};
                yield return new object[] {maxLength, "01234567890123456789", false};

                yield return new object[] {null, "あ", true};
                yield return new object[] {Resources.DisplayVersion_Validate_ASCII, "あ", false};
            }
        }

        [Theory]
        [MemberData(nameof(DisplayVersionTestData))]
        public void DisplayVersion(string actual, string id, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta {IsReadOnly = isReadOnly};

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.DisplayVersion(id));
        }

        public static IEnumerable<object> CardSizeTestData
        {
            get
            {
                var validSize = new HashSet<int> {2, 4, 8, 16, 32};

                var sample = new[] {0, 1, 2, 3, 4, 5, 6, 7, 8, 15, 16, 17, 30, 31, 32, 33, 34};

                foreach (var size in sample)
                {
                    if (validSize.Contains(size))
                    {
                        yield return new object[] {null, size, false, false};
                        yield return new object[] {null, size, false, true};
                    }
                    else if (size == 1)
                    {
                        yield return new object[] {Resources.CardSpec_Size_Validate_UnsupportedSizeError, size, false, false};
                    }
                    else
                    {
                        yield return new object[] {Resources.CardSpec_Size_Validate_SizeError, size, false, false};
                        yield return new object[] {null, size, false, true};
                    }

                    // auto
                    yield return new object[] {null, size, true, false};
                    yield return new object[] {null, size, true, true};
                }
            }
        }

        [Theory]
        [MemberData(nameof(CardSizeTestData))]
        public void CardSize(string actual, int size, bool isAutomaticSetting, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                CardSpec =
                {
                    IsAutomaticSettingSize = isAutomaticSetting,
                    Size = size
                }
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.CardSize(project.Meta));
        }

        public static IEnumerable<object> CardClockRateTestData
        {
            get
            {
                var validClock = new HashSet<int> {25, 50};

                var sample = new[] {0, 24, 25, 26, 49, 50, 51, 100, 200};

                foreach (var clock in sample)
                {
                    if (validClock.Contains(clock))
                    {
                        yield return new object[] {null, clock, 8, false, false};
                        yield return new object[] {null, clock, 8, false, true};
                    }
                    else
                    {
                        yield return
                            new object[] {Resources.CardSpec_ClockRate_Validate_ClockRateError, clock, 8, false, false};
                        yield return new object[] {null, clock, 8, false, true};
                    }

                    // auto
                    yield return new object[] {null, clock, 8, true, false};
                    yield return new object[] {null, clock, 8, true, true};
                }
            }
        }

        public static IEnumerable<object> CardClockRateUnder8GTestData
        {
            get
            {
                yield return new object[] {null, 25, 4, false, false};
                yield return new object[] {null, 25, 4, false, true};

                yield return new object[] {Resources.CardSpec_ClockRate_Validate_Under8GB, 50, 4, false, false};
                yield return new object[] {null, 50, 4, false, true};

                var sample = new[] {25, 50};

                foreach (var clock in sample)
                {
                    // auto
                    yield return new object[] {null, clock, 4, true, false};
                    yield return new object[] {null, clock, 4, true, true};
                }
            }
        }

        [Theory]
        [MemberData(nameof(CardClockRateTestData))]
        [MemberData(nameof(CardClockRateUnder8GTestData))]
        public void CardClockRate(string actual, int clock, int size, bool isAutomaticSetting, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                CardSpec =
                {
                    IsAutomaticSettingClockRate = isAutomaticSetting,
                    ClockRate = clock,
                    Size = size
                }
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.CardClockRate(project.Meta));
        }


        public static IEnumerable<object> MainThreadStackSizeTestData
        {
            get
            {
                yield return new object[] {null, 0, true};
                yield return new object[] {Resources.MainThreadStackSize_Validate_Empty, 0, false};

                yield return new object[] {null, 4096 - 1, true};
                yield return new object[] {Resources.MainThreadStackSize_Validate_Align, 4096 - 1, false};

                yield return new object[] {null, 4096, true};
                yield return new object[] {null, 4096, false};

                yield return new object[] {null, 4096 + 1, true};
                yield return new object[] {Resources.MainThreadStackSize_Validate_Align, 4096 + 1, false};
            }
        }

        [Theory]
        [MemberData(nameof(MainThreadStackSizeTestData))]
        public void MainThreadStackSize(string actual, ulong size, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta {IsReadOnly = isReadOnly};

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.MainThreadStackSize(size));
        }

        public static IEnumerable<object> ApplicationErrorCodeCategoryTestData
        {
            get
            {
                var maxLength =
                    string.Format(Resources.ApplicationErrorCodeCategory_Validate_Length,
                        Application.MaxApplicationErrorCodeCategoryLength);

                // use
                yield return new object[] {Resources.PleaseSet, true, string.Empty, false};
                yield return new object[] {null, true, string.Empty, true};
                yield return new object[] {Resources.PleaseSet, true, null, false};
                yield return new object[] {null, true, null, true};
                yield return new object[] {null, true, "01234", false};
                yield return new object[] {null, true, "01234", true};
                yield return new object[] {maxLength, true, "0123456789", false};
                yield return new object[] {null, true, "0123456789", true};

                // no use
                yield return new object[] {null, false, string.Empty, false};
                yield return new object[] {null, false, string.Empty, true};
                yield return new object[] {null, false, null, false};
                yield return new object[] {null, false, null, true};
                yield return new object[] {null, false, "01234", false};
                yield return new object[] {null, false, "01234", true};
                yield return new object[] {null, false, "0123456789", false};
                yield return new object[] {null, false, "0123456789", true};
            }
        }

        [Theory]
        [MemberData(nameof(ApplicationErrorCodeCategoryTestData))]
        public void ApplicationErrorCodeCategory(string actual, bool isUse, string category, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                Application = {IsUseApplicationErrorCode = isUse}
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.ApplicationErrorCodeCategory(project.Meta, category));
        }

        public static IEnumerable<object> DirectoryPathTestData
        {
            get
            {
                // replace
                yield return new object[] {null, true, @".", false};
                yield return new object[] {null, true, @".", true};
                yield return new object[] {Resources.DirectoryNotFound, true, "zzz:zzz", false};
                yield return new object[] {null, true, "zzz:zzz", true};
                yield return new object[] {Resources.PleaseSet, true, string.Empty, false};
                yield return new object[] {null, true, string.Empty, true};
                yield return new object[] {Resources.PleaseSet, true, null, false};
                yield return new object[] {null, true, null, true};

                // no replace
                yield return new object[] {null, false, "zzz:zzz", false};
                yield return new object[] {null, false, "zzz:zzz", true};
                yield return new object[] {null, false, string.Empty, false};
                yield return new object[] {null, false, string.Empty, true};
                yield return new object[] {null, false, null, false};
                yield return new object[] {null, false, null, true};
            }
        }

        public static IEnumerable<object> FilePathTestData
        {
            get
            {
                // replace
                yield return new object[] {null, true, @"C:\Windows\explorer.exe", false};
                yield return new object[] {null, true, @"C:\Windows\explorer.exe", true};
                yield return new object[] {Resources.FileNotFound, true, "zzz:zzz", false};
                yield return new object[] {null, true, "zzz:zzz", true};
                yield return new object[] {Resources.PleaseSet, true, string.Empty, false};
                yield return new object[] {null, true, string.Empty, true};
                yield return new object[] {Resources.PleaseSet, true, null, false};
                yield return new object[] {null, true, null, true};

                // no replace
                yield return new object[] {null, false, "zzz:zzz", false};
                yield return new object[] {null, false, "zzz:zzz", true};
                yield return new object[] {null, false, string.Empty, false};
                yield return new object[] {null, false, string.Empty, true};
                yield return new object[] {null, false, null, false};
                yield return new object[] {null, false, null, true};
            }
        }

        [Theory]
        [MemberData(nameof(DirectoryPathTestData))]
        public void HtmlDocumentPath(string actual, bool isReplace, string path, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                Application =
                {
                    IsReplaceHtmlDocumentPath = isReplace,
                    HtmlDocumentPath =  path
                }
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.HtmlDocumentPath(project.Meta.Application));
        }

        [Theory]
        [MemberData(nameof(FilePathTestData))]
        public void AccessibleUrlsFilePath(string actual, bool isReplace, string path, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                Application =
                {
                    IsReplaceAccessibleUrlsFilePath = isReplace,
                    AccessibleUrlsFilePath = path
                }
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.AccessibleUrlsFilePath(project.Meta.Application));
        }

        [Theory]
        [MemberData(nameof(FilePathTestData))]
        public void LegalInformationFilePath(string actual, bool isReplace, string path, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                Application =
                {
                    IsReplaceLegalInformationFilePath = isReplace,
                    LegalInformationFilePath = path
                }
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.LegalInformationFilePath(project.Meta.Application));
        }

        public static IEnumerable<object> SupportedLanguagesInMetaTestData
        {
            get
            {
                yield return new object[] {null, new[]
                {
                    new SupportedLanguage
                    {
                        Language  = LanguageType.AmericanEnglish,
                        IsSupported  = true,
                    }
                }, false};
                yield return new object[] {null, new[]
                {
                    new SupportedLanguage
                    {
                        Language  = LanguageType.AmericanEnglish,
                        IsSupported  = true,
                    }
                }, true};

                yield return new object[] {Resources.SupportedLanguags_Validate_Empty, new[]
                {
                    new SupportedLanguage
                    {
                        Language  = LanguageType.AmericanEnglish,
                        IsSupported  = false,
                    }
                }, false};
                yield return new object[] {null, new[]
                {
                    new SupportedLanguage
                    {
                        Language  = LanguageType.AmericanEnglish,
                        IsSupported  = false,
                    }
                }, true};
            }
        }

        [Theory]
        [MemberData(nameof(SupportedLanguagesInMetaTestData))]
        public void SupportedLanguagesInMeta(string actual, IEnumerable<SupportedLanguage> supportedLanguages, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
                Application = {SupportedLanguages = new ObservableCollection<SupportedLanguage>(supportedLanguages)}
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.SupportedLanguagesInMeta(project.Meta));
        }

        public static IEnumerable<object> TitleTestData
        {
            get
            {
                yield return new object[] {null, "a", false};
                yield return new object[] {null, "a", true};

                yield return new object[] {Resources.PleaseSet, string.Empty, false};
                yield return new object[] {null, string.Empty, true};
                yield return new object[] {Resources.PleaseSet, null, false};
                yield return new object[] {null, null, true};

                yield return new object[] {null, new string('x', 127), false};
                yield return new object[] {null, new string('x', 127), true};
                yield return new object[] {string.Format(Resources.Title_Name_Validate_Length, Title.MaxNameLength), new string('x', 128), false};
                yield return new object[] {null, new string('x', 128), true};
            }
        }

        [Theory]
        [MemberData(nameof(TitleTestData))]
        public void TitleName(string actual, string title, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.TitleName(title));
        }

        public static IEnumerable<object> PublisherTestData
        {
            get
            {
                yield return new object[] {null, "a", false};
                yield return new object[] {null, "a", true};

                yield return new object[] {Resources.PleaseSet, string.Empty, false};
                yield return new object[] {null, string.Empty, true};
                yield return new object[] {Resources.PleaseSet, null, false};
                yield return new object[] {null, null, true};

                yield return new object[] {null, new string('x', 63), false};
                yield return new object[] {null, new string('x', 63), true};
                yield return new object[] {string.Format(Resources.Title_Publisher_Validate_Length, Title.MaxPublisherLength), new string('x', 64), false};
                yield return new object[] {null, new string('x', 64), true};
            }
        }

        [Theory]
        [MemberData(nameof(PublisherTestData))]
        public void PublisherName(string actual, string title, bool isReadOnly)
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.Meta = new ApplicationMeta
            {
                IsReadOnly = isReadOnly,
            };

            var pv = _context.DiContainer.GetInstance<PageValidations>();

            Assert.Equal(actual, pv.PublisherName(title));
        }

    }
}
