﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Pages
{
    public class CardSpecPageVmTest : IDisposable
    {
        private TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Theory]
        [InlineData(false)]
        [InlineData(true)]
        public void DefaultCtor(bool canEditCardSpec)
        {
            var profile = _context.DiContainer.GetInstance<AppProfile>();

            profile.CanEditCardSpec = canEditCardSpec;

            using (var vm = _context.DiContainer.GetInstance<CardSpecPageVm>())
            {
                Assert.NotEmpty(vm.Params);
            }
        }

        [Fact]
        public void CardSizeValidations()
        {
            using (var tempDir = new DisposableDirectory())
            {
                var tempNmetaFile = tempDir.CreateFile("temp.nmeta");

                _context.DiContainer.GetInstance<AppProfile>().CanEditCardSpec = true;

                using (var project = new Project
                {
                    DiContainer = _context.DiContainer,
                    Meta = new ApplicationMeta { DiContainer = _context.DiContainer }
                })
                using (var vm = new CardSpecPageVm(_context.DiContainer, project.Meta))
                {
                    var meta = project.Meta;

                    meta.Application.DisplayVersion = "1.0.0";
                    meta.Application.SupportedLanguages.First().IsSupported = true;
                    meta.Application.Titles.Add(new Title {Name = "name", Publisher = "publisher", IsReplaceIcon = false});

                    Assert.NotNull(vm.ClockParamVm);
                    Assert.NotNull(vm.SizeParamVm);

                    var cardSpec = meta.CardSpec;

                    Assert.False(vm.HasErrors.Value);
                    Assert.False(meta.HasErrors);

                    cardSpec.IsAutomaticSettingSize = false;
                    cardSpec.Size = 1;
                    Assert.True(vm.HasErrors.Value);
                    Assert.True(meta.HasErrors);

                    cardSpec.IsAutomaticSettingSize = true;
                    Assert.False(vm.HasErrors.Value);
                    Assert.False(meta.HasErrors);

                    cardSpec.IsAutomaticSettingSize = false;
                    cardSpec.Size = 2;
                    Assert.False(vm.HasErrors.Value);
                    Assert.False(meta.HasErrors);

                    // エラーを含む状態で nmeta を出力
                    cardSpec.Size = 1;
                    project.OutputAppMetaXmlFileForAuthoringTool(tempNmetaFile);
                }

                _context.Dispose();
                _context = new TestContext();
                _context.DiContainer.GetInstance<AppProfile>().CanEditCardSpec = true;

                using (var project = Project.Import(_context.DiContainer, ImportableFileType.Meta, tempNmetaFile))
                using (var vm = new CardSpecPageVm(_context.DiContainer, project.Meta))
                {
                    // エラーを含む nmeta を読み込んだ後、エラーを訂正して保存できることを確認
                    Assert.Equal(1, project.Meta.CardSpec.Size);
                    Assert.True(vm.HasErrors.Value);
                    Assert.True(project.Meta.HasErrors);

                    project.Meta.CardSpec.Size = 2;

                    Assert.False(vm.HasErrors.Value);
                    Assert.False(project.Meta.HasErrors);
                }
            }
        }
    }
}
