﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reactive.Concurrency;
using System.Text;
using System.Threading;
using System.Xml.Linq;
using BezelEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class TestContext : IDisposable
    {
        private readonly DisposableDirectory _tempDir = new DisposableDirectory();

        public Container DiContainer { get; }= new Container();
        public string TempDirPath { get; set; }

        public static readonly string RootPath = "../../../../../../../".ToPathString().CreateAbsolutePath(Directory.GetCurrentDirectory().ToDirectoryPathString());
        public static readonly string TestDataDirPath = Path.Combine(RootPath, "Tests/Tools/Sources/Tests/AuthoringEditorTest/TestData/").ToPathString();

        public static string DefaultMetaFilePath =>
            NintendoSdkHelper.ApplicationMetaFilePath;

        public static string DefaultDescFilePath =>
            NintendoSdkHelper.ApplicationDescFilePath;

        public TestContext()
        {
            CleanTestEnv();

            DiContainer.RegisterSingleton<ApplicationCapability>();
            DiContainer.RegisterSingleton<App>();
            DiContainer.RegisterSingleton<Config>();
            DiContainer.RegisterSingleton<AppProfile>();

            DiContainer.Register(() => new Project());

            TempDirPath = _tempDir.RootPath;
        }

        public void Dispose()
        {
            DiContainer.Dispose();
            _tempDir.Dispose();
        }

        public T GetInstance<T>()where T : class
        {
            return DiContainer.GetInstance<T>();
        }

        private static int _isCleaned;

        public static void CleanTestEnv()
        {
            if (Interlocked.Increment(ref _isCleaned) != 1)
                return;

            ReactivePropertyScheduler.SetDefault(ImmediateScheduler.Instance);
        }

        public string CreateFile(string path)
        {
            _tempDir.CreateFile(path);
            return Path.Combine(_tempDir.RootPath, path);
        }

        public string CreateFolder(string path)
        {
            _tempDir.CreateFolder(path);
            return Path.Combine(_tempDir.RootPath, path);
        }

        public static string NormalizeXml(string srcXml)
        {
            return XDocument.Parse(srcXml).ToString();
        }

        public string MakeAppMetaXml(ApplicationMeta appMeta)
        {
            using (var project = new Project { Meta = appMeta })
                return project.MakeAppMetaXmlForAuthoringTool();
        }

        public XDocument MakeAppMetaXmlDocument(ApplicationMeta appMeta)
        {
            return XDocument.Parse(MakeAppMetaXml(appMeta));
        }

        public bool OutputAppMetaXml(ApplicationMeta appMeta, string filePath)
        {
            using (var project = new Project {Meta = appMeta})
                return project.OutputAppMetaXmlFileForAuthoringTool(filePath);
        }

        public ApplicationMeta LoadAppMeta(string xmlText)
        {
            return LoadProjectFromXml(xmlText).Meta;
        }

        public AocMeta LoadAocMeta(string xmlText)
        {
            return LoadProjectFromXml(xmlText).AocMeta;
        }

        public Project LoadProjectFromXml(string xmlText)
        {
            var tempMetaPath = Path.Combine(TempDirPath, Path.GetRandomFileName());
            File.WriteAllText(tempMetaPath, xmlText, Encoding.UTF8);
            return Project.Import(DiContainer, ImportableFileType.Meta, tempMetaPath);
        }
    }
}
