﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures
{
    public class BuildTestAppAndPatchNspFixture : BuildTestAppNspFixture
    {
        private string CreateRevisedFilePath(string suffix)
        {
            return Path.Combine(
                Path.GetDirectoryName(NspFilePath),
                $"{Path.GetFileNameWithoutExtension(NspFilePath)}.{suffix}");
        }

        public string RevisedV2NspFilePath => CreateRevisedFilePath("v2.nsp");

        public string RevisedV3NspFilePath => CreateRevisedFilePath("v3.nsp");

        public string HtmlContainsNspFilePath => CreateRevisedFilePath("html.nsp");

        public string AccessibleUrlContainsNspFilePath => CreateRevisedFilePath("accessible-urls.nsp");

        public string HtmlAndAccessibleUrlContainsNspFilePath => CreateRevisedFilePath("html-and-accessible-urls.nsp");

        public string PatchV1NspFilePath => CreateRevisedFilePath("patch.v1.nsp");

        public string PatchV2NspFilePath => CreateRevisedFilePath("patch.v2.nsp");

        public string PatchV3NspFilePath => CreateRevisedFilePath("patch.v3.nsp");

        private static readonly SemaphoreSlim _semaphore = new SemaphoreSlim(1, 1);

        private bool _isBuildFinished;

        public BuildTestAppAndPatchNspFixture InitializePatch()
        {
            try
            {
                _semaphore.Wait();
                if (_isBuildFinished)
                    return this;
                _isBuildFinished = true;

                Task.Run(async () =>
                {
                    var project = Project.Import(_diContainer,
                        ImportableFileType.Meta,
                        TestContext.DefaultMetaFilePath);

                    var meta = project.Meta;

                    // 修正版アプリケーション (v2) プログラムの変更を含む
                    {
                        meta.Application.ReleaseVersion = 2;
                        AppBuilder.BuildTestApp(code => $"int g_ExternalLinkagleSymbol;\n{code}");
                        var output = await AppBuilder.BuildNspAsync(RevisedV2NspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    // 修正版アプリケーション (v3)
                    {
                        meta.Application.ReleaseVersion = 3;
                        var output = await AppBuilder.BuildNspAsync(RevisedV3NspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    var htmlDocumentPath = Path.Combine(TempDirPath, "html-document");
                    var accessibleUrlsPath = Path.Combine(TempDirPath, "accessible-urls.txt");

                    // 修正版アプリケーション (v4)
                    {
                        // オフライン HTML ドキュメントのみを追加
                        meta.Application.ReleaseVersion = 4;
                        {
                            Directory.CreateDirectory(htmlDocumentPath);
                            File.WriteAllText(Path.Combine(htmlDocumentPath, "hoge.html"), "<html><body>hoge</body></html>");
                            meta.Application.HtmlDocumentPath = htmlDocumentPath;
                            meta.Application.AccessibleUrlsFilePath = null;
                            meta.Application.IsReplaceHtmlDocumentPath = true;
                            meta.Application.IsReplaceAccessibleUrlsFilePath = false;
                        }
                        var output = await AppBuilder.BuildNspAsync(HtmlContainsNspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    // 修正版アプリケーション (v5)
                    {
                        // accessible-urls のみを追加
                        meta.Application.ReleaseVersion = 5;
                        {
                            File.WriteAllText(accessibleUrlsPath, "http://example.com$/\n");
                            meta.Application.HtmlDocumentPath = null;
                            meta.Application.AccessibleUrlsFilePath = accessibleUrlsPath;
                            meta.Application.IsReplaceHtmlDocumentPath = false;
                            meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                        }
                        var output = await AppBuilder.BuildNspAsync(AccessibleUrlContainsNspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    // 修正版アプリケーション (v6)
                    {
                        // オフライン HTML ドキュメント / accessible-urls 両方を追加
                        meta.Application.ReleaseVersion = 6;
                        meta.Application.HtmlDocumentPath = htmlDocumentPath;
                        meta.Application.AccessibleUrlsFilePath = accessibleUrlsPath;
                        meta.Application.IsReplaceHtmlDocumentPath = true;
                        meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                        var output = await AppBuilder.BuildNspAsync(HtmlAndAccessibleUrlContainsNspFilePath, meta).ConfigureAwait(false);
                        Assert.Equal(NspHandleResult.Ok, output.Result);
                    }

                    // パッチ
                    if (project.AppCapability.IsSupportMakingPatch)
                    {
                        // v1 パッチ
                        await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption()
                        {
                            OriginalNspFilePath = NspFilePath,
                            CurrentNspFilePath = RevisedV1NspFilePath,
                            OutputPath = PatchV1NspFilePath
                        }).ConfigureAwait(false);

                        // v2 パッチ
                        await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption()
                        {
                            OriginalNspFilePath = NspFilePath,
                            CurrentNspFilePath = RevisedV2NspFilePath,
                            PreviousNspFilePath = PatchV1NspFilePath,
                            OutputPath = PatchV2NspFilePath
                        }).ConfigureAwait(false);

                        // v3 パッチ
                        await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption()
                        {
                            OriginalNspFilePath = NspFilePath,
                            CurrentNspFilePath = RevisedV3NspFilePath,
                            PreviousNspFilePath = PatchV2NspFilePath,
                            OutputPath = PatchV3NspFilePath
                        }).ConfigureAwait(false);
                    }
                }).Wait();

                return this;
            }
            finally
            {
                _semaphore.Release();
            }
        }
    }
}
