﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// G3dStream と IntermediateFileStream で 速度を比較する場合は下のdefineをコメントアウト
// #define DO_BENCHMARK

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;

using Microsoft.VisualStudio.TestTools.UnitTesting;

using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace IfBinaryReadWriteTest
{
    [TestClass]
    public class IfBinaryReadWriteTest
    {
        public TestContext TestContext { get; set; }

        static string inFile;
        static string outOffsetFile;
        static string outStream2File;
        static string outStreamFile;
        static string benchFile;

        [ClassInitialize]
        public static void ClassInit(TestContext context)
        {
            inFile = Path.Combine(
                Environment.CurrentDirectory,
                @"../../../../../../../../",
                @"Samples\Applications\Nw4f\g3d\G3dDemo\Resources\town\bg_WhiteTown\bg_WhiteTown.fmdb");
            outOffsetFile = Path.Combine(context.TestResultsDirectory, "bg_WhiteTown_offset.fmdb");
            outStreamFile = Path.Combine(context.TestResultsDirectory, "bg_WhiteTown_stream.fmdb");
            outStream2File = Path.Combine(context.TestResultsDirectory, "bg_WhiteTown_stream2.fmdb");
            benchFile = Path.Combine(
                Environment.CurrentDirectory,
                @"../../../../../../../../",
                @"Samples\Applications\Nw4f\g3d\G3dDemo\Resources\town\bg_WhiteTown\textures\Bake_bg_WhiteTown_g00_b00_f00.ftxb");
            benchFile = inFile;
        }

        // バイナリストリームをファイルイメージへのオフセットで扱うメソッドをテスト
        [TestMethod]
        public void IfBinaryReadUtility_BinaryStreamOffset()
        {
            var fileImage = File.ReadAllBytes(inFile);
            int streamOffset;

            var file = IfBinaryReadUtility.Read(fileImage, null, out streamOffset);
            Assert.IsNotNull(file);
            IfBinaryWriteUtility.Write(file, fileImage, outOffsetFile, streamOffset);
        }

        // 高速化したG3dStream2 のテスト
        [TestMethod]
        public void IfBinaryReadUtility_G3DStream2()
        {
            var fileImage = File.ReadAllBytes(inFile);

            // 今までのG3dStreamを使うメソッドでRead,Writeする。（比較用）
            var streamArray = new List<G3dStream>();
            var fileStandard = IfBinaryReadUtility.Read(streamArray, fileImage, null);
            Assert.IsNotNull(fileStandard);
            IfBinaryWriteUtility.Write(fileStandard, streamArray, outStreamFile);

            // IntermediateFileStream のReadをテスト
            // G3dStream2を使う新しいメソッドでReadし、G3dStream で Readしたものと内容を比較する
            var stream2Array = new List<IntermediateFileStream>();
            var fileStream2 = IfBinaryReadUtility.Read(stream2Array, fileImage, null);

            CheckIntermediateFileStreamCompatibility(streamArray, stream2Array);

            // IntermediateFileStream のWriteをテスト
            // G3dStream2を使う新しいメソッドでWriteしたものをもう一度Readし、G3dStream で Readしたものと内容を比較する
            IfBinaryWriteUtility.Write(fileStream2, stream2Array, outStream2File);
            var stream2WriteReadArray = new List<IntermediateFileStream>();
            var fileStream2WriteRead = IfBinaryReadUtility.Read(stream2WriteReadArray, outStream2File, null);

            CheckIntermediateFileStreamCompatibility(streamArray, stream2WriteReadArray);
        }


        // G3dStream と IntermediateFileStream で 内容を比較する
        private static void CheckIntermediateFileStreamCompatibility(List<G3dStream> streamArray, List<IntermediateFileStream> stream2Array)
        {
            Assert.AreEqual(streamArray.Count, stream2Array.Count);
            for (var i = 0; i < streamArray.Count; i++)
            {
                var stream = streamArray[i];
                var stream2 = stream2Array[i];

                Assert.AreEqual(stream.type, stream2.type);
                Assert.AreEqual(stream.column, stream2.column);
                switch (stream.type)
                {
                    case stream_typeType.@byte:
                        {
                            var data = stream.ByteData;
                            var data2 = stream2.ByteData;
                            Assert.IsNotNull(data);
                            Assert.IsNotNull(data2);
                            Assert.AreEqual(data.Count, data2.Length);
                            for (var j = 0; j < data.Count; j++)
                            {
                                Assert.AreEqual(data[j], data2[j]);
                            }
                        }
                        break;
                    case stream_typeType.@int:
                        {
                            var data = stream.IntData;
                            var data2 = stream2.IntData;
                            Assert.IsNotNull(data);
                            Assert.IsNotNull(data2);
                            Assert.AreEqual(data.Count, data2.Length);
                            for (var j = 0; j < data.Count; j++)
                            {
                                Assert.AreEqual(data[j], data2[j]);
                            }
                        }
                        break;
                    case stream_typeType.@float:
                        {
                            var data = stream.FloatData;
                            var data2 = stream2.FloatData;
                            Assert.IsNotNull(data);
                            Assert.IsNotNull(data2);
                            Assert.AreEqual(data.Count, data2.Length);
                            for (var j = 0; j < data.Count; j++)
                            {
                                Assert.AreEqual(data[j], data2[j]);
                            }
                        }
                        break;
                    case stream_typeType.@string:
                    case stream_typeType.@wstring:
                        Assert.IsNotNull(stream.StringData);
                        Assert.IsNotNull(stream2.StringData);
                        Assert.AreEqual(stream.StringData, stream2.StringData);
                        break;
                    default:
                        Assert.Fail();
                        break;
                }
            }
        }

#if DO_BENCHMARK
        // G3dStream と IntermediateFileStream で 速度を比較する
        private static int benchCount = 200;

        [TestMethod]
        public void IfBinaryReadUtility_G3DStreamReadBench()
        {
            var fileImage = File.ReadAllBytes(benchFile);
            var streamArray = new List<G3dStream>();
            var watch = new Stopwatch();
            watch.Start();
            for (int i = 0; i < benchCount; i++)
            {
                streamArray.Clear();
                var file = IfBinaryReadUtility.Read(streamArray, fileImage, null);
            }
            watch.Stop();
            Console.WriteLine(@"G3dStream Read {0}ms", watch.ElapsedMilliseconds);
        }

        [TestMethod]
        public void IfBinaryReadUtility_G3DStreamWriteBench()
        {
            var fileImage = File.ReadAllBytes(benchFile);
            var streamArray = new List<G3dStream>();
            var file = IfBinaryReadUtility.Read(streamArray, fileImage, null);
            var watch = new Stopwatch();
            watch.Start();
            for (int i = 0; i < benchCount; i++)
            {
                IfBinaryWriteUtility.Write(file, streamArray, outStreamFile);
            }
            watch.Stop();
            Console.WriteLine(@"G3dStream Write {0}ms", watch.ElapsedMilliseconds);
        }

        [TestMethod]
        public void IfBinaryReadUtility_G3DStream2ReadBench()
        {
            var fileImage = File.ReadAllBytes(benchFile);
            var stream2Array = new List<IntermediateFileStream>();
            var watch = new Stopwatch();
            watch.Start();
            for (int i = 0; i < benchCount; i++)
            {
                stream2Array.Clear();
                var file = IfBinaryReadUtility.Read(stream2Array, fileImage, null);
            }
            watch.Stop();
            Console.WriteLine(@"IntermediateFileStream Read {0}ms", watch.ElapsedMilliseconds);
        }

        [TestMethod]
        public void IfBinaryReadUtility_G3DStream2WriteBench()
        {
            var fileImage = File.ReadAllBytes(benchFile);
            var stream2Array = new List<IntermediateFileStream>();
            var file = IfBinaryReadUtility.Read(stream2Array, fileImage, null);
            var watch = new Stopwatch();
            watch.Start();
            for (int i = 0; i < benchCount; i++)
            {
                IfBinaryWriteUtility.Write(file, stream2Array, outStream2File);
            }
            watch.Stop();
            Console.WriteLine(@"IntermediateFileStream Write {0}ms", watch.ElapsedMilliseconds);
        }
#endif //DO_BENCHMARK
    }
}
