﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace _3dToolsTestUtility
{
    public class VertexAttr
    {
        public string Name { get; set; }
        public string Hint { get; set; }
        public int StreamIndex { get; set; }
        public vtx_attrib_typeType Type { get; set; }
        public vtx_attrib_quantize_typeType QuantizeType { get; set; }
    }

    public static class ModelDataCreationUtility
    {
        public static modelType CreateEmptyModel()
        {
            return new modelType()
            {
                skeleton = new skeletonType()
                {
                    skeleton_info = new skeletonInfoType()
                    {
                    },
                    bone_array = new bone_arrayType()
                    {
                        bone = new boneType[0]
                    }
                },
                shape_array = new shape_arrayType()
                {
                    shape = new shapeType[0]
                },
                vertex_array = new vertex_arrayType()
                {
                    vertex = new vertexType[0]
                },
                stream_array = new stream_arrayType()
                {
                    stream = new streamType[0]
                }
            };
        }

        public static boneType AddBone(
            this modelType model,
            string name, string parent)
        {
            return AddBone(model, name, parent, -1, -1, new float[] { 0.0f, 0.0f, 0.0f }, new float[] { 1.0f, 1.0f, 1.0f });
        }

        public static boneType AddBone(
            this modelType model,
            string name, string parent, int smoothSkinningMatrixIndex, int rigidSkinningMatrixIndex,
            float[] translate, float[] scale)
        {
            List<boneType> newBones = new List<boneType>(model.skeleton.bone_array.bone);
            boneType bone = new boneType()
            {
                name = name,
                parent_name = parent,
                matrix_index = new int[] { smoothSkinningMatrixIndex, rigidSkinningMatrixIndex },
                translate = translate,
                rotate = new float[] { 0.0f, 0.0f, 0.0f },
                scale = scale,
                compress_enable = true,
            };
            newBones.Add(bone);

            model.skeleton.bone_array.bone = newBones.ToArray();
            model.skeleton.bone_array.length = model.skeleton.bone_array.bone.Length;
            return bone;
        }

        public static shapeType AddShape(
            this modelType model,
            string name,
            string materialName,
            string boneName)
        {
            return AddShape(model, name, materialName, boneName, -1, -1);
        }

        public static shapeType AddShape(
            this modelType model,
            string name,
            string materialName,
            string boneName,
            int vertexIndex,
            int indexBufferStreamIndex)
        {
            List<shapeType> newShapes = new List<shapeType>(model.shape_array.shape);
            int skinningCount = 0;

            if (vertexIndex >= 0)
            {
                foreach (var vertex_attr in model.vertex_array.vertex[vertexIndex].vtx_attrib_array.vtx_attrib)
                {
                    if (vertex_attr.name.Equals("_i1"))
                    {
                        streamType stream = model.stream_array.stream[vertex_attr.stream_index];
                        skinningCount = stream.column + 4;
                        break;
                    }
                    else if (vertex_attr.name.Equals("_i0"))
                    {
                        streamType stream = model.stream_array.stream[vertex_attr.stream_index];
                        skinningCount = stream.column;
                    }
                }
            }

            var shape = new shapeType()
            {
                name = name,
                shape_info = new shape_infoType()
                {
                    mat_name = materialName,
                    bone_name = boneName,
                    original_bone_name = boneName,
                    original_material_name = materialName,
                    original_shape_count = 1,
                    vertex_index = vertexIndex,
                    vertex_skinning_count = skinningCount,
                },
                mesh_array = new mesh_arrayType()
                {
                    mesh = new meshType[]
                    {
                        new meshType()
                        {
                            stream_index = indexBufferStreamIndex,
                            mode = mesh_modeType.triangles,
                            count = indexBufferStreamIndex >= 0 ? model.stream_array.stream[indexBufferStreamIndex].count : 0,
                            submesh_array = new submesh_arrayType()
                            {
                                submesh = new submeshType[]
                                {
                                    new submeshType()
                                }
                            }
                        }
                    }
                }
            };

            newShapes.Add(shape);

            model.shape_array.shape = newShapes.ToArray();
            model.shape_array.length = model.shape_array.shape.Length;
            return shape;
        }

        public static void AddStream(this modelType model, string value, stream_typeType streamType, int streamCount, int streamColumn)
        {
            List<streamType> newStream = new List<streamType>(model.stream_array.stream);
            newStream.Add(new streamType()
            {
                stream_index = newStream.Count,
                type = streamType,
                count = streamCount,
                column = streamColumn,
                Value = value,
            });
            model.stream_array.stream = newStream.ToArray();
            model.stream_array.length = model.stream_array.stream.Length;
        }

        public static void AddVertexAttribute(this modelType model, VertexAttr[] attrs)
        {
            List<vertexType> newVertices = new List<vertexType>(model.vertex_array.vertex);
            var vertex = new vertexType()
            {
                vertex_index = model.vertex_array.vertex.Length,
                vtx_attrib_array = new vtx_attrib_arrayType()
                {
                    vtx_attrib = new vtx_attribType[attrs.Length]
                },
                vtx_buffer_array = new vtx_buffer_arrayType()
                {
                    vtx_buffer = new vtx_bufferType[]
                    {
                        new vtx_bufferType()
                        {
                            input_array = new input_arrayType()
                            {
                                input = new inputType[attrs.Length]
                            }
                        },
                    }
                }
            };
            for (int attrIndex = 0; attrIndex < attrs.Length; ++attrIndex)
            {
                VertexAttr attr = attrs[attrIndex];
                streamType stream = model.stream_array.stream.First(x => x.stream_index == attr.StreamIndex);
                vertex.vtx_attrib_array.vtx_attrib[attrIndex] = new vtx_attribType()
                {
                    attrib_index = attrIndex,
                    name = attr.Name,
                    hint = attr.Hint,
                    stream_index = attr.StreamIndex,
                    count = stream.count / stream.column,
                    type = attr.Type,
                    quantize_type = attr.QuantizeType,
                };
                vertex.vtx_buffer_array.vtx_buffer[0].input_array.input[attrIndex] = new inputType()
                {
                    attrib_index = attrIndex
                };
            }
            newVertices.Add(vertex);
            model.vertex_array.vertex = newVertices.ToArray();
            model.vertex_array.length = model.vertex_array.vertex.Length;
        }
    }
}
