﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

namespace _3dIntermediateFileLibraryTest
{
    [TestClass]
    public class PolygonReductionTest
    {
        private class ModelInfo
        {
            public string IfModelFullPath { get; set; }
            public string Args { get; set; }
            public string OptimizedModelPath { get; set; }
            public bool IsPassed { get; set; }
        }

        private class Branch
        {
            public List<Branch> BranchList { get; set; }
            public List<ModelInfo> ModelInfoList { get; set; }

            public bool IsModelInfoContain()
            {
                if (ModelInfoList.Count != 0)
                {
                    return true;
                }

                foreach (var child in BranchList)
                {
                    if (child.IsModelInfoContain())
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        private static string sigloRoot;
        private static string presetArgsFile;
        private static readonly string ExportPath = "Resources";
        private static readonly string DefaultTargetRateLodLevel = "--target-rate-lod-level1 0.5";
        private static readonly bool isVerboseOutput = false;

        /// <summary>
        /// ポリゴンリダクションを行い、バイナリコンバーターで書き出すテストです。
        /// </summary>
        [TestMethod]
        public void TestPolygonReduction()
        {
            Initialize();

            string binariesBasePath = sigloRoot + @"\Externals\TestBinaries\G3d\PolygonReductionResources";
            Assert.IsTrue(System.IO.Directory.Exists(binariesBasePath), string.Format("{0} is not found.", binariesBasePath));

            // fmdb が格納されているディレクトリツリーを作成
            Branch root = new Branch();
            root.ModelInfoList = new List<ModelInfo>();
            root.BranchList = new List<Branch>();

            SetupBranch(root, binariesBasePath);

            // ポリゴンリダクションを実行 (3d中間ファイルオプティマイザーを叩く)
            ExecutePolygonReduction(root);

            // 念のためバイナリコンバーターも走らせる
            Execute3dBinaryConverter(root);

            // ポリゴンリダクションやコンバートに失敗していた場合にアサートにかけておく
            CheckAssertion(root);
        }

        /// <summary>
        /// クラス内メンバ変数に値を設定します。
        /// </summary>
        private void Initialize()
        {
            // 各種パスの設定
            sigloRoot = System.Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
            presetArgsFile = sigloRoot + @"\Tools\Graphics\3dTools\Presets\PolygonReduction\CommonSet.txt";

            // エクスポートされるファイルをクリーン
            if (System.IO.Directory.Exists(ExportPath))
            {
                System.IO.Directory.Delete(ExportPath, true);
            }

            System.IO.Directory.CreateDirectory(ExportPath);
        }

        private void SetupBranch(Branch branch, string path)
        {
            // fmdb を 格納していく
            string[] ifModelPaths = System.IO.Directory.GetFiles(path, "*.fmdb");
            foreach (var ifModelPath in ifModelPaths)
            {
                ModelInfo modelInfo = new ModelInfo();
                modelInfo.IfModelFullPath = System.IO.Path.GetFullPath(ifModelPath);

                string filePath = GetPathWithoutExtension(modelInfo.IfModelFullPath);
                string argsFilePath = filePath + ".txt"; // とりあえず、txt にしておきます。

                StringBuilder builder = new StringBuilder("--polygon-reduction --polygon-reduction-options ");
                builder.Append(string.Format("\"{0} --args-file ", DefaultTargetRateLodLevel));

                // args ファイルが存在する場合は利用します。
                if (System.IO.File.Exists(argsFilePath))
                {
                    builder.Append(argsFilePath);
                }
                else
                {
                    builder.Append(presetArgsFile);
                }
                builder.Append("\" ");

                // 名前がかぶる可能性があるのでユニークな情報を付けておく。
                string outputFilePath = string.Format("{0}/{1}_{2}.fmdb", ExportPath, System.IO.Path.GetFileNameWithoutExtension(ifModelPath), Guid.NewGuid().ToString("N"));
                modelInfo.OptimizedModelPath = outputFilePath;

                builder.Append(string.Format("-o {0}", outputFilePath));
                builder.Append(" " + ifModelPath);

                modelInfo.Args = builder.ToString();
                branch.ModelInfoList.Add(modelInfo);
            }

            // ディレクトリを掘る
            string[] childrenDirectories = System.IO.Directory.GetDirectories(path);
            foreach (var childDirectory in childrenDirectories)
            {
                // ドットが入っているディレクトリは隠しファイルである場合が多い (.svn .vs など･･･)
                if (childDirectory.Contains("."))
                {
                    continue;
                }

                Branch child = new Branch();
                child.BranchList = new List<Branch>();
                child.ModelInfoList = new List<ModelInfo>();

                SetupBranch(child, childDirectory);
                if (child.IsModelInfoContain())
                {
                    branch.BranchList.Add(child);
                }
            }
        }

        private void ExecutePolygonReduction(Branch branch)
        {
            // 1回のポリゴンリダクション対して制限時間を設ける。
            TimeSpan timeOut = TimeSpan.FromMinutes(2);

            foreach (var modelInfo in branch.ModelInfoList)
            {
                // 3d中間ファイルオプティマイザーをプロセスで実行する。
                ProcessStartInfo startInfo = new ProcessStartInfo();
                startInfo.FileName = sigloRoot + @"\Tools\Graphics\3dTools\3dIntermediateFileOptimizer.exe";
                startInfo.Arguments = modelInfo.Args;
                startInfo.CreateNoWindow = false;
                startInfo.RedirectStandardOutput = true;
                startInfo.RedirectStandardError = true;
                startInfo.UseShellExecute = false;

                StringBuilder stdOut = new StringBuilder();
                StringBuilder stdError = new StringBuilder();
                stdOut.AppendLine(startInfo.FileName);
                stdOut.AppendLine(startInfo.Arguments);

                using (Process optimizer = Process.Start(startInfo))
                {
                    optimizer.OutputDataReceived += (sender, e) =>
                    {
                        if (e.Data != null)
                        {
                            stdOut.AppendLine(e.Data);
                        }
                    };

                    optimizer.ErrorDataReceived += (sender, e) =>
                    {
                        if(e.Data != null)
                        {
                            stdError.AppendLine(e.Data);
                        }
                    };

                    optimizer.BeginOutputReadLine();
                    optimizer.BeginErrorReadLine();

                    if (!optimizer.WaitForExit((int)timeOut.TotalMilliseconds))
                    {
                        stdOut.AppendLine("Time out error.");
                        optimizer.Kill();
                    }

                    optimizer.CancelOutputRead();
                    optimizer.CancelErrorRead();

                    if (optimizer.ExitCode != 0)
                    {
                        string[] logs = stdOut.ToString().Split('\n');
                        foreach (var line in logs)
                        {
                            if (!line.Contains("AppData"))
                            {
                                continue;
                            }

                            string errorLogPath = line;
                            errorLogPath = errorLogPath.Replace(" ", string.Empty).Replace("\r", string.Empty).Replace("/", "\\");
                            errorLogPath = System.Text.RegularExpressions.Regex.Replace(errorLogPath, @"[^\x00-\x7F]", string.Empty);

                            string errorLog = System.IO.File.ReadAllText(errorLogPath);
                            stdOut.AppendLine(errorLog);

                            // CI マシンの AppData 以下に ログファイルが溜まっていくことを防ぐため、ログファイルを破棄します。
                            System.IO.File.Delete(errorLogPath);
                            break;
                        }
                    }
                    else
                    {
                        modelInfo.IsPassed = true;
                    }
                }

                if (!modelInfo.IsPassed || isVerboseOutput)
                {
                    System.Console.Write(stdOut.ToString());
                    System.Console.Write(stdError.ToString());
                }
            }

            foreach (var child in branch.BranchList)
            {
                ExecutePolygonReduction(child);
            }
        }

        private void Execute3dBinaryConverter(Branch branch)
        {
            // 1回の処理に対して制限時間を設ける。
            TimeSpan timeOut = TimeSpan.FromSeconds(30);

            // 3d中間ファイルオプティマイザーをプロセスで実行する。
            foreach (var modelInfo in branch.ModelInfoList)
            {
                // ポリゴンリダクションに失敗していた場合にはバイナリコンバートしない。
                if (!modelInfo.IsPassed)
                {
                    continue;
                }

                ProcessStartInfo startInfo = new ProcessStartInfo();
                startInfo.FileName = sigloRoot + @"\Tools\Graphics\3dTools\3dBinaryConverter.exe";
                startInfo.Arguments = string.Format("{0} -o {1}.bfres",
                    modelInfo.OptimizedModelPath,
                    GetPathWithoutExtension(modelInfo.OptimizedModelPath));

                startInfo.CreateNoWindow = false;
                startInfo.RedirectStandardOutput = true;
                startInfo.RedirectStandardError = true;
                startInfo.UseShellExecute = false;

                StringBuilder stdOut = new StringBuilder();
                StringBuilder stdError = new StringBuilder();
                stdOut.AppendLine(startInfo.FileName);
                stdOut.AppendLine(startInfo.Arguments);

                using (var binaryConverter = Process.Start(startInfo))
                {
                    binaryConverter.OutputDataReceived += (sender, e) =>
                    {
                        if(e.Data != null)
                        {
                            stdOut.AppendLine(e.Data);
                        }
                    };

                    binaryConverter.ErrorDataReceived += (sender, e) =>
                    {
                        if(e.Data != null)
                        {
                            stdError.AppendLine(e.Data);
                        }
                    };

                    binaryConverter.BeginOutputReadLine();
                    binaryConverter.BeginErrorReadLine();

                    // エラー情報用にログを取っておく
                    if(!binaryConverter.WaitForExit((int)timeOut.TotalMilliseconds))
                    {
                        stdOut.AppendLine("Time out error.");
                        binaryConverter.Kill();
                    }

                    binaryConverter.CancelOutputRead();
                    binaryConverter.CancelErrorRead();

                    modelInfo.IsPassed = (binaryConverter.ExitCode == 0);

                    stdOut.AppendLine(startInfo.FileName);
                    stdOut.AppendLine(startInfo.Arguments);

                    // 元のファイルディレクトリも吐き出しておく
                    stdOut.AppendLine(string.Format("(Base Resource: {0})", modelInfo.IfModelFullPath));
                }

                if (!modelInfo.IsPassed || isVerboseOutput)
                {
                    System.Console.WriteLine(stdOut.ToString());
                    System.Console.WriteLine(stdError.ToString());
                }
            }

            foreach (var child in branch.BranchList)
            {
                Execute3dBinaryConverter(child);
            }
        }

        /// <summary>
        /// 指定されたパス文字列から拡張子を削除して返します
        /// </summary>
        private static string GetPathWithoutExtension(string path)
        {
            var extension = System.IO.Path.GetExtension(path);
            if (string.IsNullOrEmpty(extension))
            {
                return path;
            }
            return path.Replace(extension, string.Empty);
        }

        private void CheckAssertion(Branch branch)
        {
            foreach (var modelInfo in branch.ModelInfoList)
            {
                Assert.IsTrue(modelInfo.IsPassed);
            }

            foreach (var child in branch.BranchList)
            {
                CheckAssertion(child);
            }
        }
    }
}
