﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Reflection;
using nw.g3d.toollib;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using _3dToolsTestUtility;
using Nintendo.G3dTool.Entities;

namespace _3dIntermediateFileConverterTest
{
    [TestClass]
    public class ConvertFbxTest
    {
        public ConvertFbxTest()
        {
            IoUtility.WriteLogLine($"{this.GetType().Name}");
        }

        [TestMethod]
        public void TestFbxDisableTexture()
        {
            IoUtility.WriteLogLine($"{MethodBase.GetCurrentMethod().Name}");
            Utility.TestForAllModelFiles("*.fbx", "--disable-texture", outputFmdFilePath =>
            {
                // 出力後にテクスチャーが書き出されていないことをテスト
                string outputFolder = System.IO.Path.GetDirectoryName(outputFmdFilePath);
                string texFolderPath = System.IO.Path.Combine(outputFolder, "textures");
                Assert.IsFalse(System.IO.Directory.Exists(texFolderPath));
            });
        }

        /// <summary>
        /// 中間ファイルコンバーターの .fbx 変換テストです。
        /// </summary>
        [TestMethod]
        public void TestFbxConversion()
        {
            IoUtility.WriteLogLine($"{MethodBase.GetCurrentMethod().Name}");
            List<Vector3> originalMaxDistancePositions = new List<Vector3>();
            Utility.TestForAllModelFiles("*.fbx", string.Empty, outputFmdFilePath =>
            {
                // 出力後の fmdb が読み込めるかをテスト
                var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath, IoUtility.GetXsdBasePath());
                var model = file.GetRootEntity<Model>();
                Assert.IsNotNull(model);

                // 出力後の ftxb が読み込めるかをテスト
                string outputFolder = System.IO.Path.GetDirectoryName(outputFmdFilePath);
                string texFolderPath = System.IO.Path.Combine(outputFolder, "textures");
                if (System.IO.Directory.Exists(texFolderPath))
                {
                    var texFilePaths = System.IO.Directory.EnumerateFiles(texFolderPath, "*.ftxb", System.IO.SearchOption.TopDirectoryOnly);
                    Assert.AreNotEqual(0, texFilePaths.Count());
                    foreach (string texFilePath in texFilePaths)
                    {
                        var texFile = IfReadUtility.ReadIntermediateFile(texFilePath, IoUtility.GetXsdBasePath());
                        Assert.IsNotNull(texFile.GetRootEntity<Texture>());
                    }
                }

                // position の最大距離の頂点座標を取得
                originalMaxDistancePositions.Add(FindMaxDistancePos(model));
            });

            // 拡大変換テスト
            float magnify = 10.0f;
            int targetIndex = 0;
            Utility.TestForAllModelFiles("*.fbx", $"--magnify {magnify}", outputFmdFilePath =>
            {
                var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath, IoUtility.GetXsdBasePath());
                var model = file.GetRootEntity<Model>();

                Vector3 maxDistancePos = FindMaxDistancePos(model);
                float delta = 0.000001f;
                Assert.AreEqual(originalMaxDistancePositions[targetIndex][0] * magnify, maxDistancePos[0], delta);
                Assert.AreEqual(originalMaxDistancePositions[targetIndex][1] * magnify, maxDistancePos[1], delta);
                Assert.AreEqual(originalMaxDistancePositions[targetIndex][2] * magnify, maxDistancePos[2], delta);

                ++targetIndex;
            });
        }

        private Vector3 FindMaxDistancePos(Model model)
        {
            Shape topShape = model.Shapes[0];
            StreamFloat posStream = topShape.ShapeInfo.Vertex.VtxAttribs.First(x => x.Name == "_p0").Stream as StreamFloat;
            Vector3 maxDistancePos = new Vector3();
            maxDistancePos.SetZero();
            Vector3 tmpPos = new Vector3();
            for (int vertexIndex = 0, vertexCount = posStream.Values.Count / 3; vertexIndex < vertexCount; ++vertexIndex)
            {
                tmpPos[0] = posStream.Values[vertexIndex * 3 + 0];
                tmpPos[1] = posStream.Values[vertexIndex * 3 + 1];
                tmpPos[2] = posStream.Values[vertexIndex * 3 + 2];
                if (maxDistancePos.LengthSquared < tmpPos.LengthSquared)
                {
                    maxDistancePos[0] = tmpPos[0];
                    maxDistancePos[1] = tmpPos[1];
                    maxDistancePos[2] = tmpPos[2];
                }
            }

            return maxDistancePos;
        }
    }
}
