﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utilities/SandboxAllocator.h"
#include "Utilities/SandboxUtility.h"
#include "Utilities/SandboxNwG3dEditor.h"

#include <nw/g3d/fnd/g3d_GLUtility.h>

using namespace nw::g3d::edit;
using namespace nw::g3d;
using namespace G3dSandbox;

#ifndef _WIN32
#define ENABLE_DRC_OUTPUT
#endif

#if 0//VS2012では有効になる？
#ifdef _WIN32
#pragma execution_character_set("utf-8")
#endif
#endif

namespace
{

// #define USE_FSD_SETUP

std::vector<nw::g3d::ResFile*> s_Files;
std::vector<nw::g3d::ResShaderArchive*> s_ShaderArchives;

std::vector<AnimModelObj*> s_Models;

std::vector<nw::g3d::SkeletalAnimObj*> sSklAnims;
std::vector<nw::g3d::VisibilityAnimObj*> sBoneVisAnims;
std::vector<nw::g3d::CameraAnimObj*> sCameraAnims;
std::vector<nw::g3d::LightAnimObj*> sLightAnims;
std::vector<nw::g3d::FogAnimObj*> sFogAnims;
std::vector<nw::g3d::TexPatternAnimObj*> sTexAnims;
std::vector<nw::g3d::ShapeAnimObj*> sShapeAnims;
std::vector<nw::g3d::ShaderParamAnimObj*> sParamAnims;

std::vector<RenderModel*> s_RenderModels;
std::vector<UniformBlockClip> s_UniformBlockBox;

G3dSandbox::SandboxAllocator sAllocator;

bool s_UseCamera;

float s_ClearColor[3] = { 0.4f, 0.4f, 0.4f };

enum FileKind
{
    LOAD_BFRES,
    LOAD_BFSHA,
    LOAD_FSDB,
    NUM_FILES
};

const char* FILES[NUM_FILES] =
{
    "edittest.bfres",
#if defined(_WIN32) && !defined(USE_FSD_SETUP)
    "edittest_pc.bfsha",
#else
    "edittest.bfsha",
#endif
    "edittest.fsdb",
};

void Calc(nw::g3d::demo::Pad& pad)
{
    if (pad.IsHold(nw::g3d::demo::Pad::TRIGGER_L) && pad.IsTriggered(nw::g3d::demo::Pad::TRIGGER_R))
    {
        s_UseCamera = !s_UseCamera;
    }

#if NW_G3D_CONFIG_USE_HOSTIO
    if (!s_UseCamera)
    {
        if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            // EditServer に Attach
            for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
            {
                nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                if (!nw::g3d::edit::EditServer::Instance().HasModelObj(mdl))
                {
                    nw::g3d::edit::EditServer::AttachModelArg arg;
                    arg.modelObj = mdl;
                    if (!pad.IsHold(nw::g3d::demo::Pad::TRIGGER_L) && !pad.IsHold(nw::g3d::demo::Pad::TRIGGER_R))
                    {
                        arg.attachFileName = "%NW4F_G3D_ROOT%/SandBox/EditData/model/PrjDemActPierrot/PierrotAnimDance.fmda";
                    }

                    nw::g3d::edit::EditServer::Instance().AttachModel(arg);
                }
                if (!pad.IsHold(nw::g3d::demo::Pad::TRIGGER_L))
                {
                    break; // 1つのみAttach。
                }
            }
        }
        else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_B))
        {
            // EditServer から Detach
            for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
            {
                nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                nw::g3d::edit::EditServer::DetachModelArg arg;
                arg.modelObj = mdl;
                bool success = nw::g3d::edit::EditServer::Instance().DetachModel(arg);
                if (!success)
                {
                    G3DSANDBOX_PRINT("Detach failed\n");
                }

                if (!pad.IsHold(nw::g3d::demo::Pad::TRIGGER_L))
                {
                    break; // 1つのみDetach。
                }
            }
        }
        else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_X))
        {
            // EditServer に Attach
            for (int i = 0; i < static_cast<int>(s_ShaderArchives.size()); ++i)
            {
                nw::g3d::res::ResShaderArchive* shaderArchive = s_ShaderArchives[i];
                if (!nw::g3d::edit::EditServer::Instance().HasShaderArchive(shaderArchive))
                {
                    nw::g3d::edit::EditServer::AttachShaderArchiveArg arg;
                    arg.resShaderArchive = shaderArchive;
                    if (!pad.IsHold(nw::g3d::demo::Pad::TRIGGER_R))
                    {
                        arg.attachFileName = "%NW4F_G3D_ROOT%/SandBox/EditData/shader/edittest.fsdb";
                    }

                    nw::g3d::edit::EditServer::Instance().AttachShaderArchive(arg);
                }
                if (i >= 0)
                {
                    break; // 1つのみAttach。
                }
            }
        }
        else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_Y))
        {
            // EditServer に Attach
            for (int i = 0; i < static_cast<int>(s_ShaderArchives.size()); ++i)
            {
                nw::g3d::res::ResShaderArchive* shaderArchive = s_ShaderArchives[i];
                if (nw::g3d::edit::EditServer::Instance().HasShaderArchive(shaderArchive))
                {
                    nw::g3d::edit::EditServer::DetachShaderArchiveArg arg;
                    arg.resShaderArchive = shaderArchive;
                    nw::g3d::edit::EditServer::Instance().DetachShaderArchive(arg);
                }
                if (i >= 0)
                {
                    break; // 1つのみAttach。
                }
            }
        }
        else if (pad.IsTriggered(nw::g3d::demo::Pad::TRIGGER_L))
        {
            for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
            {
                nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                if (nw::g3d::edit::EditServer::Instance().HasModelObj(mdl))
                {
                    int materialCount = mdl->GetMaterialCount();
                    for (int j = 0; j < materialCount; ++j)
                    {
                        nw::g3d::edit::EditServer::Instance().PushPickupMaterial(mdl, j);
                    }
                }
            }
        }
        else if (pad.IsTriggered(nw::g3d::demo::Pad::TRIGGER_R))
        {
            nw::g3d::edit::EditServer::Instance().ClearPickupMaterial();
        }
    }

    // フレームコントロールのテスト
    {
        if (pad.IsHold(nw::g3d::demo::Pad::TRIGGER_Z))
        {
            float frameStep = nw::g3d::edit::EditServer::Instance().GetFrameStep();
            if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_LEFT))
            {
                frameStep -= 0.5f;
                nw::g3d::edit::EditServer::Instance().SetFrameStep(frameStep);
            }
            else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_RIGHT))
            {
                frameStep += 0.5f;
                nw::g3d::edit::EditServer::Instance().SetFrameStep(frameStep);
            }
            else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_UP))
            {
                for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
                {
                    nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                    if (nw::g3d::edit::EditServer::Instance().HasModelObj(mdl))
                    {
                        // 一つ目のモデルに対してのみ
                        nw::g3d::edit::EditServer::Instance().MovePrevModelAnim(mdl);
                        break;
                    }
                }
                G3DSANDBOX_PRINT("FrameCount = %f\n", nw::g3d::edit::EditServer::Instance().GetFrameCount());
            }
            else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_DOWN))
            {

                for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
                {
                    nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                    if (nw::g3d::edit::EditServer::Instance().HasModelObj(mdl))
                    {
                        // 一つ目のモデルに対してのみ
                        nw::g3d::edit::EditServer::Instance().MoveNextModelAnim(mdl);
                        break;
                    }
                }
            }
        }
        else
        {
            if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_UP))
            {
                nw::g3d::edit::EditServer::Instance().SetAnimFlag(true);
            }
            else if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_DOWN))
            {
                nw::g3d::edit::EditServer::Instance().SetAnimFlag(false);
            }

            float frame = nw::g3d::edit::EditServer::Instance().GetFrame();
            float frameStep = nw::g3d::edit::EditServer::Instance().GetFrameStep();
            if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_LEFT) || pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_RIGHT))
            {
                if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_LEFT))
                {
                    frame -= frameStep;
                }
                else
                {
                    frame += frameStep;
                }

                if (frame < 0.f)
                {
                    frame = nw::g3d::edit::EditServer::Instance().GetFrameCount();
                }
                else if (frame > nw::g3d::edit::EditServer::Instance().GetFrameCount())
                {
                    frame = 0.f;
                }
                nw::g3d::edit::EditServer::Instance().SetFrame(frame);
            }
        }
    }
#endif

    for (int i = 0; i < static_cast<int>(sSklAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sSklAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sSklAnims[i]->Calc();
    }

    for (int i = 0; i < static_cast<int>(sBoneVisAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sBoneVisAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sBoneVisAnims[i]->Calc();
    }

    for (int i = 0; i < static_cast<int>(sCameraAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sCameraAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sCameraAnims[i]->Calc();
        s_CameraObj.pos.Set(sCameraAnims[i]->GetResult()->view.pos);
        s_CameraObj.up.Set(0.0f, 1.0f, 0.0f);
        s_CameraObj.target.Set(sCameraAnims[i]->GetResult()->view.aim);

        if (sCameraAnims[i]->GetResource()->GetProjectionMode() == nw::g3d::ResCameraAnim::PROJ_ORTHO)
        {
            s_CameraObj.height = sCameraAnims[i]->GetResult()->proj.height;
            s_CameraObj.isOrtho = true;
        }
        else
        {
            s_CameraObj.fovy = sCameraAnims[i]->GetResult()->proj.fovy;
            s_CameraObj.isOrtho = false;
        }
        s_CameraObj.aspect = sCameraAnims[i]->GetResult()->proj.aspect;
        s_CameraObj.nearZ = sCameraAnims[i]->GetResult()->proj.nearZ;
        s_CameraObj.farZ = sCameraAnims[i]->GetResult()->proj.farZ;
        s_CameraObj.Calc();
    }

    for (int i = 0; i < static_cast<int>(sLightAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sLightAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sLightAnims[i]->Calc();
        nw::g3d::LightAnimResult* result = sLightAnims[i]->GetResult();
        VEC3Transform(nw::g3d::math::Vec3::Cast(result->dir), s_CameraObj.GetViewMtx());
        memcpy(s_EnvObj.block.lightDir.a, result->dir, sizeof(float) * 3);
        memcpy(s_EnvObj.block.diffuseColor.a, result->color[0], sizeof(float) * 3);
        memcpy(s_EnvObj.block.specularColor.a, result->color[1], sizeof(float) * 3);
    }

    for (int i = 0; i < static_cast<int>(sFogAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sFogAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sFogAnims[i]->Calc();
        const nw::g3d::FogAnimResult* result = sFogAnims[i]->GetResult();
        memcpy(s_EnvObj.fogColor.a, result->color, sizeof(float) * 3);
        memcpy(s_EnvObj.fogDist, result->distAttn, sizeof(float) * 2);
    }

    for (int i = 0; i < static_cast<int>(sTexAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sTexAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sTexAnims[i]->Calc();
    }

    for (int i = 0; i < static_cast<int>(sShapeAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sShapeAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sShapeAnims[i]->Calc();
    }

    for (int i = 0; i < static_cast<int>(sParamAnims.size()); ++i)
    {
        //if (pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_A))
        {
            sParamAnims[i]->GetFrameCtrl().UpdateFrame();
        }
        sParamAnims[i]->Calc();
    }

    // env の更新
    std::vector<AnimModelObj*>::iterator comp =
        std::find_if(s_Models.begin(), s_Models.end(), ModelFinder<AnimModelObj*>("EnvModel"));

    if (comp != s_Models.end())
    {
        nw::g3d::MaterialObj* envMat = (*comp)->mdl->GetMaterial("EnvMat");
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_HEMISPHERE_DIR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.hemiDir.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.hemiDir));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_HEMISPHERE_SKY_COLOR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.skyColor.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.skyColor));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_HEMISPHERE_GROUND_COLOR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.groundColor.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.groundColor));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_FOG_START_Z]);
            if (idx != -1)
            {
                s_EnvObj.fogStartZ = *envMat->GetShaderParam<float>(idx);
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_FOG_END_Z]);
            if (idx != -1)
            {
                s_EnvObj.fogEndZ = *envMat->GetShaderParam<float>(idx);
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_FOG_COLOR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.fogColor.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.fogColor));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_LIGHT_DIR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.lightDir.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.lightDir));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_DIFFUSE_COLOR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.diffuseColor.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.diffuseColor));
            }
        }
        {
            int idx= envMat->GetShaderParamIndex(ENV_NAMES[ENV_SPECULAR_COLOR]);
            if (idx != -1)
            {
                memcpy(s_EnvObj.specularColor.a, envMat->GetShaderParam(idx), sizeof(s_EnvObj.specularColor));
            }
        }
    }

    s_EnvObj.CalcHemisphereLight();
    s_EnvObj.CalcDirectionalLight();
    s_EnvObj.CalcFog();

    for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
    {
        AnimModelObj* animModel = s_Models[i];

        nw::g3d::math::Mtx34 baseMtx;
        baseMtx.SetSR(animModel->scale, animModel->rotate);
        baseMtx.SetT(animModel->translate);

        if (animModel->constraintModel != NULL && animModel->constraintBoneIndex != -1)
        {
            std::vector<AnimModelObj*>::iterator comp =
                std::find_if(s_Models.begin(), s_Models.end(), ModelFinder<AnimModelObj*>(animModel->constraintModel));
            if (comp != s_Models.end())
            {
                nw::g3d::Mtx34& constraintMtx = (*comp)->mdl->GetSkeleton()->GetWorldMtxArray()[animModel->constraintBoneIndex];
                baseMtx.Mul(constraintMtx, baseMtx);
            }
        }

        if (animModel->mdl->GetSkeleton())
        {
            if (animModel->sklAnim != NULL)
            {
                animModel->sklAnim->ApplyTo(animModel->mdl->GetSkeleton());
            }

            if (animModel->boneVisAnim != NULL)
            {
                animModel->boneVisAnim->ApplyTo(animModel->mdl);
            }

            if (animModel->texAnim != NULL)
            {
                animModel->texAnim->ApplyTo(animModel->mdl);
            }

            if (animModel->shapeAnim != NULL)
            {
                animModel->shapeAnim->ApplyTo(animModel->mdl);
            }

            if (animModel->paramAnim != NULL)
            {
                animModel->paramAnim->ApplyTo(animModel->mdl);
            }

#if NW_G3D_CONFIG_USE_HOSTIO
            // アタッチされているかの判定は CalcSkeletalAnimations の中で行われているので省略。
            nw::g3d::edit::EditServer::Instance().CalcSkeletalAnimations(animModel->mdl);
#endif

            animModel->mdl->CalcWorld(baseMtx);
        }
    }

    // RenderPriority に基づいてソート
    for (int renderModelIdx = 0; renderModelIdx < static_cast<int>(s_RenderModels.size()); ++renderModelIdx)
    {
        RenderModel* renderModel = s_RenderModels[renderModelIdx];

        std::sort(renderModel->objArray.begin(), renderModel->objArray.end(), GreaterPriority());
    }
} // NOLINT (readability/fn_size)

void CalcBlock()
{
    // view の編集
    {
        std::vector<UniformBlockClip>::iterator comp =
            std::find_if(s_UniformBlockBox.begin(), s_UniformBlockBox.end(), UBOFinder("view"));

        if (comp != s_UniformBlockBox.end())
        {
            void* data = comp->uBlock->GetData();
            nw::g3d::Copy32<true>(data, &s_CameraObj.GetBlock(), sizeof(CameraBlock) >> 2);

            comp->uBlock->DCFlush();
        }
    }

    // env の編集
    {
        std::vector<UniformBlockClip>::iterator comp =
            std::find_if(s_UniformBlockBox.begin(), s_UniformBlockBox.end(), UBOFinder("env"));

        if (comp != s_UniformBlockBox.end())
        {
            void* data = comp->uBlock->GetData();

            // 実機版の場合は内部で EndianSwap を行います。
            nw::g3d::Copy32<true>(data, &s_EnvObj.block, sizeof(EnvBlock) >> 2);

            comp->uBlock->DCFlush();
        }
    }

    // Model の UniformBlock 更新
    for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
    {
        nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
        mdl->CalcView(0, s_CameraObj.GetViewMtx());
        mdl->CalcShape();
        mdl->CalcMaterial();

        mdl->CalcMtxBlock();
    }
}

void Draw()
{
    for (int renderModelIdx = 0; renderModelIdx < static_cast<int>(s_RenderModels.size()); ++renderModelIdx)
    {
        RenderModel* renderModel = s_RenderModels[renderModelIdx];
        for (int renderObjIdx = 0; renderObjIdx < static_cast<int>(renderModel->objArray.size()); ++renderObjIdx)
        {
            RenderObj* obj = renderModel->objArray[renderObjIdx];
            if (!renderModel->mdl->IsBoneVisible(obj->shape->GetBoneIndex()) ||
                !renderModel->mdl->IsMatVisible(obj->mat->GetResource()->GetIndex()))
            {
                continue;
            }

            nw::g3d::ResShadingModel* rShadingModel = obj->shaderAssign->shadingModel;
            nw::g3d::ResShaderProgram* rProgram = obj->shaderAssign->program;

            nw::g3d::SetShaderMode(rProgram->GetShaderMode());

            rProgram->Update();
            rProgram->Load();

            // UniformBlock をロードする
            for (std::vector<UniformBlockClip>::iterator iter = s_UniformBlockBox.begin(); iter != s_UniformBlockBox.end(); ++iter)
            {
                nw::g3d::ResUniformBlockVar* userUbo = rShadingModel->GetUniformBlock(iter->name.c_str());

                if (userUbo == NULL)
                {
                    continue;
                }

                for (int i = 0; i < nw::g3d::ResShaderProgram::NUM_STAGE; ++i)
                {
                    nw::g3d::ResShaderProgram::Stage stage =
                        static_cast<nw::g3d::ResShaderProgram::Stage>(i);

                    u32 location = rProgram->GetUniformBlockLocation(userUbo->GetIndex(), stage);

                    if (location != nw::g3d::SHADER_LOCATION_NONE)
                    {
                        if (stage == nw::g3d::ResShaderProgram::STAGE_VERTEX)
                        {
                            iter->uBlock->LoadVertexUniforms(location);
                        }
                        else if(stage == nw::g3d::ResShaderProgram::STAGE_GEOMETRY)
                        {
                            iter->uBlock->LoadGeometryUniforms(location);
                        }
                        else if(stage == nw::g3d::ResShaderProgram::STAGE_FRAGMENT)
                        {
                            iter->uBlock->LoadFragmentUniforms(location);
                        }
                    }
                }
            }

            // skeleton
            if (renderModel->mdl->GetSkeleton())
            {
                int skeletonIndex = rShadingModel->GetSystemBlockIndex(nw::g3d::ResUniformBlockVar::TYPE_SKELETON);
                if (skeletonIndex != -1 && renderModel->mdl->GetSkeleton()->GetMtxBlock().GetSize() != 0)
                {
                    for (int i = 0; i < nw::g3d::ResShaderProgram::NUM_STAGE; ++i)
                    {
                        nw::g3d::ResShaderProgram::Stage stage =
                            static_cast<nw::g3d::ResShaderProgram::Stage>(i);
                        u32 location = rProgram->GetUniformBlockLocation(skeletonIndex, stage);

                        if (location != nw::g3d::SHADER_LOCATION_NONE)
                        {
                            if (stage == nw::g3d::ResShaderProgram::STAGE_VERTEX)
                            {
                                renderModel->mdl->GetSkeleton()->GetMtxBlock().LoadVertexUniforms(location);
                            }
                            else if(stage == nw::g3d::ResShaderProgram::STAGE_GEOMETRY)
                            {
                                renderModel->mdl->GetSkeleton()->GetMtxBlock().LoadGeometryUniforms(location);
                            }
                            else if(stage == nw::g3d::ResShaderProgram::STAGE_FRAGMENT)
                            {
                                renderModel->mdl->GetSkeleton()->GetMtxBlock().LoadFragmentUniforms(location);
                            }
                        }
                    }
                }
            }

            // shape
            int shapeIndex = rShadingModel->GetSystemBlockIndex(nw::g3d::ResUniformBlockVar::TYPE_SHAPE);
            if (shapeIndex != -1 && obj->shape->GetShpBlock(0).GetSize() != 0)
            {
                for (int i = 0; i < nw::g3d::ResShaderProgram::NUM_STAGE; ++i)
                {
                    nw::g3d::ResShaderProgram::Stage stage =
                        static_cast<nw::g3d::ResShaderProgram::Stage>(i);

                    u32 location = rProgram->GetUniformBlockLocation(shapeIndex, stage);

                    if (location != nw::g3d::SHADER_LOCATION_NONE)
                    {
                        if (stage == nw::g3d::ResShaderProgram::STAGE_VERTEX)
                        {
                            obj->shape->GetShpBlock(0).LoadVertexUniforms(location);
                        }
                        else if(stage == nw::g3d::ResShaderProgram::STAGE_GEOMETRY)
                        {
                            obj->shape->GetShpBlock(0).LoadGeometryUniforms(location);
                        }
                        else if(stage == nw::g3d::ResShaderProgram::STAGE_FRAGMENT)
                        {
                            obj->shape->GetShpBlock(0).LoadFragmentUniforms(location);
                        }
                    }
                }
            }

            // material
            {
                int matIndex = rShadingModel->GetSystemBlockIndex(nw::g3d::ResUniformBlockVar::TYPE_MATERIAL);
                if (matIndex != -1 && obj->mat->GetMatBlock().GetSize() != 0)
                {
                    for (int i = 0; i < nw::g3d::ResShaderProgram::NUM_STAGE; ++i)
                    {
                        nw::g3d::ResShaderProgram::Stage stage =
                            static_cast<nw::g3d::ResShaderProgram::Stage>(i);

                        u32 location = rProgram->GetUniformBlockLocation(matIndex, stage);

                        if (location != nw::g3d::SHADER_LOCATION_NONE)
                        {
                            if (stage == nw::g3d::ResShaderProgram::STAGE_VERTEX)
                            {
                                obj->mat->GetMatBlock().LoadVertexUniforms(location);
                            }
                            else if(stage == nw::g3d::ResShaderProgram::STAGE_GEOMETRY)
                            {
                                obj->mat->GetMatBlock().LoadGeometryUniforms(location);
                            }
                            else if(stage == nw::g3d::ResShaderProgram::STAGE_FRAGMENT)
                            {
                                obj->mat->GetMatBlock().LoadFragmentUniforms(location);
                            }
                        }
                    }
                }

                obj->mat->GetResRenderState()->Load();
            }

            // sampler
            for (int i = 0; i < static_cast<int>(obj->samplerTable.size()); ++i)
            {
                SamplerClip* clip = obj->samplerTable[i];
                nw::g3d::ResSampler* sampler = clip->sampler;

                for (int i = 0; i < nw::g3d::ResShaderProgram::NUM_STAGE; ++i)
                {
                    nw::g3d::ResShaderProgram::Stage stage =
                        static_cast<nw::g3d::ResShaderProgram::Stage>(i);

                    u32 location = rProgram->GetSamplerLocation(clip->shaderSamplerIdx, stage);

                    if (location != nw::g3d::SHADER_LOCATION_NONE)
                    {
                        if (stage == nw::g3d::ResShaderProgram::STAGE_VERTEX)
                        {
                            sampler->GetGfxSampler()->LoadVertexSampler(location);
                            obj->mat->GetResTexture(sampler->GetIndex())->GetGfxTexture()->LoadVertexTexture(location);
                        }
                        else if (stage == nw::g3d::ResShaderProgram::STAGE_GEOMETRY)
                        {
                            sampler->GetGfxSampler()->LoadGeometrySampler(location);
                            obj->mat->GetResTexture(sampler->GetIndex())->GetGfxTexture()->LoadGeometryTexture(location);
                        }
                        else if (stage == nw::g3d::ResShaderProgram::STAGE_FRAGMENT)
                        {
                            sampler->GetGfxSampler()->LoadFragmentSampler(location);
                            obj->mat->GetResTexture(sampler->GetIndex())->GetGfxTexture()->LoadFragmentTexture(location);
                        }
                    }
                }
            }

            // shape
            obj->clip->fShader->Load();

            const nw::g3d::ResVertex* vertex = obj->shape->GetResVertex();
            int bufferCount = vertex->GetVtxBufferCount();
            for (int i = 0; i < bufferCount; ++i)
            {
                const nw::g3d::ResBuffer* vtxBuffer = vertex->GetVtxBuffer(i);

                vtxBuffer->GetGfxBuffer()->LoadVertices(i);
            }
            obj->shape->GetResMesh()->Draw();
        }
    }
} // NOLINT (readability/fn_size)
} // namespace

int DataEditMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    // 初期化処理。
#if defined(PLATFORM) && PLATFORM == CAFE
    DEMOInit();
#endif

    nw::g3d::demo::InitDisplayArg initDisplayArg;
#ifndef ENABLE_DRC_OUTPUT
    initDisplayArg.modeDRC = GX2_DRC_NONE;
#endif
    nw::g3d::demo::Init();
    nw::g3d::demo::InitDisplay(initDisplayArg);

    nw::g3d::demo::Pad pad;
    if (!pad.Reset())
    {
        nw::g3d::demo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext* pCtx = nw::g3d::demo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(nw::g3d::GfxContext), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。

    nw::g3d::demo::FrameBuffer frameBufferTV;
    {
        nw::g3d::demo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = nw::g3d::demo::FrameBuffer::CalcSize(initArg);
        frameBufferTV.Init(initArg, nw::g3d::demo::AllocMem2(bufferSize), bufferSize);
        frameBufferTV.Setup();
    }

    nw::g3d::GfxColorBuffer& colorBufferTV =
        frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBufferTV = frameBufferTV.GetDepthBufferTexture()->renderBuffer;

#if NW_G3D_CONFIG_USE_HOSTIO
    // Callback インスタンス生成
    SandboxNwG3dEditor::CreateInstance(
        &s_Files, &s_ShaderArchives, &s_Models, &s_RenderModels,
        s_ClearColor[0], s_ClearColor[1], s_ClearColor[2]);

    {
        nw::g3d::edit::EditServer::CreateArg arg;
        arg.allocator = &sAllocator;
#ifndef _WIN32//VisualC++の仕様
        arg.codePage = 65001; // UTF8
#else
        arg.codePage = 932;//ShiftJIS
#endif
        arg.editCallback = SandboxNwG3dEditor::Instance();

        bool createResult = nw::g3d::edit::EditServer::CreateInstance( arg );
        NW_G3D_ASSERT(createResult);

        bool openConnectionResult = nw::g3d::edit::EditServer::Instance().Open();
        NW_G3D_ASSERT(openConnectionResult);
    }
#endif

#ifdef USE_FSD_SETUP
#ifdef _WIN32
    // PC 版は 3dShaderConverter.dll の初期化を行う。
    const char* options[] =
    {
        "--silent", NULL,
        NULL
    };
    nw::g3d::ResShaderArchive::InitShdrCvtr(options);
#endif
#endif

    SetupFile(FILES[LOAD_BFRES], s_Files);
#ifdef USE_FSD_SETUP
    SetupShaderArchive(FILES[LOAD_BFSHA], FILES[LOAD_FSDB], s_ShaderArchives);
#else
    SetupShaderArchive(FILES[LOAD_BFSHA], s_ShaderArchives);
#endif

    int maxSklCount = 0;
    int maxMatCount = 0;
    int maxShapeCount = 0;
    for (int i = 0; i < static_cast<int>(s_Files.size()); ++i)
    {
        // モデルインスタンスの構築
        for (int j = 0; j < s_Files[i]->GetModelCount(); ++j)
        {
            nw::g3d::ResModel* rModel = s_Files[i]->GetModel(j);
            NW_G3D_ASSERT_NOT_NULL(rModel);
            int boneCount = rModel->GetSkeleton()->GetBoneCount();
            maxSklCount =
                (boneCount > maxSklCount) ? boneCount : maxSklCount;
            int matCount = rModel->GetMaterialCount();
            maxMatCount =
                (matCount > maxMatCount) ? matCount : maxMatCount;
            int shapeCount = rModel->GetShapeCount();
            maxShapeCount =
                (shapeCount > maxShapeCount) ? shapeCount : maxShapeCount;
            BuildModel(rModel, s_Models);
            RenderModel* renderModel = new (nw::g3d::demo::AllocMem2(sizeof(RenderModel))) RenderModel();
            renderModel->mdl = s_Models.back()->mdl;
            s_RenderModels.push_back(renderModel);
        }
    }

    for (int i = 0; i < static_cast<int>(s_Files.size()); ++i)
    {
        // アニメーションインスタンスの構築
        for (int j = 0; j < s_Files[i]->GetSkeletalAnimCount(); ++j)
        {
            BuildSklAnim(s_Files[i]->GetSkeletalAnim(j), maxSklCount, sSklAnims);
        }
        for (int j = 0; j < s_Files[i]->GetBoneVisAnimCount(); ++j)
        {
            BuildBoneVisAnim(s_Files[i]->GetBoneVisAnim(j), maxSklCount, sBoneVisAnims);
        }
        for (int j = 0; j < s_Files[i]->GetSceneAnimCount(); ++j)
        {
            BuildSceneAnim(s_Files[i]->GetSceneAnim(j), sCameraAnims, sLightAnims, sFogAnims);
        }
        for (int j = 0; j < s_Files[i]->GetTexPatternAnimCount(); ++j)
        {
            BuildTexPatternAnim(s_Files[i]->GetTexPatternAnim(j), maxMatCount, sTexAnims);
        }
        for (int j = 0; j < s_Files[i]->GetShapeAnimCount(); ++j)
        {
            BuildShapeAnim(s_Files[i]->GetShapeAnim(j), maxShapeCount, sShapeAnims);
        }
        for (int j = 0; j < s_Files[i]->GetTexSrtAnimCount(); ++j)
        {
            BuildParamAnim(s_Files[i]->GetTexSrtAnim(j), maxMatCount, sParamAnims);
        }
        for (int j = 0; j < s_Files[i]->GetColorAnimCount(); ++j)
        {
            BuildParamAnim(s_Files[i]->GetColorAnim(j), maxMatCount, sParamAnims);
        }
        for (int j = 0; j < s_Files[i]->GetShaderParamAnimCount(); ++j)
        {
            BuildParamAnim(s_Files[i]->GetShaderParamAnim(j), maxMatCount, sParamAnims);
        }
    }

    // コンテキスト有効化
    pCtx->Activate();

    // シェーダーバインド
    for (int i = 0; i < static_cast<int>(s_RenderModels.size()); ++i)
    {
        SetupShaderAssign(s_RenderModels[i]->mdl->GetResource(), NULL, s_ShaderArchives);
        BindShader(s_RenderModels[i], true);
        SetupShaderBinding(s_RenderModels[i]);
    }

    // アニメーションバインド
    for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
    {
        BindSklAnim(s_Models[i], sSklAnims);
        BindBoneVisAnim(s_Models[i], sBoneVisAnims);
        BindTexAnim(s_Models[i], sTexAnims);
        BindShapeAnim(s_Models[i], sShapeAnims);
        BindParamAnim(s_Models[i], sParamAnims);
    }

    // 全モデル共通の UBO 構築

    // viewMatrix/projectionMatrix の計算
    {
        s_UseCamera = false;

        s_CameraObj.pos = nw::g3d::math::Vec3::Make(20.0f, 20.0f, 80.0f);
        s_CameraObj.up = nw::g3d::math::Vec3::Make(0.0f, 1.0f, 0.0f);
        s_CameraObj.target = nw::g3d::math::Vec3::Make(0.0f, 7.5f, 0.0f);

        s_CameraObj.fovy = nw::g3d::math::Math::Pi() * 27.0f / 180.0f;
        s_CameraObj.aspect = 1280.0f / 720.0f;
        s_CameraObj.nearZ = 0.1f;
        s_CameraObj.farZ = 1000.0f;

        s_CameraObj.isOrtho = false;
        s_CameraObj.Calc();

        UniformBlockClip clip;
        clip.name = "view";
        clip.uBlock = new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::fnd::GfxBuffer))) nw::g3d::fnd::GfxBuffer();
        void* ubo = nw::g3d::demo::AllocMem2(sizeof(CameraBlock), GX2_UNIFORM_BLOCK_ALIGNMENT);
        clip.uBlock->SetData(ubo, sizeof(CameraBlock));
        clip.uBlock->Setup();

        s_UniformBlockBox.push_back(clip);
    }

    // env の計算
    {
        const nw::g3d::math::Mtx34& view = s_CameraObj.GetViewMtx();
        nw::g3d::math::Vec3 temp = nw::g3d::math::Vec3::Make(0.33f, 0.66f, 0.33f);
        s_EnvObj.hemiDir.x = view.m00 * temp.x + view.m01 * temp.y + view.m02 * temp.z;
        s_EnvObj.hemiDir.y = view.m10 * temp.x + view.m11 * temp.y + view.m12 * temp.z;
        s_EnvObj.hemiDir.z = view.m20 * temp.x + view.m21 * temp.y + view.m22 * temp.z;
        s_EnvObj.skyColor.Set(s_SkyColor);
        s_EnvObj.groundColor.Set(s_GroundColor);
        s_EnvObj.CalcHemisphereLight();

        s_EnvObj.fogStartZ = s_EnvObj.fogEndZ = 0.0f;
        s_EnvObj.fogDist[0] = s_EnvObj.fogDist[1] = 0.0f;
        s_EnvObj.fogColor.Set(s_FogColor);
        s_EnvObj.CalcFog();

        s_EnvObj.lightDir.Set(s_EnvObj.hemiDir);
        s_EnvObj.diffuseColor.Set(s_DiffuseColor);
        s_EnvObj.specularColor.Set(s_SpecularColor);
        s_EnvObj.CalcDirectionalLight();

        UniformBlockClip clip;
        clip.name = "env";
        clip.uBlock = new(nw::g3d::demo::AllocMem2(sizeof(nw::g3d::fnd::GfxBuffer))) nw::g3d::fnd::GfxBuffer();
        void* ubo = nw::g3d::demo::AllocMem2(sizeof(EnvBlock), GX2_UNIFORM_BLOCK_ALIGNMENT);
        clip.uBlock->SetData(ubo, sizeof(EnvBlock));
        clip.uBlock->Setup();

        s_UniformBlockBox.push_back(clip);
    }

    frameBufferTV.Alloc(nw::g3d::demo::AllocMem1);

    // メインループ
    u32 counter = 0;
    while (nw::g3d::demo::ProcessMsg())
    {
        counter += 0x1000000;

        if (!pad.Read() || pad.IsTriggered(nw::g3d::demo::Pad::BUTTON_START))
        {
            nw::g3d::demo::PostQuitMsg();
        }

        // 計算
        {
            Calc(pad);

            if (s_UseCamera)
            {
                nw::g3d::demo::ControllCamera(&s_CameraObj.pos, &s_CameraObj.up, &s_CameraObj.target, &s_CameraObj.GetViewMtx(), &pad);
                s_CameraObj.Calc();
            }

#if NW_G3D_CONFIG_USE_HOSTIO
            SandboxNwG3dEditor::Instance()->CalcEditor();

#if 0
            for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
            {
                nw::g3d::ModelObj* mdl = s_Models[i]->mdl;
                int boundCount = nw::g3d::edit::EditServer::Instance().GetModelAnimBoundCount(mdl);
                for (int j = 0; j < boundCount; ++j)
                {
                    int editAnimIndex = nw::g3d::edit::EditServer::Instance().GetActiveModelAnimIndex(mdl, j);
                    const char* editAnimName = nw::g3d::edit::EditServer::Instance().GetModelAnimName(editAnimIndex);
                    nw::g3d::DebugPrint("%d : %s\n", editAnimIndex, editAnimName);
                }
            }
#endif

#endif
        }

        {
            pCtx->Activate();
#if NW_G3D_CONFIG_USE_HOSTIO
            SandboxNwG3dEditor::Instance()->CalcBlock();
#endif
            // UniformBlock 計算
            CalcBlock();

            // メモリバリア
            nw::g3d::CPUCache::Sync();
        }

        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();

        // TV 描画。
        {
            nw::g3d::ClearBuffers(&colorBufferTV, &depthBufferTV,
                s_ClearColor[0], s_ClearColor[1], s_ClearColor[2], 1.0f, GX2_CLEAR_BOTH);

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            frameBufferTV.Load();

            Draw();

#if defined(_WIN32)
            glUseProgram(0);
            glBindVertexArray(0);
#else
            // シェーダモード切り替え
            nw::g3d:: SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);
#endif

            nw::g3d::demo::CopyOut(&colorBufferTV, GX2_SCAN_TARGET_TV);
#ifdef ENABLE_DRC_OUTPUT
            nw::g3d::demo::CopyOut(&colorBufferTV, GX2_SCAN_TARGET_DRC_FIRST);
#endif
        }

        nw::g3d::demo::SwapScanBuffers();
        nw::g3d::GfxManage::FlushCommands();
    }

    // グラフィックスリソースの終了処理。
    nw::g3d::GfxContext::Prepare(); // GL のテクスチャ解放に GL コンテクストが必要。

    frameBufferTV.Free(nw::g3d::demo::FreeMem1);

    for (std::vector<UniformBlockClip>::iterator iter = s_UniformBlockBox.begin(); iter != s_UniformBlockBox.end(); ++iter)
    {
        DeleteUniformBlockClip(*iter);
    }
    s_UniformBlockBox.clear();
    std::vector<UniformBlockClip>().swap(s_UniformBlockBox);


    // アニメーションの破棄
    {
        for (int i = 0; i < static_cast<int>(sParamAnims.size()); ++i)
        {
            DeleteAnim(sParamAnims[i]);
        }
        sParamAnims.clear();
        std::vector<nw::g3d::ShaderParamAnimObj*>().swap(sParamAnims);

        for (int i = 0; i < static_cast<int>(sShapeAnims.size()); ++i)
        {
            DeleteAnim(sShapeAnims[i]);
        }
        sShapeAnims.clear();
        std::vector<nw::g3d::ShapeAnimObj*>().swap(sShapeAnims);

        for (int i = 0; i < static_cast<int>(sTexAnims.size()); ++i)
        {
            DeleteAnim(sTexAnims[i]);
        }
        sTexAnims.clear();
        std::vector<nw::g3d::TexPatternAnimObj*>().swap(sTexAnims);

        for (int i = 0; i < static_cast<int>(sCameraAnims.size()); ++i)
        {
            DeleteAnim(sCameraAnims[i]);
        }
        sCameraAnims.clear();
        std::vector<nw::g3d::CameraAnimObj*>().swap(sCameraAnims);

        for (int i = 0; i < static_cast<int>(sLightAnims.size()); ++i)
        {
            DeleteAnim(sLightAnims[i]);
        }
        sLightAnims.clear();
        std::vector<nw::g3d::LightAnimObj*>().swap(sLightAnims);

        for (int i = 0; i < static_cast<int>(sFogAnims.size()); ++i)
        {
            DeleteAnim(sFogAnims[i]);
        }
        sFogAnims.clear();
        std::vector<nw::g3d::FogAnimObj*>().swap(sFogAnims);

        for (int i = 0; i < static_cast<int>(sBoneVisAnims.size()); ++i)
        {
            DeleteAnim(sBoneVisAnims[i]);
        }
        sBoneVisAnims.clear();
        std::vector<nw::g3d::VisibilityAnimObj*>().swap(sBoneVisAnims);

        for (int i = 0; i < static_cast<int>(sSklAnims.size()); ++i)
        {
            DeleteAnim(sSklAnims[i]);
        }
        sSklAnims.clear();
        std::vector<nw::g3d::SkeletalAnimObj*>().swap(sSklAnims);
    }

#if NW_G3D_CONFIG_USE_HOSTIO
    {
        nw::g3d::edit::EditServer::Instance().Clear(true);
        nw::g3d::edit::EditServer::Instance().Close();
        nw::g3d::edit::EditServer::DeleteInstance();
        SandboxNwG3dEditor::DeleteInstance();
    }
#endif

    for (std::vector<RenderModel*>::iterator iter = s_RenderModels.begin(); iter != s_RenderModels.end(); ++iter)
    {
        DeleteRenderModel(*iter);
    }
    s_RenderModels.clear();
    std::vector<RenderModel*>().swap(s_RenderModels);

    for (int i = 0; i < static_cast<int>(s_Models.size()); ++i)
    {
        DeleteModel(s_Models[i]);
    }
    s_Models.clear();
    std::vector<AnimModelObj*>().swap(s_Models);

    for (int i = 0; i < static_cast<int>(s_ShaderArchives.size()); ++i)
    {
        CleanupShaderArchive(s_ShaderArchives[i]);
    }
    s_ShaderArchives.clear();
    std::vector<nw::g3d::ResShaderArchive*>().swap(s_ShaderArchives);

    for (int i = 0; i < static_cast<int>(s_Files.size()); ++i)
    {
        CleanupFile(s_Files[i]);
    }
    s_Files.clear();
    std::vector<nw::g3d::ResFile*>().swap(s_Files);

    frameBufferTV.Cleanup();
    nw::g3d::demo::FreeMem2(frameBufferTV.GetBufferPtr());

    pCtx->Cleanup();
    nw::g3d::demo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    nw::g3d::demo::ShutdownDisplay();
    nw::g3d::demo::Shutdown();

    sAllocator.PrintMemoryLeaks();

#if defined(PLATFORM) && PLATFORM == CAFE
    DEMOShutdown();
#endif

    // 再入しない。
    return 0;
} // NOLINT (readability/fn_size)
