﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using NintendoWare.G3d.Edit;
using System.Diagnostics;
using System.Threading;

namespace G3dHioCommonTest
{
    [TestClass]
    public class SendCommandQueueTest
    {
        public class DummyModel : IEditModelTarget
        {
            public uint ResModelKey { get { return 0; } }
            public uint ModelObjKey { get { return 0; } }
            public bool IsSendAttached { get { return false; } }
            public uint Key { get { return 0; } }
            public uint ResFileKey { get { return 0; } }
            public bool IsAttached { get { return false; } }
            public bool IsDeleted { get { return false; } }
            public void ResetStatus() { }
        }

        public class DummySystemBeginCommand : ISendCommand
        {
            public byte[] PacketBuffer { get { return null; } }
            public bool MakeCommandPacket() { return true; }
            public bool IsSystemBeginCommand { get { return true; } }
            public bool IsSystemEndCommand { get { return false; } }
            public bool CanProcess() { return true; }
            public TargetEndianKind TargetEndian { get { return TargetEndianKind.LittleEndian; } }
            public uint NumberOfTrials { get { return 1000; } }
            public bool IsInOrder { get { return false; } }
            public void OnRemoved() { }
        }

        public class DummySystemEndCommand : ISendCommand
        {
            public byte[] PacketBuffer { get { return null; } }
            public bool MakeCommandPacket() { return true; }
            public bool IsSystemBeginCommand { get { return false; } }
            public bool IsSystemEndCommand { get { return true; } }
            public bool CanProcess() { return true; }
            public TargetEndianKind TargetEndian { get { return TargetEndianKind.LittleEndian; } }
            public uint NumberOfTrials { get { return 1000; } }
            public bool IsInOrder { get { return false; } }
            public void OnRemoved() { }
        }

        public class DummyCommand : ISendCommand
        {
            public byte[] PacketBuffer { get { return null; } }
            public bool MakeCommandPacket() { return true; }
            public bool IsSystemBeginCommand { get { return false; } }
            public bool IsSystemEndCommand { get { return false; } }
            public bool CanProcess() { return true; }
            public TargetEndianKind TargetEndian { get { return TargetEndianKind.LittleEndian; } }
            public uint NumberOfTrials { get { return 1000; } }
            public bool IsInOrder { get { return false; } }
            public void OnRemoved() { }
        }

        [TestMethod]
        public void TestCommandQueueMutexLock()
        {
            SendCommandQueue queue = new SendCommandQueue();

            var beginCommand = new DummySystemBeginCommand();
            var endCommand = new DummySystemEndCommand();
            var command1 = new DummyCommand();
            var command2 = new DummyCommand();
            var command3 = new DummyCommand();

            queue.PushBack(command1);
            queue.PushBack(command2);
            Assert.AreEqual(2, queue.Count);

            Thread thread = new Thread(() =>
                {
                    try
                    {
                        while (queue.IsCreatingTransaction) { }

                        // トランザクション用コマンドとbiginCommandとendCommandが追加されているはず
                        Assert.AreEqual(5, queue.Count);
                        queue.Remove(command1);
                        queue.Remove(command2);
                        Assert.AreEqual(3, queue.Count);
                    }
                    catch (Exception e)
                    {
                        Assert.Fail(e.Message);
                    }
                });

            // トランザクション作成中に排他制御ができているかテスト
            try
            {
                using (queue.CreateCommandTransaction(beginCommand, endCommand))
                {
                    thread.Start();

                    while (!thread.IsAlive) { }

                    // 削除処理が割り込みで入らないことを確認するためにわざと待つ
                    Thread.Sleep(1000);

                    queue.PushBack(command3);

                    // 別スレッドからコマンドが消されていないことを確認
                    Assert.AreEqual(3, queue.Count);
                }
            }
            catch (Exception)
            {
                Assert.Fail();
            }

            thread.Join();

            // 最終的にトランザクションだけが残るはず
            Assert.AreEqual(3, queue.Count);
        }

        [TestMethod]
        public void TestTransactionCommandQueue()
        {
            SendCommandQueue queue = new SendCommandQueue();

            var beginCommand = new DummySystemBeginCommand();
            var endCommand = new DummySystemEndCommand();

            // 通常のテスト
            {
                queue.PushBack(new DummyCommand());
                queue.PushBack(new DummyCommand());
                queue.PushBack(new DummyCommand());
                Assert.AreEqual(3, queue.Count);

                var command1 = new DummyCommand();
                var command2 = new DummyCommand();
                var command3 = new DummyCommand();

                using (queue.CreateCommandTransaction(beginCommand, endCommand))
                {
                    queue.PushBack(command1);
                    queue.PushBack(command2);
                    queue.PushBack(command3);
                    Assert.AreEqual(6, queue.Count);
                }

                Assert.AreEqual(8, queue.Count);

                queue.PopFront();
                queue.PopFront();
                queue.PopFront();
                Assert.AreEqual(beginCommand, queue.PopFront());
                Assert.AreEqual(command1, queue.PopFront());
                Assert.AreEqual(command2, queue.PopFront());
                Assert.AreEqual(command3, queue.PopFront());
                Assert.AreEqual(endCommand, queue.PopFront());
            }

            // トランザクションのDisposeが呼ばれる前にコマンドが減った場合
            {
                queue.PushBack(new DummyCommand());
                queue.PushBack(new DummyCommand());
                queue.PushBack(new DummyCommand());
                Assert.AreEqual(3, queue.Count);
                using (queue.CreateCommandTransaction(beginCommand, endCommand))
                {
                    queue.PopBack();
                    Assert.AreEqual(2, queue.Count);
                }

                Assert.AreEqual(4, queue.Count);
                {
                    var commandPopped = queue.PopFront();
                    Assert.IsFalse(commandPopped.IsSystemBeginCommand);
                    Assert.IsFalse(commandPopped.IsSystemEndCommand);
                }

                {
                    var commandPopped = queue.PopFront();
                    Assert.IsFalse(commandPopped.IsSystemBeginCommand);
                    Assert.IsFalse(commandPopped.IsSystemEndCommand);
                }
            }
        }
    }
}
