﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnd/tmp451/tmp451.h>

namespace nnt { namespace tmp451 {

struct PorValues
{
    nnd::tmp451::MeasurementRange   range;
    int                             n;
    bool                            timeoutEnable;
};

const PorValues g_PorValues = {nnd::tmp451::MeasurementRange::Standard, 4, false};
const PorValues g_ChangeValues = {nnd::tmp451::MeasurementRange::Extended, 2, true};


// 計測中に設定の取得が可能なことを確認するテスト
TEST(MeasurementSettingTest, Get)
{
    using namespace nnd::tmp451;
    Initialize();

    auto result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    MeasurementRange range;
    int n;
    bool timeoutEnable;
    result = GetMeasurementRange(&range);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementRateBy2ToThePowerN(&n);
    EXPECT_TRUE(result.IsSuccess());
    result = GetBusTimeoutFunctionEnabled(&timeoutEnable);
    EXPECT_TRUE(result.IsSuccess());

    // 初期値の確認
    EXPECT_EQ(g_PorValues.range, range);
    EXPECT_EQ(g_PorValues.n, n);
    EXPECT_EQ(g_PorValues.timeoutEnable, timeoutEnable);

    Finalize();
}

// 設定の変更ができることを確認するテスト
TEST(MeasurementSettingTest, Set)
{
    using namespace nnd::tmp451;
    Initialize();

    // 設定を変更
    auto result = SetMeasurementRange(g_ChangeValues.range);
    EXPECT_TRUE(result.IsSuccess());
    result = SetMeasurementRateBy2ToThePowerN(g_ChangeValues.n);
    EXPECT_TRUE(result.IsSuccess());
    result = SetBusTimeoutFunctionEnabled(g_ChangeValues.timeoutEnable);
    EXPECT_TRUE(result.IsSuccess());

    // 設定が変更されているかの確認
    MeasurementRange rangeLatest;
    int nLatest;
    bool timeoutEnableLatest;
    result = GetMeasurementRange(&rangeLatest);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementRateBy2ToThePowerN(&nLatest);
    EXPECT_TRUE(result.IsSuccess());
    result = GetBusTimeoutFunctionEnabled(&timeoutEnableLatest);
    EXPECT_TRUE(result.IsSuccess());

    // 比較
    EXPECT_EQ(g_ChangeValues.range, rangeLatest);
    EXPECT_EQ(g_ChangeValues.n, nLatest);
    EXPECT_EQ(g_ChangeValues.timeoutEnable, timeoutEnableLatest);

    Finalize();
}

// 計測中に設定の変更ができないことを確認するテスト
TEST(MeasurementSettingTest, SetError)
{
    using namespace nnd::tmp451;
    Initialize();

    // 初期値を取得
    MeasurementRange range;
    int n;
    bool timeoutEnable;
    auto result = GetMeasurementRange(&range);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementRateBy2ToThePowerN(&n);
    EXPECT_TRUE(result.IsSuccess());
    result = GetBusTimeoutFunctionEnabled(&timeoutEnable);
    EXPECT_TRUE(result.IsSuccess());

    // 計測開始
    result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());

    // 計測中はエラーになるはず
    result = SetMeasurementRange(range);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetMeasurementRateBy2ToThePowerN(n);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);
    result = SetBusTimeoutFunctionEnabled(timeoutEnable);
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidState, result);

    Finalize();
}

}}
