﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_CalendarTime.h>

using namespace ::nn::time;

namespace
{
    const int TestLeapYearList[] =
    {
        1600,

        1980,
        1984,
        1988,
        1992,
        1996,
        1996,
        2000,
        2004,
        2008,
        2012,
        2016,
        2020,
        2024,
        2028,
        2032,
        2036,
        2040,

        2400,
    };

    const int TestNotLeapYearList[] =
    {
        1700,
        1800,
        1900,

        // 1980,
        1981,
        1982,
        1983,
        // 1984,
        1985,
        1986,
        1987,
        // 1988,
        1989,
        1990,
        1991,
        // 1992,
        1993,
        1994,
        1995,
        // 1996,
        1997,
        1998,
        1999,
        // 2000, // 400で割り切れる
        2001,
        2002,
        2003,
        // 2004,
        2005,
        2006,
        2007,
        // 2008,
        2009,
        2010,
        2011,
        // 2012,
        2013,
        2014,
        2015,
        // 2016,
        2017,
        2018,
        2019,
        // 2020,


        2100,
        2200,
    };
}

TEST(TimeUtility, IsLeapYear)
{
    for(auto y : TestLeapYearList)
    {
        EXPECT_TRUE(IsLeapYear(y)) << "year:" << y;
    }

    for(auto y : TestNotLeapYearList)
    {
        EXPECT_FALSE(IsLeapYear(y)) << "year" << y;
    }
}

TEST(TimeUtility, GetDaysInMonth)
{
    for(auto y : TestLeapYearList)
    {
        EXPECT_EQ(31, GetDaysInMonth(y,  1)) << "year:" << y;
        EXPECT_EQ(29, GetDaysInMonth(y,  2)) << "year:" << y; // leap year
        EXPECT_EQ(31, GetDaysInMonth(y,  3)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  4)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  5)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  6)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  7)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  8)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  9)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y, 10)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y, 11)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y, 12)) << "year:" << y;
    }

    for(auto y : TestNotLeapYearList)
    {
        EXPECT_EQ(31, GetDaysInMonth(y,  1)) << "year:" << y;
        EXPECT_EQ(28, GetDaysInMonth(y,  2)) << "year:" << y; // not leap year
        EXPECT_EQ(31, GetDaysInMonth(y,  3)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  4)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  5)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  6)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  7)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y,  8)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y,  9)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y, 10)) << "year:" << y;
        EXPECT_EQ(30, GetDaysInMonth(y, 11)) << "year:" << y;
        EXPECT_EQ(31, GetDaysInMonth(y, 12)) << "year:" << y;
    }
}

TEST(TimeUtility, IsValidDate)
{
    EXPECT_TRUE(IsValidDate(1, 1, 1));

    for(auto y : TestLeapYearList)
    {
        // 1月
        EXPECT_TRUE (IsValidDate(y, 1,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 1, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 1, 32)) << "year:" << y;
        // 2月
        EXPECT_TRUE (IsValidDate(y, 2,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 2, 29)) << "year:" << y; // leap year
        EXPECT_FALSE(IsValidDate(y, 2, 30)) << "year:" << y;
        // 3月
        EXPECT_TRUE (IsValidDate(y, 3,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 3, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 3, 32)) << "year:" << y;
        // 4月
        EXPECT_TRUE (IsValidDate(y, 4,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 4, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 4, 31)) << "year:" << y;
        // 5月
        EXPECT_TRUE (IsValidDate(y, 5,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 5, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 5, 32)) << "year:" << y;
        // 6月
        EXPECT_TRUE (IsValidDate(y, 6,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 6, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 6, 31)) << "year:" << y;
        // 7月
        EXPECT_TRUE (IsValidDate(y, 7,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 7, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 7, 32)) << "year:" << y;
        // 8月
        EXPECT_TRUE (IsValidDate(y, 8,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 8, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 8, 32)) << "year:" << y;
        // 9月
        EXPECT_TRUE (IsValidDate(y, 9,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 9, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 9, 31)) << "year:" << y;
        // 10月
        EXPECT_TRUE (IsValidDate(y, 10,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 10, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 10, 32)) << "year:" << y;
        // 11月
        EXPECT_TRUE (IsValidDate(y, 11,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 11, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 11, 31)) << "year:" << y;
        // 12月
        EXPECT_TRUE (IsValidDate(y, 12,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 12, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 12, 32)) << "year:" << y;
    }

    for(auto y : TestNotLeapYearList)
    {
        // 1月
        EXPECT_TRUE (IsValidDate(y, 1,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 1, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 1, 32)) << "year:" << y;
        // 2月
        EXPECT_TRUE (IsValidDate(y, 2,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 2, 28)) << "year:" << y; // not leap year
        EXPECT_FALSE(IsValidDate(y, 2, 29)) << "year:" << y;
        // 3月
        EXPECT_TRUE (IsValidDate(y, 3,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 3, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 3, 32)) << "year:" << y;
        // 4月
        EXPECT_TRUE (IsValidDate(y, 4,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 4, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 4, 31)) << "year:" << y;
        // 5月
        EXPECT_TRUE (IsValidDate(y, 5,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 5, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 5, 32)) << "year:" << y;
        // 6月
        EXPECT_TRUE (IsValidDate(y, 6,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 6, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 6, 31)) << "year:" << y;
        // 7月
        EXPECT_TRUE (IsValidDate(y, 7,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 7, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 7, 32)) << "year:" << y;
        // 8月
        EXPECT_TRUE (IsValidDate(y, 8,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 8, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 8, 32)) << "year:" << y;
        // 9月
        EXPECT_TRUE (IsValidDate(y, 9,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 9, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 9, 31)) << "year:" << y;
        // 10月
        EXPECT_TRUE (IsValidDate(y, 10,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 10, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 10, 32)) << "year:" << y;
        // 11月
        EXPECT_TRUE (IsValidDate(y, 11,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 11, 30)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 11, 31)) << "year:" << y;
        // 12月
        EXPECT_TRUE (IsValidDate(y, 12,  1)) << "year:" << y;
        EXPECT_TRUE (IsValidDate(y, 12, 31)) << "year:" << y;
        EXPECT_FALSE(IsValidDate(y, 12, 32)) << "year:" << y;
    }
}

TEST(TimeUtility, DateToDays)
{
    int year = 1;
    int month = 1;
    int day = 1;

    // 1000000 日経過で、2738年くらいなのでそこまで適当にテスト

    for(int days = 0 ; days < 1000000 ; ++days)
    {
        EXPECT_EQ(days, DateToDays(year, month, day)) << "year:" << year << " month:" << month << " day:" << day;

        day++;
        if(day > GetDaysInMonth(year, month))
        {
            day = 1;
            month++;
        }
        if(month > 12)
        {
            month = 1;
            year++;
        }
    }
}

TEST(TimeUtility, DateToDaysOutOfDate)
{
    EXPECT_EQ(DateToDays(2000, 2, 1), DateToDays(2000,  1, 32));
    EXPECT_EQ(DateToDays(2000, 12, 31), DateToDays(2000,  1, 366));

    EXPECT_EQ(DateToDays(2001, 1, 1), DateToDays(2000,  1, 367));
    EXPECT_EQ(DateToDays(2002, 1, 1), DateToDays(2000,  1, 367 + 365));
    EXPECT_EQ(DateToDays(2003, 1, 1), DateToDays(2000,  1, 367 + 365 + 365));
    EXPECT_EQ(DateToDays(2004, 1, 1), DateToDays(2000,  1, 367 + 365 + 365 + 365));

    EXPECT_EQ(DateToDays(2015, 12, 31), DateToDays(2015,  1, 365));
    EXPECT_EQ(DateToDays(2016, 1, 1), DateToDays(2015,  1, 366));
    EXPECT_EQ(DateToDays(2015, 2, 28), DateToDays(2015,  1, 59));
    EXPECT_EQ(DateToDays(2015, 3,  1), DateToDays(2015,  1, 60));
    EXPECT_EQ(DateToDays(2015, 3,  2), DateToDays(2015,  2, 30));
    EXPECT_EQ(DateToDays(2016, 2, 29), DateToDays(2016,  1, 60));
    EXPECT_EQ(DateToDays(2016, 3,  1), DateToDays(2016,  1, 61));
    EXPECT_EQ(DateToDays(2016, 3,  1), DateToDays(2016,  2, 30));
    EXPECT_EQ(DateToDays(2016, 12, 31), DateToDays(2016,  1, 366));
    EXPECT_EQ(DateToDays(2017, 1, 1), DateToDays(2016,  1, 367));

    EXPECT_EQ(DateToDays(2017,  1, 1), DateToDays(2016, 13, 1));
    EXPECT_EQ(DateToDays(2017,  2, 1), DateToDays(2016, 14, 1));
    EXPECT_EQ(DateToDays(2017,  3, 1), DateToDays(2016, 15, 1));
    EXPECT_EQ(DateToDays(2017,  4, 1), DateToDays(2016, 16, 1));
    EXPECT_EQ(DateToDays(2017,  5, 1), DateToDays(2016, 17, 1));
    EXPECT_EQ(DateToDays(2017,  6, 1), DateToDays(2016, 18, 1));
    EXPECT_EQ(DateToDays(2017,  7, 1), DateToDays(2016, 19, 1));
    EXPECT_EQ(DateToDays(2017,  8, 1), DateToDays(2016, 20, 1));
    EXPECT_EQ(DateToDays(2017,  9, 1), DateToDays(2016, 21, 1));
    EXPECT_EQ(DateToDays(2017, 10, 1), DateToDays(2016, 22, 1));
    EXPECT_EQ(DateToDays(2017, 11, 1), DateToDays(2016, 23, 1));
    EXPECT_EQ(DateToDays(2017, 12, 1), DateToDays(2016, 24, 1));
    EXPECT_EQ(DateToDays(2018,  1, 1), DateToDays(2016, 25, 1));
    EXPECT_EQ(DateToDays(2018,  2, 1), DateToDays(2016, 26, 1));
    EXPECT_EQ(DateToDays(2018,  3, 1), DateToDays(2016, 27, 1));
    EXPECT_EQ(DateToDays(2018,  4, 1), DateToDays(2016, 28, 1));
    EXPECT_EQ(DateToDays(2018,  5, 1), DateToDays(2016, 29, 1));
    EXPECT_EQ(DateToDays(2018,  6, 1), DateToDays(2016, 30, 1));
    EXPECT_EQ(DateToDays(2018,  7, 1), DateToDays(2016, 31, 1));
    EXPECT_EQ(DateToDays(2018,  8, 1), DateToDays(2016, 32, 1));
    EXPECT_EQ(DateToDays(2018,  9, 1), DateToDays(2016, 33, 1));
    EXPECT_EQ(DateToDays(2018, 10, 1), DateToDays(2016, 34, 1));
    EXPECT_EQ(DateToDays(2018, 11, 1), DateToDays(2016, 35, 1));
    EXPECT_EQ(DateToDays(2018, 12, 1), DateToDays(2016, 36, 1));

    EXPECT_EQ(DateToDays(2019, 12, 1), DateToDays(2016, 48, 1));
}

TEST(TimeUtility, DaysToDate)
{
    int year = 1;
    int month = 1;
    int day = 1;

    // 1000000 日経過で、2738年くらいなのでそこまで適当にテスト

    for(int days = 0 ; days < 1000000 ; ++days)
    {
        int outYear;
        int outMonth;
        int outDay;
        DaysToDate(&outYear, &outMonth, &outDay, days);
        EXPECT_TRUE(IsValidDate(outYear, outMonth, outDay));
        EXPECT_EQ(year, outYear) <<
            "expect:" << year << "/" << month  << "/" << day << "    actual:" << outYear << "/" << outMonth << "/" << outDay;
        EXPECT_EQ(month, outMonth) <<
            "expect:" << year << "/" << month  << "/" << day << "    actual:" << outYear << "/" << outMonth << "/" << outDay;
        EXPECT_EQ(day, outDay) <<
            "expect:" << year << "/" << month  << "/" << day << "    actual:" << outYear << "/" << outMonth << "/" << outDay;

        EXPECT_EQ(days, DateToDays(outYear, outMonth, outDay));

        day++;
        if(day > GetDaysInMonth(year, month))
        {
            day = 1;
            month++;
        }
        if(month > 12)
        {
            month = 1;
            year++;
        }
    }
}


TEST(TimeUtility, GetDayOfWeek)
{

    int year = 1;
    int month = 1;
    int day = 1;

    // 1000000 日経過で、2738年くらいなのでそこまで適当にテスト
    for(int days = 0 ; days < 1000000 ; ++days)
    {
        auto dayOfWeek = GetDayOfWeek(year, month, day);
        DayOfWeek expect = static_cast<DayOfWeek>( (days + 1) % 7 );

        EXPECT_EQ(expect, dayOfWeek) <<
            "expect:" << expect << "    actual:" << dayOfWeek << " / " << year << "/" << month  << "/" << day;

        day++;
        if(day > GetDaysInMonth(year, month))
        {
            day = 1;
            month++;
        }
        if(month > 12)
        {
            month = 1;
            year++;
        }
    }

    // ピンポイントでも適当にテストしておく
    EXPECT_EQ(nn::time::DayOfWeek_Monday, GetDayOfWeek(1900, 1, 1));
    EXPECT_EQ(nn::time::DayOfWeek_Tuesday, GetDayOfWeek(1900, 1, 2));
    EXPECT_EQ(nn::time::DayOfWeek_Wednesday, GetDayOfWeek(1900, 1, 3));
    EXPECT_EQ(nn::time::DayOfWeek_Thursday, GetDayOfWeek(1900, 1, 4));

    EXPECT_EQ(nn::time::DayOfWeek_Saturday, GetDayOfWeek(2000, 1, 1));
    EXPECT_EQ(nn::time::DayOfWeek_Sunday, GetDayOfWeek(2000, 12, 31));

    EXPECT_EQ(nn::time::DayOfWeek_Thursday, GetDayOfWeek(2016, 3, 17));
    EXPECT_EQ(nn::time::DayOfWeek_Friday, GetDayOfWeek(2016, 3, 18));
    EXPECT_EQ(nn::time::DayOfWeek_Saturday, GetDayOfWeek(2016, 3, 19));
}

TEST(TimeUtility, CalendarIsValid)
{
    auto Test = [](int16_t year, int8_t month, int8_t day, int8_t hour, int8_t minute, int8_t second) NN_NOEXCEPT
    {
        nn::time::CalendarTime c = {
            year, month, day, hour, minute, second
        };

        return c.IsValid();
    };

    EXPECT_FALSE(Test(-1, 1, 1, 0, 0, 0));
    EXPECT_FALSE(Test(0, 1, 1, 0, 0, 0));
    EXPECT_TRUE(Test(1, 1, 1, 0, 0, 0));

    EXPECT_FALSE(Test(2018, 1, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 1, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 1, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 1, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 1, 1, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 1, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 1, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 1, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 1, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 1, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 2, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 2, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 2, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 2, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 2, 28, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 2, 28, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 2, 29, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 2, 28, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 2, 28, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 2, 28, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 3, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 3, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 3, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 3, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 3, 1, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 3, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 3, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 3, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 3, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 3, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 4, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 4, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 4, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 4, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 4, 1, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 4, 30, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 4, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 4, 30, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 4, 30, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 4, 30, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 5, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 5, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 5, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 5, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 5, 31, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 5, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 5, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 5, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 5, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 5, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 6, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 6, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 6, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 6, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 6, 30, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 6, 30, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 6, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 6, 30, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 6, 30, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 6, 30, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 7, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 7, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 7, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 7, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 7, 31, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 7, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 7, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 7, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 7, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 7, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 8, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 8, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 8, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 8, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 8, 31, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 8, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 8, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 8, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 8, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 8, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 9, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 9, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 9, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 9, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 9, 30, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 9, 30, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 9, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 9, 30, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 9, 30, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 9, 30, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 10, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 10, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 10, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 10, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 10, 31, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 10, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 10, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 10, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 10, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 10, 31, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 11, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 11, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 11, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 11, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 11, 30, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 11, 30, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 11, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 11, 30, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 11, 30, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 11, 30, 23, 59, 60));

    EXPECT_FALSE(Test(2018, 12, 0, 0, 0, 0));
    EXPECT_FALSE(Test(2018, 12, 1, -1, 0, 0));
    EXPECT_FALSE(Test(2018, 12, 1, 0, -1, 0));
    EXPECT_FALSE(Test(2018, 12, 1, 0, 0, -1));
    EXPECT_TRUE(Test(2018, 12, 31, 0, 0, 0));
    EXPECT_TRUE(Test(2018, 12, 31, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 12, 32, 23, 59, 59));
    EXPECT_FALSE(Test(2018, 12, 31, 24, 59, 59));
    EXPECT_FALSE(Test(2018, 12, 31, 23, 60, 59));
    EXPECT_FALSE(Test(2018, 12, 31, 23, 59, 60));

    // うるう年
    for(int16_t year = 1900 ; year < 2100 ; year++)
    {
        if(nn::time::IsLeapYear(year))
        {
            EXPECT_TRUE(Test(year, 2, 28, 23, 59, 59)) << year;
            EXPECT_TRUE(Test(year, 2, 29, 23, 59, 59)) << year;
            EXPECT_FALSE(Test(year, 2, 30, 23, 59, 59)) << year;
        }
        else
        {
            EXPECT_TRUE(Test(year, 2, 28, 23, 59, 59)) << year;
            EXPECT_FALSE(Test(year, 2, 29, 23, 59, 59)) << year;
            EXPECT_FALSE(Test(year, 2, 30, 23, 59, 59)) << year;
        }
    }
} // NOLINT(impl/function_size)

TEST(TimeUtility, CalendarTimeOperator)
{
    auto Calendar = [](int16_t year, int8_t month, int8_t day, int8_t hour, int8_t minute, int8_t second) NN_NOEXCEPT
    {
        nn::time::CalendarTime result = {
            year, month, day, hour, minute, second
        };
        return result;
    };

    {
        // 繰り上げなし, 繰り下げなし

        // +, +=
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 46), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 24, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 3, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 10, 3, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromDays(1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 46), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 24, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 3, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 10, 3, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromDays(1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 44), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 22, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 1, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 10, 1, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) + nn::TimeSpan::FromDays(-1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 44), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 22, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 1, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 10, 1, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) += nn::TimeSpan::FromDays(-1));

        // -, -=
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 46), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 24, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 3, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 10, 3, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromDays(-1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 46), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 24, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 10, 2, 3, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 10, 3, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromDays(-1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 44), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 22, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 1, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 10, 1, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) - nn::TimeSpan::FromDays(1));

        EXPECT_EQ(Calendar(2017, 10, 2, 2, 23, 44), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 2, 22, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 10, 2, 1, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 10, 1, 2, 23, 45), Calendar(2017, 10, 2, 2, 23, 45) -= nn::TimeSpan::FromDays(1));
    }

    {
        // 繰り上げあり

        // +, +=
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0,  0), Calendar(2017, 12, 31, 23, 59, 59) + nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0, 59), Calendar(2017, 12, 31, 23, 59, 59) + nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) + nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2018, 1, 1, 23, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) + nn::TimeSpan::FromDays(1));

        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0,  0), Calendar(2017, 12, 31, 23, 59, 59) += nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0, 59), Calendar(2017, 12, 31, 23, 59, 59) += nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) += nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2018, 1, 1, 23, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) += nn::TimeSpan::FromDays(1));

        // -, -=
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0,  0), Calendar(2017, 12, 31, 23, 59, 59) - nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0, 59), Calendar(2017, 12, 31, 23, 59, 59) - nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) - nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2018, 1, 1, 23, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) - nn::TimeSpan::FromDays(-1));

        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0,  0), Calendar(2017, 12, 31, 23, 59, 59) -= nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0,  0, 59), Calendar(2017, 12, 31, 23, 59, 59) -= nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2018, 1, 1,  0, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) -= nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2018, 1, 1, 23, 59, 59), Calendar(2017, 12, 31, 23, 59, 59) -= nn::TimeSpan::FromDays(-1));
    }
    {
        // 繰り下げあり

        // +, +=
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59, 59), Calendar(2018, 1, 1, 0, 0, 0) + nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59,  0), Calendar(2018, 1, 1, 0, 0, 0) + nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) + nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 12, 31,  0,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) + nn::TimeSpan::FromDays(-1));

        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59, 59), Calendar(2018, 1, 1, 0, 0, 0) += nn::TimeSpan::FromSeconds(-1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59,  0), Calendar(2018, 1, 1, 0, 0, 0) += nn::TimeSpan::FromMinutes(-1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) += nn::TimeSpan::FromHours(-1));
        EXPECT_EQ(Calendar(2017, 12, 31,  0,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) += nn::TimeSpan::FromDays(-1));

        // -, -=
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59, 59), Calendar(2018, 1, 1, 0, 0, 0) - nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59,  0), Calendar(2018, 1, 1, 0, 0, 0) - nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) - nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 12, 31,  0,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) - nn::TimeSpan::FromDays(1));

        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59, 59), Calendar(2018, 1, 1, 0, 0, 0) -= nn::TimeSpan::FromSeconds(1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23, 59,  0), Calendar(2018, 1, 1, 0, 0, 0) -= nn::TimeSpan::FromMinutes(1));
        EXPECT_EQ(Calendar(2017, 12, 31, 23,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) -= nn::TimeSpan::FromHours(1));
        EXPECT_EQ(Calendar(2017, 12, 31,  0,  0,  0), Calendar(2018, 1, 1, 0, 0, 0) -= nn::TimeSpan::FromDays(1));

    }

    {
        nn::TimeSpan span;

        span = Calendar(2017, 10, 2, 12, 34, 56) - Calendar(2017, 10, 2, 12, 34, 56);
        EXPECT_EQ(span, nn::TimeSpan(0));

        // 結果がプラス
        span = Calendar(2017, 10, 2, 12, 34, 56) - Calendar(2017, 10, 2, 12, 34, 55);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(1));

        span = Calendar(2017, 10, 2, 12, 34, 56) - Calendar(2017, 10, 2, 12, 33, 55);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(61));

        span = Calendar(2017, 10, 2, 12, 34, 56) - Calendar(2017, 10, 2, 11, 33, 55);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(3661));

        // 結果がマイナス
        span = Calendar(2017, 10, 2, 12, 34, 55) - Calendar(2017, 10, 2, 12, 34, 56);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(-1));

        span = Calendar(2017, 10, 2, 12, 33, 55) - Calendar(2017, 10, 2, 12, 34, 56);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(-61));

        span = Calendar(2017, 10, 2, 11, 33, 55) - Calendar(2017, 10, 2, 12, 34, 56);
        EXPECT_EQ(span, nn::TimeSpan::FromSeconds(-3661));
    }

    {
        // うるう年狙い撃ちテスト

        auto c = Calendar(2016, 2, 28, 0, 0, 0);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c + nn::TimeSpan::FromDays(1));
        c += nn::TimeSpan::FromDays(1);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c);

        c = Calendar(2016, 2, 28, 23, 59, 59);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c + nn::TimeSpan::FromSeconds(1));
        c += nn::TimeSpan::FromSeconds(1);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c);

        c = Calendar(2016, 3, 1, 0, 0, 0);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c - nn::TimeSpan::FromDays(1));
        c -= nn::TimeSpan::FromDays(1);
        EXPECT_EQ(Calendar(2016, 2, 29, 0, 0, 0), c);

        c = Calendar(2016, 3, 1, 0, 0, 0);
        EXPECT_EQ(Calendar(2016, 2, 29, 23, 59, 59), c - nn::TimeSpan::FromSeconds(1));
        c -= nn::TimeSpan::FromSeconds(1);
        EXPECT_EQ(Calendar(2016, 2, 29, 23, 59, 59), c);
    }

    {
        // ==
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017, 10, 2, 3, 45, 56));  // 同じ
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017, 10, 2, 3, 45, 55)); // second違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017, 10, 2, 3, 44, 56)); // minute違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017, 10, 2, 2, 45, 56)); // hour違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017, 10, 1, 3, 45, 56)); // day違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2017,  9, 2, 3, 45, 56)); // month違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) == Calendar(2016, 10, 2, 3, 45, 56)); // year違う
    }

    {
        // !=
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017, 10, 2, 3, 45, 56)); // 同じ
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017, 10, 2, 3, 45, 55));  // second違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017, 10, 2, 3, 44, 56));  // minute違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017, 10, 2, 2, 45, 56));  // hour違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017, 10, 1, 3, 45, 56));  // day違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2017,  9, 2, 3, 45, 56));  // month違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) != Calendar(2016, 10, 2, 3, 45, 56));  // year違う
    }

    {
        // >=
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017, 10, 2, 3, 45, 56)); // 同じ
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017, 10, 2, 3, 45, 55)); // second違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017, 10, 2, 3, 44, 56)); // minute違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017, 10, 2, 2, 45, 56)); // hour違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017, 10, 1, 3, 45, 56)); // day違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2017,  9, 2, 3, 45, 56)); // month違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >= Calendar(2016, 10, 2, 3, 45, 56)); // year違う

        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >=  Calendar(2017, 10, 2, 3, 47, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >=  Calendar(2017, 10, 2, 4, 44, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >=  Calendar(2017, 10, 3, 2, 45, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >=  Calendar(2017, 11, 2, 3, 45, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >=  Calendar(2017, 11, 1, 3, 45, 56));
    }

    {
        // >
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 3, 45, 56)); // 同じ
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 3, 45, 55));  // second違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 3, 44, 56));  // minute違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 2, 45, 56));  // hour違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 1, 3, 45, 56));  // day違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017,  9, 2, 3, 45, 56));  // month違う
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2016, 10, 2, 3, 45, 56));  // year違う

        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 3, 47, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 2, 4, 44, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 10, 3, 2, 45, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 11, 2, 3, 45, 55));
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) >  Calendar(2017, 11, 1, 3, 45, 56));
    }

    {
        // <=
        EXPECT_TRUE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017, 10, 2, 3, 45, 56));  // 同じ
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017, 10, 2, 3, 45, 55)); // second違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017, 10, 2, 3, 44, 56)); // minute違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017, 10, 2, 2, 45, 56)); // hour違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017, 10, 1, 3, 45, 56)); // day違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2017,  9, 2, 3, 45, 56)); // month違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) <= Calendar(2016, 10, 2, 3, 45, 56)); // year違う

        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 46, 56) <= Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 10, 2, 4, 45, 56) <= Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 10, 3, 3, 45, 56) <= Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 11, 2, 3, 45, 56) <= Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 11, 1, 3, 45, 57) <= Calendar(2017, 10, 2, 3, 45, 57));
    }

    {
        // <
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017, 10, 2, 3, 45, 56)); // 同じ
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017, 10, 2, 3, 45, 55)); // second違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017, 10, 2, 3, 44, 56)); // minute違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017, 10, 2, 2, 45, 56)); // hour違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017, 10, 1, 3, 45, 56)); // day違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2017,  9, 2, 3, 45, 56)); // month違う
        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 45, 56) < Calendar(2016, 10, 2, 3, 45, 56)); // year違う

        EXPECT_FALSE( Calendar(2017, 10, 2, 3, 46, 56) < Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 10, 2, 4, 45, 56) < Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 10, 3, 3, 45, 56) < Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 11, 2, 3, 45, 56) < Calendar(2017, 10, 2, 3, 45, 57));
        EXPECT_FALSE( Calendar(2017, 11, 1, 3, 45, 57) < Calendar(2017, 10, 2, 3, 45, 57));
    }
} // NOLINT(impl/function_size)
