﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/timesrv/detail/fs/timesrv_TimeZoneBinaryManager.h>

using namespace nn::timesrv::detail::fs;

TEST(TimeZoneBinary, Mount)
{
    NNT_EXPECT_RESULT_SUCCESS( TimeZoneBinaryManager::MountFs() );
}

TEST(TimeZoneBinary, GetLocationNameCount)
{
    // 何個が正解か不明なので 0 より大きいことだけチェック
    EXPECT_GE(TimeZoneBinaryManager::GetLocationNameCount(), 0);
}

TEST(TimeZoneBinary, IsValidLocationName)
{
    {
        const nn::time::LocationName name = {"UTC"};
        EXPECT_TRUE( TimeZoneBinaryManager::IsValidLocationName(name) );
    }
    {
        const nn::time::LocationName name = {"Asia/Tokyo"};
        EXPECT_TRUE( TimeZoneBinaryManager::IsValidLocationName(name) );
    }

    {
        const nn::time::LocationName name = {"Asia/ Tokyo"};
        EXPECT_FALSE( TimeZoneBinaryManager::IsValidLocationName(name) );
    }
    {
        const nn::time::LocationName name = {""};
        EXPECT_FALSE( TimeZoneBinaryManager::IsValidLocationName(name) );
    }
}

TEST(TimeZoneBinary, ReadTimeZoneBinary)
{
    char* pBinary;
    size_t binarySize;
    const nn::time::LocationName name = {"Asia/Tokyo"};
    NNT_EXPECT_RESULT_SUCCESS(TimeZoneBinaryManager::ReadTimeZoneBinary(&pBinary, &binarySize, name));
}

TEST(TimeZoneBinary, LoadLocationNameList)
{
    static nn::time::LocationName locationNameBuffer[1024];
    int32_t outCount;

    NNT_ASSERT_RESULT_SUCCESS(TimeZoneBinaryManager::LoadLocationNameList(
        &outCount,
        locationNameBuffer,
        1024,
        0));

    // 失敗時、locationNameBuffer の数が足りない可能性がある.1024 は将来においても十分足りると思われる数.
    EXPECT_EQ(TimeZoneBinaryManager::GetLocationNameCount(), outCount);

    for(int32_t i = 0 ; i < outCount ; ++i)
    {
        ASSERT_TRUE(TimeZoneBinaryManager::IsValidLocationName(locationNameBuffer[i])) << locationNameBuffer[i]._value;

        char* pBinary;
        size_t binarySize;
        NNT_ASSERT_RESULT_SUCCESS(TimeZoneBinaryManager::ReadTimeZoneBinary(&pBinary, &binarySize, locationNameBuffer[i])) << locationNameBuffer[i]._value;
    }
}

TEST(TimeZoneBinary, RetrieveLocationNameList)
{

    class TestTimeZoneBinaryManager : public TimeZoneBinaryManager
    {
    public:
        static nn::Result RetrieveLocationNameList(
            int32_t* pOutCount,
            nn::time::LocationName* pOutLocationNameList,
            size_t count,
            int32_t offset,
            const char* pListBuffer,
            size_t listBufferSize) NN_NOEXCEPT
            {
                return TimeZoneBinaryManager::RetrieveLocationNameList(
                    pOutCount,
                    pOutLocationNameList,
                    count,
                    offset,
                    pListBuffer,
                    listBufferSize);
            }
    };

    {
        int32_t outCount;
        nn::time::LocationName outLocationName[2];
        const char ListBuffer[] = "Asia/Tokyo\r\n12345678901234567890123456789012345\r\n"; // 35文字はぎりぎりOK
        NNT_EXPECT_RESULT_SUCCESS(TestTimeZoneBinaryManager::RetrieveLocationNameList(
            &outCount,
            outLocationName,
            2,
            0,
            ListBuffer,
            sizeof(ListBuffer)));

        EXPECT_STREQ(outLocationName[0]._value, "Asia/Tokyo");
        EXPECT_STREQ(outLocationName[1]._value, "12345678901234567890123456789012345");
    }

    {
        int32_t outCount;
        nn::time::LocationName outLocationName[2];
        const char ListBuffer[] = "Asia/Tokyo\r\n123456789012345678901234567890123456\r\n"; // 36文字は大きすぎてエラー
        NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOutOfMemory, TestTimeZoneBinaryManager::RetrieveLocationNameList(
            &outCount,
            outLocationName,
            2,
            0,
            ListBuffer,
            sizeof(ListBuffer)));
    }
}

