﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardUserSystemClockCore.h>

namespace
{
    class TestSteadyClock : public nn::timesrv::detail::core::SteadyClockCore
    {
    public:
        TestSteadyClock() NN_NOEXCEPT
        {
            m_TestClockSourceId.FromString("28ea74e0-febe-4627-a8e6-c7a9a8a85ae4"); // 適当
        }

    private:
        nn::util::Uuid m_TestClockSourceId;
        nn::TimeSpan m_InternalOffset;

        nn::Result GetCurrentTimePointImpl(nn::time::SteadyClockTimePoint* pSteadyClockTimePoint) const NN_NOEXCEPT NN_OVERRIDE
        {
            pSteadyClockTimePoint->value = 987654321LL; // 単調増加クロックは増えないものとしてテスト
            pSteadyClockTimePoint->sourceId = m_TestClockSourceId;
            NN_RESULT_SUCCESS;
        }
        nn::TimeSpan GetTestOffsetImpl() const NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::TimeSpan(0);
        }
        nn::TimeSpan GetInternalOffsetImpl() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_InternalOffset;
        }
        void SetInternalOffsetImpl(const nn::TimeSpan& internalOffset) NN_NOEXCEPT NN_OVERRIDE
        {
            m_InternalOffset = internalOffset;
        }
    };

    class TestSystemClock : public nn::timesrv::detail::core::SystemClockCore
    {
    public:
        explicit TestSystemClock (nn::timesrv::detail::core::SteadyClockCore* pSteadyClockCore) NN_NOEXCEPT:
            nn::timesrv::detail::core::SystemClockCore(pSteadyClockCore)
        {
        }

        // 時計の無効化
        void Invalidate() NN_NOEXCEPT
        {
            m_SystemClockContext.timeStamp.sourceId = nn::util::InvalidUuid;
        }
    };
}

TEST(StandardUserSystemClockCore, Basic1)
{
    TestSteadyClock steadyClock;
    TestSystemClock clock1(&steadyClock);
    TestSystemClock clock2(&steadyClock);
    nn::timesrv::detail::core::StandardUserSystemClockCore userClock(&clock1, &clock2);

    // userClock 自動補正off、 clock1 補正済、clock2 未補正
    {
        NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(false));

        nn::time::PosixTime current1 = {123456};
        NNT_ASSERT_RESULT_SUCCESS(clock1.SetCurrentTime(current1));
        clock2.Invalidate();
    }

    nn::time::PosixTime current;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(123456, current.value);

    // userClock 自動補正 on 後でも clock1 が取得できるかどうか
    NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(true));
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(123456, current.value);
}

TEST(StandardUserSystemClockCore, Basic2)
{
    TestSteadyClock steadyClock;
    TestSystemClock clock1(&steadyClock);
    TestSystemClock clock2(&steadyClock);
    nn::timesrv::detail::core::StandardUserSystemClockCore userClock(&clock1, &clock2);

    // userClock 自動補正off、 clock1 補正済、clock2 補正済
    {
        NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(false));

        nn::time::PosixTime current1 = {123456};
        NNT_ASSERT_RESULT_SUCCESS(clock1.SetCurrentTime(current1));

        nn::time::PosixTime current2 = {654321};
        NNT_ASSERT_RESULT_SUCCESS(clock2.SetCurrentTime(current2));
    }

    nn::time::PosixTime current;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(123456, current.value);

    // userClock 自動補正 on 後に clock2 が取得できるかどうか
    NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(true));
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(654321, current.value);
}

TEST(StandardUserSystemClockCore, Basic3)
{
    TestSteadyClock steadyClock;
    TestSystemClock clock1(&steadyClock);
    TestSystemClock clock2(&steadyClock);
    nn::timesrv::detail::core::StandardUserSystemClockCore userClock(&clock1, &clock2);

    // userClock 自動補正on、 clock1 補正済、clock2 未補正
    {
        NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(true));

        nn::time::PosixTime current1 = {123456};
        NNT_ASSERT_RESULT_SUCCESS(clock1.SetCurrentTime(current1));

        clock2.Invalidate();
    }

    nn::time::PosixTime current;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(123456, current.value);

    // clock2 補正したら、userClock が clock2 の時刻を返すかどうか
    nn::time::PosixTime current2 = {654321};
    NNT_ASSERT_RESULT_SUCCESS(clock2.SetCurrentTime(current2));
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(654321, current.value);

    // 自動補正 off 後も clock2 の時刻を返すかどうか
    NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(false));
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(654321, current.value);
}

TEST(StandardUserSystemClockCore, Basic4)
{
    TestSteadyClock steadyClock;
    TestSystemClock clock1(&steadyClock);
    TestSystemClock clock2(&steadyClock);
    nn::timesrv::detail::core::StandardUserSystemClockCore userClock(&clock1, &clock2);

    // userClock 自動補正on、 clock1 補正済、clock2 補正済
    {
        NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(true));

        nn::time::PosixTime current1 = {123456};
        NNT_ASSERT_RESULT_SUCCESS(clock1.SetCurrentTime(current1));

        nn::time::PosixTime current2 = {654321};
        NNT_ASSERT_RESULT_SUCCESS(clock2.SetCurrentTime(current2));
    }

    nn::time::PosixTime current;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(654321, current.value);

    // 自動補正 off 後も同じ時刻を返すかどうか
    NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(false));
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(654321, current.value);
}

TEST(StandardUserSystemClockCore, InternalOffset)
{
    TestSteadyClock steadyClock;
    TestSystemClock clock1(&steadyClock);
    TestSystemClock clock2(&steadyClock);
    nn::timesrv::detail::core::StandardUserSystemClockCore userClock(&clock1, &clock2);

    // userClock 自動補正off、 clock1 補正済、clock2 補正済
    {
        NNT_ASSERT_RESULT_SUCCESS(userClock.SetAutomaticCorrectionEnabled(false));

        nn::time::PosixTime current1 = {123456};
        NNT_ASSERT_RESULT_SUCCESS(clock1.SetCurrentTime(current1));

        nn::time::PosixTime current2 = {654321};
        NNT_ASSERT_RESULT_SUCCESS(clock2.SetCurrentTime(current2));
    }

    nn::time::SystemClockContext context1;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetSystemClockContext(&context1));

    // SteadyClock の InternalOffset 入れる
    steadyClock.SetInternalOffset(nn::TimeSpan::FromSeconds(100));

    // 時刻はその分ずれる
    nn::time::PosixTime current;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetCurrentTime(&current));
    EXPECT_EQ(123456 + 100, current.value);

    // SteadyClock の InternalOffset のセット前後で SystemClockContext は変化しない
    nn::time::SystemClockContext context2;
    NNT_ASSERT_RESULT_SUCCESS(userClock.GetSystemClockContext(&context2));
    EXPECT_EQ(context1, context2);
}

