﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardLocalSystemClockCore.h>

namespace
{
    // SystemClockContext 初期値
    const nn::time::SystemClockContext InitialSystemClockContext =
    {
        0LL,
        {
            0LL,
            nn::util::InvalidUuid
        }
    };
}

// 時刻セット後に時刻取得して値が近いかテストするサブ関数.
// IPCや設定の永続化で誤差が生じるため一定範囲の誤差を許容.
// currentValue : 直前に取得した時刻の値
// setValue : 時計にセットした時刻の値
#define NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(currentValue, setValue) \
    do { \
        const int64_t allowedRange = 5; \
        EXPECT_LE((setValue), (currentValue)); \
        auto diff = (currentValue) - (setValue); \
        EXPECT_LE(diff, allowedRange) << "diff:" << diff << " currentValue:" << (currentValue) << " setValue:" << (setValue); \
    } while(NN_STATIC_CONDITION(0))

TEST(StandardLocalSystemClockCoreSuite, InitialStateTest)
{
    // Win環境においては StandardLocalSystemClockCore::GetInitialPosixTime() に PC 時計が使用されており、
    // 時間経過とともに進みテスト失敗の原因となるので、StandardLocalSystemClockCore のインスタンス生成前にキャッシュしておく
    const nn::time::PosixTime InitialPosixTime =
        nn::timesrv::detail::core::StandardLocalSystemClockCore::GetInitialPosixTime();

    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    localSystemClock.Initialize(InitialSystemClockContext);

    nn::time::SteadyClockTimePoint tp;
    NNT_ASSERT_RESULT_SUCCESS(steadyClock.GetCurrentTimePoint(&tp));

    // GetSystemClockContext 経由で取得した SourceId 確認
    nn::time::SystemClockContext systemClockContext;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetSystemClockContext(&systemClockContext) );
    EXPECT_EQ(systemClockContext.timeStamp.sourceId, tp.sourceId);
    EXPECT_NE(systemClockContext.timeStamp.sourceId, nn::util::InvalidUuid);

    // 初期状態で補正がされていないと、決まった値からのカウントを開始しているはず
    nn::time::PosixTime posixTime;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetCurrentTime(&posixTime) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        posixTime.value,
        InitialPosixTime.value);
}

// StandardLocalSystemClock, StandardSteadyClock ともに同じソース ID
TEST(StandardLocalSystemClockCoreSuite, NormalStateTest)
{
    nn::time::SourceId availableSourceId;
    availableSourceId.FromString("12345678-1234-1234-1234-123456789012"); // 適当に有効な UUID を作る

    const nn::time::SystemClockContext availableSystemClockContext =
    {
        0LL,
        {
            0LL,
            availableSourceId
        }
    };

    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, availableSourceId);
    localSystemClock.Initialize(availableSystemClockContext);

    nn::time::SteadyClockTimePoint tp;
    NNT_ASSERT_RESULT_SUCCESS(steadyClock.GetCurrentTimePoint(&tp));
    EXPECT_EQ(availableSourceId, tp.sourceId);

    // GetSystemClockContext 経由で取得した SourceId 確認
    nn::time::SystemClockContext systemClockContext;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetSystemClockContext(&systemClockContext) );
    EXPECT_EQ(availableSourceId, systemClockContext.timeStamp.sourceId);
}

// localSystemClock の SystemClockContext は有効な値だが、SteadyClock の SourceId は InvalidUuid だった場合のテスト.
// 永続化データが壊れていた場合や読み出し失敗、 RTC リセットの発生で SteadyClock の SourceId を作り直すケースに相当.
TEST(StandardLocalSystemClockCoreSuite, SteadyClockSourceIdError)
{
    // Win環境においては StandardLocalSystemClockCore::GetInitialPosixTime() に PC 時計が使用されており、
    // 時間経過とともに進みテスト失敗の原因となるので、StandardLocalSystemClockCore のインスタンス生成前にキャッシュしておく
    const nn::time::PosixTime InitialPosixTime =
        nn::timesrv::detail::core::StandardLocalSystemClockCore::GetInitialPosixTime();

    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    // StandardSteadyClockCore の SourceId だけ初期値へ
    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);

    // StandardLocalSystemClockCore の SystemClockContext は有効な値に
    nn::time::SourceId availableSourceId;
    availableSourceId.FromString("12345678-1234-1234-1234-123456789012"); // 適当に有効な UUID を作る
    const nn::time::SystemClockContext availableSystemClockContext =
    {
        0LL,
        {
            0LL,
            availableSourceId
        }
    };
    localSystemClock.Initialize(availableSystemClockContext);

    nn::time::SteadyClockTimePoint tp;
    NNT_ASSERT_RESULT_SUCCESS(steadyClock.GetCurrentTimePoint(&tp));

    // GetSystemClockContext 経由で取得した SourceId 確認
    nn::time::SystemClockContext systemClockContext;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetSystemClockContext(&systemClockContext) );
    EXPECT_EQ(systemClockContext.timeStamp.sourceId, tp.sourceId);
    EXPECT_NE(systemClockContext.timeStamp.sourceId, nn::util::InvalidUuid);

    // 初期状態で補正がされていないと、決まった値からのカウントを開始しているはず
    nn::time::PosixTime posixTime;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetCurrentTime(&posixTime) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        posixTime.value,
        InitialPosixTime.value);
}

TEST(StandardLocalSystemClockCoreSuite, CalculatePosixTime)
{
    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    localSystemClock.Initialize(InitialSystemClockContext);

    nn::time::SteadyClockTimePoint tp;
    NNT_ASSERT_RESULT_SUCCESS(steadyClock.GetCurrentTimePoint(&tp));

    nn::time::SteadyClockTimePoint steadyClockTimePoint;
    NNT_EXPECT_RESULT_SUCCESS( steadyClock.GetCurrentTimePoint(&steadyClockTimePoint) );

    nn::time::SystemClockContext systemClockContext;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetSystemClockContext(&systemClockContext) );

    nn::time::PosixTime posixTime;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.CalculatePosixTime(&posixTime, steadyClockTimePoint, systemClockContext) );
    EXPECT_EQ(steadyClockTimePoint.value + systemClockContext.offset, posixTime.value);


    // SourceId が違った場合
    steadyClockTimePoint.sourceId.data[0]++; // 適当に変える
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOffsetInvalid, localSystemClock.CalculatePosixTime(&posixTime, steadyClockTimePoint, systemClockContext) );
}

TEST(StandardLocalSystemClockCoreSuite, SetCurrentTime)
{
    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    localSystemClock.Initialize(InitialSystemClockContext);

    nn::time::SteadyClockTimePoint tp;
    NNT_ASSERT_RESULT_SUCCESS(steadyClock.GetCurrentTimePoint(&tp));

    // デフォルトとは異なる値を適当にセットしてみる
    const nn::time::PosixTime SetPosixTime = nn::timesrv::detail::core::StandardLocalSystemClockCore::GetInitialPosixTime() + nn::TimeSpan::FromSeconds(123456);
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.SetCurrentTime(SetPosixTime) );

    nn::time::PosixTime posixTime;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetCurrentTime(&posixTime) );

    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        posixTime.value,
        SetPosixTime.value);

    // steadyClock にテストオフセットを入れてみる
    steadyClock.SetTestOffset(nn::TimeSpan::FromSeconds(10));

    // 10 秒ずれるはず
    nn::time::PosixTime posixTimeWithTestOffset;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetCurrentTime(&posixTimeWithTestOffset) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        posixTimeWithTestOffset.value,
        SetPosixTime.value + 10);

    // 同じ値をセットしてみる
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.SetCurrentTime(SetPosixTime) );
    // テストオフセット分ずていない PosixTime が返るはず
    nn::time::PosixTime current;
    NNT_EXPECT_RESULT_SUCCESS( localSystemClock.GetCurrentTime(&current) );
    NNT_TIME_TEST_NEAR_CURRENT_POSIX_TIME_VALUE(
        current.value,
        SetPosixTime.value);
}

TEST(StandardLocalSystemClockCoreSuite, Callback)
{
    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    localSystemClock.Initialize(InitialSystemClockContext);

    class Callback : public nn::timesrv::detail::core::SystemClockContextUpdateCallback
    {
    public:
        Callback() NN_NOEXCEPT:
            m_Count(0),
            m_LastContext(InitialSystemClockContext)
        {}

        virtual nn::Result Set(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT NN_OVERRIDE
        {
            m_LastContext = systemClockContext;
            m_Count++;
            NN_RESULT_SUCCESS;
        }

        int m_Count;
        nn::time::SystemClockContext m_LastContext;
    };

    Callback callback;
    localSystemClock.SetSystemClockContextUpdateCallback(&callback);

    nn::time::SystemClockContext context;
    NNT_ASSERT_RESULT_SUCCESS( localSystemClock.GetSystemClockContext(&context) );
    EXPECT_EQ(0, callback.m_Count);

    context.offset = 10;
    NNT_ASSERT_RESULT_SUCCESS( localSystemClock.SetSystemClockContext(context) );
    EXPECT_EQ(1, callback.m_Count);
    EXPECT_EQ(context, callback.m_LastContext);

    context.offset = 20;
    NNT_ASSERT_RESULT_SUCCESS( localSystemClock.SetSystemClockContext(context) );
    EXPECT_EQ(2, callback.m_Count);
    EXPECT_EQ(context, callback.m_LastContext);
}

TEST(StandardLocalSystemClockCoreSuite, InternalOffset)
{
    nn::timesrv::detail::core::StandardSteadyClockCore steadyClock;
    nn::timesrv::detail::core::StandardLocalSystemClockCore localSystemClock(&steadyClock);

    steadyClock.SetupSourceId(nullptr, nn::util::InvalidUuid);
    localSystemClock.Initialize(InitialSystemClockContext);

    nn::time::SystemClockContext context1;
    NNT_ASSERT_RESULT_SUCCESS(localSystemClock.GetSystemClockContext(&context1));

    nn::time::PosixTime posixTime;
    NNT_ASSERT_RESULT_SUCCESS(localSystemClock.GetCurrentTime(&posixTime));

    // SteadyClock の InternalOffset 入れる
    steadyClock.SetInternalOffset(nn::TimeSpan::FromSeconds(100));

    // 時刻はその分ずれる
    nn::time::PosixTime posixTimeWithOffset;
    NNT_ASSERT_RESULT_SUCCESS(localSystemClock.GetCurrentTime(&posixTimeWithOffset));
    // 実行時間の誤差もあるので1秒以内の誤差でチェック
    EXPECT_NEAR(
        static_cast<double>(posixTime.value + 100),
        static_cast<double>(posixTimeWithOffset.value), 1);

    // SteadyClock の InternalOffset のセット前後で SystemClockContext は変化しない
    nn::time::SystemClockContext context2;
    NNT_ASSERT_RESULT_SUCCESS(localSystemClock.GetSystemClockContext(&context2));
    EXPECT_EQ(context1, context2);
}
