﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_VirtualTemperature.h>
#include <nn/tc/impl/tc_PowerMode.h>

#include <nnt/nnt_Argument.h>

#include <nnt/gtest/gtest.h>

namespace nnt { namespace tc {

namespace {

struct ThermalCoordinatorTestSettingEntry
{
    nn::tc::impl::PowerMode powerMode;
    nn::tc::OperatingMode operatingMode;

    int temperatureInternalStart;
    int temperatureInternalEnd;
    int temperatureExternalStart;
    int temperatureExternalEnd;
    int steps;

    nn::fan::RotationSpeedLevel speedLevelMin;
    nn::fan::RotationSpeedLevel speedLevelMax;
    nn::fan::RotationSpeedLevel speedLevelGradientMax;
};

struct ThermalCoordinatorTestSetting
{
    const ThermalCoordinatorTestSettingEntry* pEntry;
    int size;
};

#define PERCENTAGE_TO_SPEED_LEVEL(percentage) ((percentage) * 255 / 100)

const nn::fan::RotationSpeedLevel MinSpeedLevelHandheld = PERCENTAGE_TO_SPEED_LEVEL(0);
const nn::fan::RotationSpeedLevel MaxSpeedLevelHandheld = PERCENTAGE_TO_SPEED_LEVEL(100);
const nn::fan::RotationSpeedLevel MinSpeedLevelConsole  = PERCENTAGE_TO_SPEED_LEVEL(0);
const nn::fan::RotationSpeedLevel MaxSpeedLevelConsole  = PERCENTAGE_TO_SPEED_LEVEL(100);

const ThermalCoordinatorTestSettingEntry TestEntriesHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 35, 76, 35, 78, 60, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 76, 35, 78, 35, 60, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 6},
};

const ThermalCoordinatorTestSettingEntry TestEntriesConsoleToHandheld[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  35, 76, 35, 78, 60, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 76, 35, 78, 35, 60, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console,  35, 76, 35, 78, 60, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},
};

const ThermalCoordinatorTestSettingEntry TestEntriesConsole[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, 35, 76, 35, 78, 60, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, 76, 35, 78, 35, 60, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},
};

const ThermalCoordinatorTestSettingEntry TestEntriesHandheldToConsole[] =
{
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 35, 76, 35, 78, 60, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 6},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Console, 76, 35, 78, 35, 60, MinSpeedLevelConsole, MaxSpeedLevelConsole, 14},
    { nn::tc::impl::PowerMode_FullAwake, nn::tc::OperatingMode_Handheld, 35, 76, 35, 78, 60, MinSpeedLevelHandheld, MaxSpeedLevelHandheld, 6},
};

const ThermalCoordinatorTestSetting ThermalCoordinatorTestSettings[] =
{
    // FullAwake
    { &TestEntriesHandheld[0],          sizeof(TestEntriesHandheld)          / sizeof(TestEntriesHandheld[0])          },
    { &TestEntriesConsoleToHandheld[0], sizeof(TestEntriesConsoleToHandheld) / sizeof(TestEntriesConsoleToHandheld[0]) },
    { &TestEntriesConsole[0],           sizeof(TestEntriesConsole)           / sizeof(TestEntriesConsole[0])           },
    { &TestEntriesHandheldToConsole[0], sizeof(TestEntriesHandheldToConsole) / sizeof(TestEntriesHandheldToConsole[0]) },
};

class ThermalEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        nn::fan::Initialize();
        nn::tc::Initialize();
    }

    virtual void TearDown() override
    {
        nn::tc::Finalize();
        nn::fan::Finalize();
    }
};

class ThermalCoordinatorTest : public ::testing::TestWithParam<ThermalCoordinatorTestSetting>{};

INSTANTIATE_TEST_CASE_P(ThermalCoordinatorTestName, ThermalCoordinatorTest, ::testing::ValuesIn(ThermalCoordinatorTestSettings));

void DispatchEntry(const ThermalCoordinatorTestSettingEntry* pEntry) NN_NOEXCEPT
{
    const nn::fan::RotationSpeedLevel LowestSpeedLevelThreshold = 51;

    nn::tc::SetOperatingMode(pEntry->operatingMode);

    nn::fan::Controller controller;
    nn::fan::OpenController(&controller, nn::fan::FanName_Cpu);

    nn::fan::RotationSpeedLevel preSpeedLevel = nn::fan::GetRotationSpeedLevel(&controller);

    for ( int step = 0; step < pEntry->steps; step++ )
    {
        int internalTemperature;
        if ( pEntry->temperatureInternalEnd == pEntry->temperatureInternalStart )
        {
            internalTemperature = pEntry->temperatureInternalStart;
        }
        else
        {
            internalTemperature = pEntry->temperatureInternalStart + (step * (pEntry->temperatureInternalEnd - pEntry->temperatureInternalStart)) / pEntry->steps;
        }

        int externalTemperature;
        if ( pEntry->temperatureExternalEnd == pEntry->temperatureExternalStart )
        {
            externalTemperature = pEntry->temperatureExternalStart;
        }
        else
        {
            externalTemperature = pEntry->temperatureExternalStart + (step * (pEntry->temperatureExternalEnd - pEntry->temperatureExternalStart)) / pEntry->steps;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorInternal,
            static_cast<nn::tc::TemperatureMilliC>(internalTemperature * 1000)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorExternal,
            static_cast<nn::tc::TemperatureMilliC>(externalTemperature * 1000)));

        // 結果の反映を待ちます。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000LL));

        nn::fan::RotationSpeedLevel postSpeedLevel = nn::fan::GetRotationSpeedLevel(&controller);

        NN_LOG("[PCB: %d C, SOC: %d C] RotationSpeedLevel:%d\n", internalTemperature, externalTemperature, postSpeedLevel);

        EXPECT_GE(postSpeedLevel, pEntry->speedLevelMin);
        EXPECT_LE(postSpeedLevel, pEntry->speedLevelMax);

        postSpeedLevel = std::max(postSpeedLevel, LowestSpeedLevelThreshold);

        if ( step > 0 )
        {
            EXPECT_GE(std::abs(postSpeedLevel - preSpeedLevel), 0);

            // TC による更新が 2 回入る可能性を考慮して傾きの 2 倍までの変化を許可する。
            EXPECT_LE(std::abs(postSpeedLevel - preSpeedLevel), pEntry->speedLevelGradientMax * 2);
        }

        preSpeedLevel = postSpeedLevel;
    }

    nn::fan::CloseController(&controller);
}

} // namespace

//
// 携帯および据置モードでの温度上昇および温度低下を仮想温度を使用してシミュレートして、実際にファンが期待通りに回転することを確認します。
//
// OperatingMode を上書きして復旧しません。
// 実行後はターゲットのリセットをする必要があります。
//
TEST_P(ThermalCoordinatorTest, TestEachTransition)
{
    nn::tc::SetOperatingMode(GetParam().pEntry->operatingMode);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorInternal, GetParam().pEntry->temperatureInternalStart * 1000));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorExternal, GetParam().pEntry->temperatureExternalStart * 1000));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::EnableVirtualTemperature(nn::tc::Location_ThermalSensorInternal));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::EnableVirtualTemperature(nn::tc::Location_ThermalSensorExternal));

    for ( int index = 0; index < GetParam().size; index++ )
    {
        DispatchEntry(GetParam().pEntry + index);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::DisableVirtualTemperature(nn::tc::Location_ThermalSensorInternal));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::DisableVirtualTemperature(nn::tc::Location_ThermalSensorExternal));
}

}} // namespace nnt::tc

extern "C" void nninitStartup()
{
    ::testing::AddGlobalTestEnvironment(new nnt::tc::ThermalEnvironment);
}
