﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nnt/gtest/gtest.h>

#include <nn/fan/fan.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_FanControl.h>

namespace {

bool g_IsFanControlEnabled = false;

class FanTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        g_IsFanControlEnabled = false;
        nn::tc::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::IsFanControlEnabled(&g_IsFanControlEnabled));
        if ( g_IsFanControlEnabled )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::DisableFanControl());
        }
    }

    static void TearDownTestCase()
    {
        if ( g_IsFanControlEnabled )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tc::EnableFanControl());
        }
        nn::tc::Finalize();
    }
};

} // namespace

TEST_F( FanTest, SetSpeedLevelByDiff )
{
    nn::fan::Initialize();

    nn::fan::Controller fanController;
    nn::fan::OpenController(&fanController, nn::fan::FanName_Cpu);

    // 初期値退避
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&fanController);

    int start;
    int diff;

    // 加算テスト
    start = nn::fan::RotationSpeedLevelMax - 1;
    diff = 1;
    nn::fan::SetRotationSpeedLevel(&fanController, start);
    EXPECT_EQ(start, nn::fan::GetRotationSpeedLevel(&fanController));
    EXPECT_EQ(start + diff, nn::fan::SetRotationSpeedLevelByDiff(&fanController, diff));
    EXPECT_EQ(start + diff, nn::fan::GetRotationSpeedLevel(&fanController));

    // 減算テスト
    start = 1;
    diff = -1;
    nn::fan::SetRotationSpeedLevel(&fanController, start);
    EXPECT_EQ(start, nn::fan::GetRotationSpeedLevel(&fanController));
    EXPECT_EQ(start + diff, nn::fan::SetRotationSpeedLevelByDiff(&fanController, diff));
    EXPECT_EQ(start + diff, nn::fan::GetRotationSpeedLevel(&fanController));

    // nn::fan::RotationSpeedLevelMax でクランプされるテスト
    start = 1;
    diff = nn::fan::RotationSpeedLevelMax;
    nn::fan::SetRotationSpeedLevel(&fanController, start);
    EXPECT_EQ(start, nn::fan::GetRotationSpeedLevel(&fanController));
    EXPECT_EQ(nn::fan::RotationSpeedLevelMax, nn::fan::SetRotationSpeedLevelByDiff(&fanController, diff));
    EXPECT_EQ(nn::fan::RotationSpeedLevelMax, nn::fan::GetRotationSpeedLevel(&fanController));

    // 0 でクランプされるテスト
    start = nn::fan::RotationSpeedLevelMax - 1;
    diff = -nn::fan::RotationSpeedLevelMax;
    nn::fan::SetRotationSpeedLevel(&fanController, start);
    EXPECT_EQ(start, nn::fan::GetRotationSpeedLevel(&fanController));
    EXPECT_EQ(0, nn::fan::SetRotationSpeedLevelByDiff(&fanController, diff));
    EXPECT_EQ(0, nn::fan::GetRotationSpeedLevel(&fanController));

    // 初期値に戻す
    nn::fan::SetRotationSpeedLevel(&fanController, initialLevel);
    EXPECT_EQ(initialLevel, nn::fan::GetRotationSpeedLevel(&fanController));

    nn::fan::CloseController(&fanController);

    nn::fan::Finalize();
}

TEST_F( FanTest, SetSpeedLevel )
{
    nn::fan::Initialize();

    nn::fan::Controller fanController;
    nn::fan::OpenController(&fanController, nn::fan::FanName_Cpu);

    // 初期値退避
    auto initialLevel = nn::fan::GetRotationSpeedLevel(&fanController);

    nn::fan::RotationSpeedLevel target;
    target = 0;
    nn::fan::SetRotationSpeedLevel(&fanController, target);
    EXPECT_EQ(target, nn::fan::GetRotationSpeedLevel(&fanController));

    target = nn::fan::RotationSpeedLevelMax;
    nn::fan::SetRotationSpeedLevel(&fanController, target);
    EXPECT_EQ(target, nn::fan::GetRotationSpeedLevel(&fanController));

    // 初期値に戻す
    nn::fan::SetRotationSpeedLevel(&fanController, initialLevel);
    EXPECT_EQ(initialLevel, nn::fan::GetRotationSpeedLevel(&fanController));

    nn::fan::CloseController(&fanController);

    nn::fan::Finalize();
}

TEST_F( FanTest, SetRate )
{
    nn::fan::Initialize();

    nn::fan::Controller fanController;
    nn::fan::OpenController(&fanController, nn::fan::FanName_Cpu);

    // 初期値退避
    auto initialRate = nn::fan::GetRotationRate(&fanController);

    int target;
    target = 0;
    nn::fan::SetRotationRate(&fanController, target);
    EXPECT_EQ(target, nn::fan::GetRotationRate(&fanController));

    target = 100;
    nn::fan::SetRotationRate(&fanController, target);
    EXPECT_EQ(target, nn::fan::GetRotationRate(&fanController));

    // 初期値に戻す
    nn::fan::SetRotationRate(&fanController, initialRate);
    EXPECT_EQ(initialRate, nn::fan::GetRotationRate(&fanController));

    nn::fan::CloseController(&fanController);

    nn::fan::Finalize();
}

TEST_F( FanTest, EnableFan )
{
    nn::fan::Initialize();

    nn::fan::Controller fanController;
    nn::fan::OpenController(&fanController, nn::fan::FanName_Cpu);

    int target;
    target = nn::fan::GetRotationRate(&fanController);

    // DisableFan は FAN サービスの保持する回転数に影響しない。
    nn::fan::DisableFan(&fanController);
    EXPECT_EQ(target, nn::fan::GetRotationRate(&fanController));

    // EnableFan は FAN サービスの保持する回転数に影響しない。
    nn::fan::EnableFan(&fanController);
    EXPECT_EQ(target, nn::fan::GetRotationRate(&fanController));

    nn::fan::CloseController(&fanController);

    nn::fan::Finalize();
}
