﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#define _POSIX_

#include <Abuse.h>
#include <TaskList.h>
#include <Platform.h>
#include <Log.h>
#include <FileUtility.h>
#include <Tasks/UnitTestTask.h>
#include <nnt/nnt_Argument.h>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <glv_resources.h>

#include <nv/nv_MemoryManagement.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

// For declearation of 'glslcSetAllocator'
#include <nvnTool/nvnTool_GlslcInterface.h>

#include "UI/abuse_ui_RootContext.h"
#include "UI/abuse_ui_Notifier.h"

static const size_t AppletHeapSize = 256 * 1024 * 1024;
static const size_t AppletAllocatableSize = 256 * 1024 * 1024;
static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

// ui specific
nnt::abuse::RootContext *uiContext = NULL;

// thread for the UI
static const size_t STACK_SIZE = 4096;
nn::os::ThreadType threadForUse;
NN_OS_ALIGNAS_THREAD_STACK char threadStack1[STACK_SIZE];


void* Allocate(size_t size) NN_NOEXCEPT
{
    return std::malloc(size);
}

void Deallocate(void* addr, size_t) NN_NOEXCEPT
{
    std::free(addr);
}

//------------------------------------------------------------------------------
// Memory allocation functions for graphics systems
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

//------------------------------------------------------------------------------
// GLSLC memory allocation functions
//------------------------------------------------------------------------------
static void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void GlslcFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

void InitializePeripherals() NN_NOEXCEPT
{
    // this memory allocation will be used from the nvn graphics systems at runtime.
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);

    nv::InitializeGraphics(Allocate(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);

    // this memory allocation interface will be used when compiling of shader code at runtime.
    glslcSetAllocator(GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL);
}

void FinalizePeripherals() NN_NOEXCEPT
{
}

const glv::InitialConfiguration    LocalGfxConfiguration = glv::InitialConfiguration(512 * 1024, 1 * 1024 * 1024, 8);
const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
    glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary
);


// UI thread function
void UIThreadFunc(void *args)
{
    glv::Application::run();
}

void StartMainUI()
{
    glv::Style::standard().color.set(glv::Color(0.6f, 1.0f), 0.4f); // SmokyGray

    const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    nnt::abuse::RootContext *uiContext = new nnt::abuse::RootContext(width, height);

    glv::ApplicationFrameworkRegisterLoopCallback(uiContext);

    // create a thread to run the UI
    nn::Result result = nn::os::CreateThread(
        &threadForUse,
        UIThreadFunc,
        NULL,
        threadStack1,
        STACK_SIZE,
        nn::os::HighestThreadPriority,
        3);
    NN_ASSERT(result.IsSuccess(), "Cannot create thread for UI");

    // start the thread
    NN_LOG("Thread started : %i\n", &threadForUse);
    nn::os::StartThread(&threadForUse);

    // start all notifications to the UI
    nnt::abuse::g_Notifier.MonitorStart();
}


void EndMainUI()
{
    nn::os::WaitThread(&threadForUse);

    NN_LOG("Thread destroyed : %i\n", &threadForUse);
    nn::os::DestroyThread(&threadForUse);

    delete uiContext;

    nnt::abuse::g_Notifier.MonitorStop();
    glv::Application::quit();
}


NN_OS_EXTERN_C void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(AppletHeapSize));
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, MallocMemorySize));
    nn::init::InitializeAllocator(reinterpret_cast< void* >(address), MallocMemorySize);
}



extern "C" void nnMain()
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // run all Google tests
    // ::testing::InitGoogleTest(&argc, argv);

    #ifdef WIN32
        const char* configFile = "\\Abuse\\Abuse.config";
    #else
        const char* configFile = "/Abuse/Abuse.config";
    #endif


    if(argc > 1)
    {
        for(int i = 1; i < argc; ++i)
        {
            if(strcmp(argv[i], "-config") == 0)
            {
                if( i + 1 < argc )
                {
                    configFile = argv[i + 1];
                }
            }
        }
    }

    // initialize the UI
    InitializePeripherals();
    glv::ApplicationFrameworkInitialize(LocalHidConfiguration, LocalGfxConfiguration);
    StartMainUI();

    nnt::abuse::Platform::Initialize();
    nnt::abuse::Abuse::Initialize(configFile);
    nnt::abuse::RegisterTasks();

    nnt::abuse::Abuse::Run();

    EndMainUI();
    FinalizePeripherals();
    nnt::abuse::Abuse::Finalize();
    nnt::abuse::Platform::Finalize();
}

