﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <ctime>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <Abuse.h>
#include <Tasks/BaseTask.h>
#include <Log.h>
#include <nn/util/util_FormatString.h>

namespace nnt
{
    namespace abuse
    {
        const int Log::WORK_BUFFER_SIZE = 1024;
        const int Log::DEFAULT_BUFFER_SIZE = 4096;
        bool Log::s_forwardToConsole = false;
        bool Log::s_useTimestamps = false;
        enum LogVerbosity Log::globalVerbosity = VerbosityVerbose;

        Log::Log(const String& name, int id, enum LogVerbosity verbosityLevel, int bufferSize)
        : m_buffer(0),m_name(name), m_id(id), m_bufferSize(bufferSize), m_cursor(0), m_flushCursor(0),
        m_verbosity(verbosityLevel), m_writeMutex(), m_fileHandle(), m_filePath(), m_cursorWrapped(false), m_fileOpen(false), m_forwardToConsole(SettingUseGlobal), m_useTimestamps(SettingUseGlobal)
        {
            m_buffer = (char*)Platform::Allocate(bufferSize);

            // __INT64_MAX__ has 19 digits. We'll also allocate one char for the space character.
            int lengthOfInt64MAXPlusOneSpace = 20;
            size_t workBufferLength = strlen(Platform::GetLogDirectory()) + m_name.length() + lengthOfInt64MAXPlusOneSpace;
            char* workBuffer = (char*)Platform::Allocate(workBufferLength);

            if(id != -1)
                nn::util::SNPrintf(workBuffer, workBufferLength, "%s/%s_%d.txt", Platform::GetLogDirectory(), m_name.c_str(), m_id);
            else
                nn::util::SNPrintf(workBuffer, workBufferLength, "%s/%s.txt", Platform::GetLogDirectory(), m_name.c_str());

            m_filePath = workBuffer;

            memset(m_buffer, 0, bufferSize + 1);

            m_fileHandle = FILE_INVALID_HANDLE;

            OpenOptions options;
            options.create = true;
            options.write  = true;
            m_fileHandle = Platform::FileOpen(m_filePath.c_str(), options);
            m_fileOpen = m_fileHandle != FILE_INVALID_HANDLE;


            Platform::Free(workBuffer);
            nn::os::InitializeMutex(&m_writeMutex, false, 0);
        }

        Log::~Log()
        {
            flush(ReasonLogDeleted);
            Platform::Free(m_buffer);
            nn::os::FinalizeMutex(&m_writeMutex);
            Platform::FileClose(m_fileHandle);
        }

        void Log::SetGlobalVerbosity(enum LogVerbosity verbosity)
        {
            globalVerbosity = verbosity;
        }

        enum LogVerbosity Log::GetGlobalVerbosity()
        {
            return globalVerbosity;
        }

        void Log::SetGlobalForwardToConsole(bool value)
        {
            s_forwardToConsole = value;
        }

        void Log::ToggleGlobalTimesamps(bool value)
        {
            s_useTimestamps = value;
        }

        void Log::SetForwardToConsole(LogSetting value)
        {
            m_forwardToConsole=value;
        }

        void Log::SetVerbosity(enum LogVerbosity minVerbosity)
        {
            m_verbosity = minVerbosity;
        }

        enum LogVerbosity Log::GetVerbosity()
        {
            return m_verbosity;
        }

        void Log::ToggleTimestamps(LogSetting value)
        {
            m_useTimestamps = value;
        }

        void Log::Flush(bool crashing)
        {
            if(crashing)
                flush(ReasonCrash);
            else flush(ReasonUserRequested);
        }

        void Log::VWrite(LogVerbosity verbosity, const char* format, va_list args)
        {
            nn::os::LockMutex(&m_writeMutex);
            if(shouldWrite(verbosity))
            {
                write(format, args);

                if(verbosity == VerbosityError)
                    flush(LogFlushReason::ReasonErrorWrite);
            }
            nn::os::UnlockMutex(&m_writeMutex);
        }

        void Log::Write(LogVerbosity verbosity, const char* format, ...)
        {
            va_list marker;
            va_start(marker, format);

            VWrite(verbosity, format, marker);

            va_end(marker);
        }

        bool Log::shouldWrite(enum LogVerbosity writeVerbosity)
        {
            if(m_verbosity != VerbosityUseGlobal)
                return m_verbosity >= writeVerbosity;
            return globalVerbosity >= writeVerbosity;
        }

        bool Log::shouldConsoleWrite()
        {
            if(m_forwardToConsole != SettingUseGlobal)
                return m_forwardToConsole == SettingEnabled;
            return s_forwardToConsole;
        }

        bool Log::shouldTimestamp()
        {
            if(m_useTimestamps != SettingUseGlobal)
                return m_useTimestamps == SettingEnabled;
            return s_useTimestamps;
        }

        void Log::write(const char* format, va_list args)
        {
            char workBuffer[WORK_BUFFER_SIZE];

            int prefixLength = 0;

            if(shouldTimestamp())
            {
                nn::os::Tick ticks = nn::os::GetSystemTick() - Abuse::GetStartTime();

                nn::TimeSpan upTime = ticks.ToTimeSpan();
                unsigned hours = upTime.GetHours() % 100;
                unsigned minutes = upTime.GetMinutes() % 60;
                unsigned seconds = upTime.GetSeconds() % 60;
                unsigned millis = upTime.GetMilliSeconds() % 1000;

                prefixLength += nn::util::SNPrintf(workBuffer + prefixLength, WORK_BUFFER_SIZE - prefixLength, "%02u:%02u:%02u:%03u: ", hours, minutes, seconds, millis);
            }

            int size = (int)nn::util::VSNPrintf(workBuffer + prefixLength, WORK_BUFFER_SIZE - prefixLength, format, args);

            size += prefixLength;

            if(size == 0)
                return;

            if(shouldConsoleWrite())
                NN_LOG("%s", workBuffer);

            if((size + m_cursor) < m_bufferSize)
            {
                //Append to end of buffer
                memcpy((void*)(m_buffer + m_cursor), (void*)workBuffer, size);
                m_cursor += size;
            }
            else
            {
                //Buffer's full. Flush to file if necessary, then begin overwriting buffer
                flush(LogFlushReason::ReasonBufferFull);

                memcpy((void*)m_buffer, (void*)workBuffer, size);
                m_cursor = size;

                m_flushCursor = 0;
                m_cursorWrapped = true;
            }
        }

        void Log::flush(int reason)
        {
            NN_UNUSED(reason);

            flushToFile(m_buffer + m_flushCursor, m_cursor - m_flushCursor);

            m_flushCursor = m_cursor;

        }

        void Log::flushToFile(char* buffer, int size)
        {
            if(size > 0 && m_fileOpen)
                Platform::FileWrite(m_fileHandle, buffer, size);

        }
    }
}
